/* -*- Mode:Javascript; javscript-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 25/03/2025
 */
/**
 *Virtual class interface for websocket login (see Vmp_Cmp_Main)<br>
 *Component register:<br>
 *("username",Vmp_Cmp_Input) username input<br>
 *("password",Vmp_Cmp_Password) password input<br>
 *("blogin",Vmp_Cmp_Button) send login data<br>
 *("breset",Vmp_Cmp_Button) reset username and password value
**/
class Vmp_Cmp_Login_I extends Vmp_Cmp_Div
{
  /**constructor**/
  constructor()
  {
    super();
    this.register_cmp("username",vmp_cmp_input(25,"username"));
    this.register_cmp("password",vmp_cmp_password(25,"password"));
    let blogin=vmp_cmp_button("Login","login");
    this.register_cmp("blogin",blogin);
    blogin.addEventListener("click",this.login.bind(this));
    let breset=vmp_cmp_button("Reset","reset")
    this.register_cmp("breset",breset);
    breset.addEventListener("click",(e)=>{this.input_username_.set_value();this.input_password_.set_value();});
  }
  /**Event called to log in */
  login(event)
  {
    let username=this.search_cmp("username");
    if(username.get_value() != "")
    {
      window.main_.connect(this.url(event),username.get_value());
      this.search_cmp("password").reset();
    }
    else
      vmp_wrap_alert("Input login empty!!!!");
  }
  /**Generate the url to connect the websocket*/
  url(event){vmp_wrap_alert("Vmp_Cmp_Login_I.url() not implemented");}
}
/**Virtual class interface wait for an operation before continuing(see Vmp_Cmp_Main)**/
class Vmp_Cmp_Waiting_I extends Vmp_Cmp_Div
{
  /**constructor*/
  constructor()
  {
    super();
    /**
     *Message waiting
     *@type {text}
     *@public
    */
    this.msg_="";
    /**
     *waiting timer
     *@type {cmp}
     *@public
    */
    this.timer_=new Vmp_Timeout();
  }
  /**
   *Active Component of waiting,if the timeout is triggered, it closes the connection
   *@param msg {String} information message
   *@param timeout {Number} timeout in ms
  */
  start(msg,timeout)
  {
    this.msg_=msg;
    this.timer_.start(()=>{vmp_wrap_alert(`Timeout:${msg}`);window.main_.timeout(msg);},timeout);
    this.view(`${msg}...`);
  }
  /**end waiting*/
  stop(){this.timer_.stop();}
  /**
    *set the display of the message
    *@param msg {String} message display
  */
  view(msg){vmp_wrap_alert("Vmp_Cmp_Waiting_I.view() not implemented");}
}
/**Virtual class to implement the component activated when the login has been made and the program is running(see Vmp_Cmp_Main)
**/
class Vmp_Cmp_Session_I extends Vmp_Cmp_Div 
{
  /**constructor*/
  constructor()
  {
    super();
    /**
     *actions table
     *@type {Cmp}
     *@public
    */
    this.actions_=vmp_wrap_map();
  }
  /**
    *virtual function add action
    *@param name {String} action name
    *@param text {String} text label
    *@param cb {Callback} callback callback call when event is activated
    *@param img {Element} image label
  */
  add_action(name,text,cb,img=null){vmp_wrap_alert("Vmp_Cmp_Run_I.add_action() not implemented");}
  
  /** 
    *virtual function active and disactive action
    *@param name {String} action name
    *@param value {Bool} true active,false disactive
  */
  run_action(name,value){vmp_wrap_alert("Vmp_Cmp_Run_I.run_action() not implemented");}
  /** 
    *view component
    *@param cmp {component} component to view
  */
  set_view(cmp){vmp_wrap_alert("Vmp_Cmp_Run_I.set_view() not implemented");}
}
/**
  *The default component for websocket login(see Vmp_Cmp_Main)
  *@extends Vmp_Cmp_Login_I
*/
class Vmp_Cmp_Login extends Vmp_Cmp_Login_I
{
  /**constructor*/
  constructor()
  {
    super();
    this.style.setProperty("margin","0px 0px 0px 0px");
    this.style.setProperty("position","absolute");
    this.style.setProperty("top","50%");
    this.style.setProperty("left","50%");
    this.style.setProperty("transform","translate(-50%, -50%)");
    this.style.setProperty("text-align","center");
    this.style.setProperty("padding","10px 10px 10px 10px");
    this.style.setProperty("border",`${vmp_rootvar("--border-width")} ${vmp_rootvar("--border-style")} ${vmp_rootvar("--border-color")}`);
    this.style.setProperty("background-color",vmp_rootvar("--background-color-main"));
    /**
     *logo img 
     *@type {Element}
     *@public
    */
    this.logo_=vmp_cmp_img(Vmp_Img.logo,"ragnu logo");
    this.logo_.style.setProperty("height","56px");
    this.logo_.style.setProperty("width","100px");
    this.appendChild(this.logo_);
    /**
     *logo tag title 
     *@type {Element}
     *@public
    */
    this.title_=vmp_wrap_createElement('h2');
    this.title_.innerHTML=document.title;
    this.appendChild(this.title_);
    /**
     *login container
     *@type {Element}
     *@public
    */
    this.container_=vmp_cmp_div();
    this.container_.style.setProperty("text-align","left");
    this.container_.style.setProperty("width","100%");
    this.container_.style.setProperty("font-size",vmp_rootvar("--font-size-medium"));
    this.appendChild(this.container_);
    /**
     *label element for username 
     *@type {Element}
     *@publicjavascript table row
    */
    this.label_username_=vmp_wrap_createElement('label');
    let b=vmp_wrap_createElement('b');
    b.appendChild(vmp_wrap_createTextNode("Username:"));
    this.label_username_.appendChild(b);
    this.container_.appendChild(this.label_username_);
    this.container_.appendChild(vmp_wrap_createElement('br'));
    this.container_.appendChild(this.search_cmp("username"));
    this.container_.appendChild(vmp_wrap_createElement('br'));
    this.container_.appendChild(vmp_wrap_createElement('br'));
    /**
     *label element for password 
     *@type {Element}
     *@public
    */
    this.label_password_=vmp_wrap_createElement('label');
    b=vmp_wrap_createElement('b');
    b.appendChild(vmp_wrap_createTextNode("Password:"));
    this.label_password_.appendChild(b);
    this.container_.appendChild(this.label_password_);
    this.container_.appendChild(vmp_wrap_createElement('br'));
    this.container_.appendChild(this.search_cmp("password"));
    this.container_.appendChild(vmp_wrap_createElement('br'));
    this.container_.appendChild(vmp_wrap_createElement('br'));
    /**
     *button container 
     *@type {Element}
     *@public
    */
    this.bcontainer_=vmp_wrap_createElement('div');
    this.appendChild(this.bcontainer_);
    this.bcontainer_.appendChild(this.search_cmp("blogin"));
    this.bcontainer_.appendChild(this.search_cmp("breset"));
  }
  /**virtual function*/
  url(event)
  {
    let data=vmp_japi_login(this.search_cmp("username").get_value(),this.search_cmp("password").get_value());
    if(window.location.port == '')
      return `wss://${window.location.hostname}/login?auth-token=${vmp_unicode_base64_encode(vmp_wrap_json_stringify(data.obj_))}`
    return `wss://${window.location.hostname}:${window.location.port}/login?auth-token=${vmp_unicode_base64_encode(vmp_wrap_json_stringify(data.obj_))}`
  }
}
/**
  *The default component wait for an operation before continuing(see Vmp_Cmp_Main)
  *@extends Vmp_Cmp_Waiting_I
*/
class Vmp_Cmp_Waiting extends Vmp_Cmp_Waiting_I
{
  /**constructor*/
  constructor()
  {
    super();
    this.style.setProperty("margin","0px 0px 0px 0px");
    this.style.setProperty("position","absolute");
    this.style.setProperty("text-align","center");
    this.style.setProperty("top","50%");
    this.style.setProperty("left","50%");
    this.style.setProperty("font-size",vmp_rootvar("--font-size-large"));
    this.style.setProperty("transform","translate(-50%, -50%)");
    this.style.setProperty("padding","10px 10px 10px 10px");
    this.style.setProperty("border",`${vmp_rootvar("--border-width")} ${vmp_rootvar("--border-style")} ${vmp_rootvar("--border-color")}`);
    this.style.setProperty("background-color",vmp_rootvar("--background-color-main"));
    this.text_=vmp_wrap_createElement('p');
    this.text_.innerHTML='waiting...';
    this.clessidra_=vmp_cmp_img(Vmp_Img.wait,"wait logo");
    this.clessidra_.style.setProperty("height","32px");
    this.clessidra_.style.setProperty("width","32px");
    let table=vmp_cmp_table();
    let row=table.tbody_row();
    table.row_td(row,0).appendChild(this.text_);
    table.row_td(row,1).appendChild(this.clessidra_);
    this.appendChild(table);
  }
  /**virtual function*/
  view(msg){this.text_.innerHTML=`${msg}...`;}
  /**virtual function*/
  reset_impl(){this.view("");}
}
/**
  *The default component activated when the login has been made and the program is running(see Vmp_Cmp_Main)
  *@extends Vmp_Cmp_Session_I
**/
class Vmp_Cmp_Session extends Vmp_Cmp_Session_I
{
  constructor()
  {
    super();
    this.style.setProperty("width",vmp_rootvar("--width-size"));
    this.style.setProperty("margin","0px 0px 0xp 0px");
    this.style.setProperty("padding","0px 0px 0px 0px");
    this.style.setProperty("font-family",vmp_rootvar("--font-family"));
    this.style.setProperty("font-style",vmp_rootvar("--font-style"));
    this.style.setProperty("font-size",vmp_rootvar("--font-size-medium"));
    this.style.setProperty("position","absolute");
    this.style.setProperty("left","50%");
    this.style.setProperty("transform","translateX(-50%)");
    this.style.setProperty("background-color",vmp_rootvar("--background-color-main"));
    this.style.setProperty("border",`${vmp_rootvar("--border-width")} ${vmp_rootvar("--border-style")} ${vmp_rootvar("--border-color")}`);
    this.bar_=vmp_cmp_div();
    this.bar_.style.setProperty("text-align","right");
    this.bar_.style.setProperty("font-size",vmp_rootvar("--font-size-small"));
    this.bar_.style.setProperty("background-color",vmp_rootvar("--background-color-bar"));
    this.bar_.style.setProperty("margin","5px 5px 30px 5px");
    this.bar_.style.setProperty("border-radius","10px 10px 10px 10px");
    this.body_=vmp_cmp_div();
    this.body_.style.setProperty("min-height","400px");
    this.body_.style.setProperty("overflow","scroll");
    this.body_.style.setProperty("margin","30px 5px 30px 5px");
    this.footer_=vmp_cmp_div();
    this.footer_.style.setProperty("font-size",vmp_rootvar("--font-size-small"));
    this.footer_.style.setProperty("background-color",vmp_rootvar("--background-color-bar"));
    this.footer_.style.setProperty("margin","30px 5px 30px 5px");
    this.footer_.innerHTML=`<center><i>author:${vmp_env_author()},email:${vmp_env_email()},webpage:${vmp_env_webpage()}</i></center>`;
    this.childInsert(this.bar_,0);
    this.childInsert(this.body_,1);
    this.childInsert(this.footer_,2);
  }
  /**virtual function*/
  add_action(name,text,cb,img=null)
  {
    if(this.actions_.has(name))vmp_except(`Vmp_Cmp_Run_I.add_button(name=${name})`);
    let button;
    if(img == null)
      button=vmp_cmp_button(text,name);
    else
      button=vmp_cmp_button_img(img,text,name);
    button.addEventListener("click",cb);
    this.actions_.set(name,button);
    this.bar_.appendChild(button);
    return button; 
  }
  /**virtual function*/
  run_action(name,value)
  {
    let tag=this.actions_.get(name);
    if(tag != undefined)
    {
      tag.disabled=!value;
      tag.style.setProperty("cursor",vmp_rootvar((value)?"--cursor-pointer":"--cursor-unpointer"));
    }
  }
  /**virtual function*/
  set_view(cmp)
  {
    if(cmp == null)
      this.body_.childRemove(0);
    else
      this.body_.childReplace(cmp,0);
  }
  /**virtual function*/
  reset_impl(){this.set_view(null);}
}
/**
  *User password management component
**/
class Vmp_Cmp_Passwd extends Vmp_Cmp_Div
{
  /**constructor*/
  constructor()
  {
    super();
    let center=vmp_wrap_createElement("center")
    this.appendChild(center);
    let info=vmp_cmp_info();
    let tmp=vmp_cmp_password(25,"password");
    this.register_cmp("ipasswd",tmp);
    info.add_line("Password:",tmp);
    tmp=vmp_cmp_password(25,"new password");
    this.register_cmp("inpasswd",tmp);
    info.add_line("New Password:",tmp);
    tmp=vmp_cmp_password(25,"new password");
    this.register_cmp("icpasswd",tmp);
    info.add_line("Confirm New Password:",tmp);
    let bsend=vmp_cmp_button("Change Password");
    this.register_cmp("bsend",bsend);
    bsend.addEventListener("click",this.change.bind(this));
    let breset=vmp_cmp_button("Reset");
    this.register_cmp("breset",breset);
    breset.addEventListener("click",(e)=>{this.search_cmp("ipasswd").reset();this.search_cmp("inpasswd").reset();this.search_cmp("icpasswd").reset()});
    let div=vmp_cmp_div();
    div.appendChild(bsend);
    div.appendChild(breset);
    info.add_single_line(div);
    center.appendChild(info);
  }
  /**internal usage*/
  change(e)
  {
    let ipasswd_t=this.search_cmp("ipasswd");
    let ipasswd=ipasswd_t.get_value();
    let inpasswd_t=this.search_cmp("inpasswd");
    let inpasswd=inpasswd_t.get_value();
    let icpasswd_t=this.search_cmp("icpasswd");
    let icpasswd=icpasswd_t.get_value();
    if(inpasswd != icpasswd)vmp_wrap_alert("New and confirm passwords don't match");
    else
    {
      let jdata=vmp_japi_passwd(ipasswd,inpasswd);
      window.main_.request(this,jdata);
      ipasswd_t.reset();
      inpasswd_t.reset();
      icpasswd_t.reset();
      window.main_.waiting("Change password",10000,Vmp_WebSocketCode.Protocol,"Change Password timeout");
    }
  }
  /**virtual function*/
  req_close(req)
  {
    window.main_.main();
    if(req.close_status_ === Vmp_Jrp_StatusCode.ok)
      vmp_wrap_alert(`Change Password ok`);
    else
      vmp_wrap_alert(`Change Password error '${req.close_msg_}'`); 
  }
}
/**
  *Web user management component
**/
class Vmp_Cmp_User extends Vmp_Cmp_Div
{
  /**constructor*/
  constructor()
  {
    super();
    let center=vmp_wrap_createElement("center")
    this.appendChild(center);
    let info=vmp_cmp_info();
    this.register_cmp("info",info);
    let oper=vmp_cmp_select({value:"userlist",text:"Users list"},
                              {value:"useradd",text:"Add User"},
                              {value:"usermod",text:"Modify user options"},
                              {value:"userpasswd",text:"Modify user password"},
                              {value:"userdel",text:"Delete user"});
    oper.addEventListener("change",this.change.bind(this));
    this.register_cmp("oper",oper);
    info.add_line("Operation name:",oper);
    let user=vmp_cmp_input(25,"username");
    this.register_cmp("user",user);
    info.add_line("User name:",user);
    let password=vmp_cmp_password(25,'password');
    this.register_cmp("password",password);
    info.add_line("Password:",password);
    let cpassword=vmp_cmp_password(25,'confirm password');
    this.register_cmp("cpassword",cpassword);
    info.add_line("Confirm Password:",cpassword);
    let permits=vmp_cmp_input(3,"permits");
    this.register_cmp("permits",permits);
    info.add_line("Permits:",permits);
    let only=vmp_cmp_checkbox("only_localhost");
    this.register_cmp("only",only);
    info.add_line("Only Localhost:",only);
    let div=vmp_cmp_div();
    let exec=vmp_cmp_button("Exec");
    exec.addEventListener("click",this.oper.bind(this));
    div.appendChild(exec);
    let reset=vmp_cmp_button("Reset");
    reset.addEventListener("click",this.reset_field.bind(this));
    div.appendChild(reset);
    info.add_single_line(div);
    center.appendChild(info);
    let title=vmp_wrap_createElement("h3");
    title.innerHTML="User Info";
    this.appendChild(title);
    let list=vmp_cmp_list({name:'user',order:true,width:'50%'}, 
                          {name:'permits',order:true,width:'20%'}, 
                          {name:'only_localhost',order:true,width:'20%'},
                          {name:'select',width:'10%'});
    this.register_cmp("list",list);
    this.appendChild(list);
    this.change();
  }
  /**internal usage*/
  change(event)
  {
    let voper=this.search_cmp("oper").get_value();
    let user=this.search_cmp("user");
    let password=this.search_cmp("password");
    let cpassword=this.search_cmp("cpassword");
    let permits=this.search_cmp("permits");
    let only=this.search_cmp("only");
    if(voper === "userlist")
    {
      user.disable();
      password.disable();
      cpassword.disable();
      permits.disable();
      only.disable();
    }
    else if(voper === "useradd")
    {
      user.enable();
      password.enable();
      cpassword.enable();
      permits.enable();
      only.enable();
    }
    else if(voper === "usermod")
    {
      user.enable();
      password.disable();
      cpassword.disable();
      permits.enable();
      only.enable();
    }
    else if(voper === "userpasswd")
    {
      user.enable();
      password.enable();
      cpassword.enable();
      permits.disable();
      only.disable();
    }
    else if(voper === "userdel")
    {
      user.enable();
      password.disable();
      cpassword.disable();
      permits.disable();
      only.disable();
    }
  }
  oper(event)
  {
    let voper=this.search_cmp("oper").get_value();
    let list=this.search_cmp("list");
    let user=this.search_cmp("user");
    let password=this.search_cmp("password");
    let cpassword=this.search_cmp("cpassword");
    let permits=this.search_cmp("permits");
    let only=this.search_cmp("only");
    if(voper === "userlist")
    {
      list.reset();
      let jdata=vmp_japi_userlist();
      window.main_.request(this,jdata);
      window.main_.waiting("UserList",10000,Vmp_WebSocketCode.Protocol,"UserList timeout");
    }
    else if(voper === "useradd")
    {
      let p=vmp_permits_value(permits.get_value());
      if(user.get_value() == '')
        vmp_wrap_alert("The user field is empty");
      else if(p === -1)
        vmp_wrap_alert(`The permit must be a number between ${Vmp_jrp_min_permits} and ${Vmp_jrp_max_permits}`);
      else if(password.get_value() != password.get_value())
        vmp_wrap_alert("Password and confirm passwords don't match");
      else
      {
        if(vmp_wrap_confirm(`Confirm user add ${user.get_value()}`))
        {
          list.reset();
          let jdata=vmp_japi_useradd(user.get_value(),password.get_value(),p,only.get_select());
          window.main_.request(this,jdata);
          window.main_.waiting("UserAdd",10000,Vmp_WebSocketCode.Protocol,"UserAdd timeout");
        }
      }
    }
    else if(voper === "usermod")
    {
      let p=vmp_permits_value(permits.get_value());
      if(user.get_value() == '')
        vmp_wrap_alert("The user field is empty");
      else if(p === -1)
        vmp_wrap_alert(`The permit must be a number between ${Vmp_jrp_min_permits} and ${Vmp_jrp_max_permits}`);
      else
      {
        if(vmp_wrap_confirm(`Confirm user modify of  ${user.get_value()}`))
        {
          list.reset();
          let jdata=vmp_japi_usermod(user.get_value(),p,only.get_select());
          window.main_.request(this,jdata);
          window.main_.waiting("UserMod",10000,Vmp_WebSocketCode.Protocol,"UserMod timeout");
        }
      }
    }
    else if(voper === "userpasswd")
    {
      if(user.get_value() == '')
        vmp_wrap_alert("The user field is empty");
      else if(password.get_value() != cpassword.get_value())
        vmp_wrap_alert("Password and confirm passwords don't match");
      else
      {
        if(vmp_wrap_confirm(`Confirm password change of ${user.get_value()}`))
        {
          list.reset();
          let jdata=vmp_japi_userpasswd(user.get_value(),password.get_value());
          window.main_.request(this,jdata);
          window.main_.waiting("UserPasswd",10000,Vmp_WebSocketCode.Protocol,"UserPasswd timeout");
        }
      }
    }
    else if(voper === "userdel")
    {
      if(user.get_value() == '')
        vmp_wrap_alert("The user field is empty");
      else 
      {
        if(vmp_wrap_confirm(`Confirm User Deletion of ${user.get_value()}`))
        list.reset();
        let jdata=vmp_japi_userdel(user.get_value());
        window.main_.request(this,jdata);
        window.main_.waiting("UserDel",10000,Vmp_WebSocketCode.Protocol,"UserDel timeout");
      }
    }
    this.reset_field();
  }
  /**virtual function*/
  req_recv(req,jdata,payload)
  {
    let list=this.search_cmp("list");
    let user=jdata.get_text("user");
    let permits=jdata.get_integer_range("permits",Vmp_jrp_min_permits,Vmp_jrp_max_permits);
    let only_localhost=(jdata.get_bool("only_localhost"))?"yes":"no";
    let k=vmp_cmp_button_img(Vmp_Img.select,user,user);
    k.addEventListener("click",this.uselect.bind(this));
    list.insert(user,[user,permits,only_localhost,k]); 
  }
  uselect(event)
  {
    let list=this.search_cmp("list");
    let user=event.currentTarget.bid();
    let row=list.lines_.get(user)['row'];
    this.search_cmp("user").set_value(user);
    this.search_cmp("permits").set_value(list.row_getcell(row,1).innerHTML);
    if(row.cells[2].innerHTML === "yes")
      this.search_cmp("only").set_select(true);
    else
      this.search_cmp("only").set_select(false);
  }
  /**virtual function*/
  req_close(req)
  {
    window.main_.main();
    let voper=this.search_cmp("oper").get_value();
    if(req.close_status_ === Vmp_Jrp_StatusCode.ok)
      vmp_wrap_alert(`${voper} ok`);
    else
      vmp_wrap_alert(`${voper} error '${req.close_msg_}'`) 
  }
  /**internal usage*/
  reset_field(event)
  {
    this.search_cmp("user").reset();
    this.search_cmp("password").reset();
    this.search_cmp("cpassword").reset();
    this.search_cmp("permits").reset();
    this.search_cmp("only").reset();
  }
}
/**
  *Component for managing peers
**/
class Vmp_Cmp_Peer extends Vmp_Cmp_Div
{
  /**constructor*/
  constructor()
  {
    super();
    let info=vmp_cmp_info();
    this.register_cmp("info",info);
    this.appendChild(info);
    let oper=vmp_cmp_select({value:"peerlist",text:"Peers list"},
                            {value:"peeradd",text:"Add Peer"},
                            {value:"peermod",text:"Modify peer options"},
                            {value:"peerdel",text:"Delete peer"});
    oper.addEventListener("change",this.change.bind(this));
    this.register_cmp("oper",oper);
    info.add_line("Operation name:",oper);
    let fingerprint=vmp_cmp_input(70,"fingerprint");
    this.register_cmp("fingerprint",fingerprint);
    info.add_line("Fingerprint:",fingerprint);
    let subject=vmp_cmp_input(70,'subject');
    this.register_cmp("subject",subject);
    info.add_line("Subject:",subject);
    let permits=vmp_cmp_input(3,"permits");
    this.register_cmp("permits",permits);
    info.add_line("Permits:",permits);
    let div=vmp_cmp_div();
    let exec=vmp_cmp_button("Exec");
    exec.addEventListener("click",this.oper.bind(this));
    div.appendChild(exec);
    let reset=vmp_cmp_button("Reset");
    reset.addEventListener("click",this.reset_field.bind(this));
    div.appendChild(reset);
    info.add_single_line(div);
    let title=vmp_wrap_createElement("h3");
    title.innerHTML="Peer Info";
    this.appendChild(title);
    let list=vmp_cmp_list({name:'subject',order:true,width:'80%'},
                          {name:'permits',order:true,width:'10%'}, 
                          {name:'select',width:'10%'});
    this.register_cmp("list",list);
    this.appendChild(list);
    this.change();
  }
  /**internal usage*/
  change(event)
  {
    let voper=this.search_cmp("oper").get_value();
    let fingerprint=this.search_cmp("fingerprint");
    let subject=this.search_cmp("subject");
    let permits=this.search_cmp("permits");
    if(voper === "peerlist")
    {
        fingerprint.disable();
        subject.disable();
        permits.disable();
    }
    else if(voper === "peeradd")
    {
        fingerprint.enable();
        subject.enable();
        permits.enable();
    }
    else if(voper === "peermod")
    {
        fingerprint.enable();
        subject.disable();
        permits.enable();
    }
    else if(voper === "peerdel")
    {
        fingerprint.enable();
        subject.disable();
        permits.disable();
    }
  }
  oper(event)
  {
    let voper=this.search_cmp("oper").get_value();
    let list=this.search_cmp("list");
    let fingerprint=this.search_cmp("fingerprint");
    let subject=this.search_cmp("subject");
    let permits=this.search_cmp("permits");
    if(voper === "peerlist")
    {
        list.reset();
        let jdata=vmp_japi_peerlist();
        window.main_.request(this,jdata);
        window.main_.waiting("PeerList",10000,Vmp_WebSocketCode.Protocol,"PeerList timeout");
    }
    else if(voper === "peeradd")
    {
      let p=vmp_permits_value(permits.get_value());
      if(fingerprint.get_value() == '')
        vmp_wrap_alert("The fingerprint field is empty");
      else if(subject.get_value() == '')
        vmp_wrap_alert("The subject field is empty");
      else if(p === -1)
        vmp_wrap_alert(`The permit must be a number between ${Vmp_jrp_min_permits} and ${Vmp_jrp_max_permits}`);
      else
      {
        list.reset();
        if(vmp_wrap_confirm(`Confirm Peer Add ${fingerprint.get_value()}`))
        {
          let jdata=vmp_japi_peeradd(fingerprint.get_value(),subject.get_value(),p);
          window.main_.request(this,jdata);
          window.main_.waiting("PeerAdd",10000,Vmp_WebSocketCode.Protocol,"PeerAdd timeout");
        }
      }   
    }
    else if(voper === "peermod")
    {
      let p=vmp_permits_value(permits.get_value());
      if(fingerprint.get_value() == '')
        vmp_wrap_alert("The fingerprint field is empty");
      else if(p === -1)
        vmp_wrap_alert(`The permit must be a number between ${Vmp_jrp_min_permits} and ${Vmp_jrp_max_permits}`);
      else
      {
        if(vmp_wrap_confirm(`Confirm Peer modify of ${fingerprint.get_value()}`))
        {
          list.reset();
          let jdata=vmp_japi_peermod(fingerprint.get_value(),p);
          window.main_.request(this,jdata);
          window.main_.waiting("PeerMod",10000,Vmp_WebSocketCode.Protocol,"PeerMod timeout");
        }
      }    
    }
    else if(voper === "peerdel")
    {
      if(fingerprint.get_value() == '')
        vmp_wrap_alert("The fingerprint field is empty");
      else
      {
        if(vmp_wrap_confirm(`Confirm Peer Deletion of ${fingerprint.get_value()}`))
        {
          list.reset();
          let jdata=vmp_japi_peerdel(fingerprint.get_value());
          window.main_.request(this,jdata);
          window.main_.waiting("PeerDel",10000,Vmp_WebSocketCode.Protocol,"PeerDel timeout");
        }
      }
    }
    this.reset_field();
  }
  /**virtual function*/
  req_recv(req,jdata,payload)
  {
    let list=this.search_cmp("list");
    let fingerprint=jdata.get_text("fingerprint");
    let subject=jdata.get_text("subject");
    let permits=jdata.get_integer_range("permits",Vmp_jrp_min_permits,Vmp_jrp_max_permits);
    let k=vmp_cmp_button_img(Vmp_Img.select,fingerprint,fingerprint);
    k.addEventListener("click",this.uselect.bind(this));
    list.insert(fingerprint,[subject,permits,k]); 
  }
  uselect(event)
  {
    let list=this.search_cmp("list");
    let fingerprint=event.currentTarget.bid();
    let row=list.lines_.get(fingerprint)['row'];
    this.search_cmp("fingerprint").set_value(fingerprint);
    this.search_cmp("subject").set_value(list.row_getcell(row,0).innerHTML);
    this.search_cmp("permits").set_value(list.row_getcell(row,1).innerHTML);
  }
  /**virtual function*/
  req_close(req)
  {
    window.main_.main();
    let voper=this.search_cmp("oper").get_value();
    if(req.close_status_ === Vmp_Jrp_StatusCode.ok)
      vmp_wrap_alert(`${voper} ok`);
    else
      vmp_wrap_alert(`${voper} error '${req.close_msg_}'`) 
  }
  /**internal usage*/
  reset_field(event)
  {
    this.search_cmp("fingerprint").reset();
    this.search_cmp("subject").reset();
    this.search_cmp("permits").reset();
  }
}
/**
  *Component for managing connection
**/
class Vmp_Cmp_ConnectStatus extends Vmp_Cmp_Div
{
  /**constructor*/
  constructor()
  {
    super();
    let exec=vmp_cmp_button("Connection list");
    exec.addEventListener("click",this.oper.bind(this));
    this.appendChild(exec);
    let list=vmp_cmp_list({name:'id',order:true,width:'30%'},
                          {name:'aux data',order:true,width:'25%'}, 
                          {name:'type',order:true,width:'7%'}, 
                          {name:'permits',order:true,width:'5%'}, 
                          {name:'address',order:true,width:'14%'},
                          {name:'port',order:true,width:'7%'},
                          {name:'close',width:'7%'});
    this.register_cmp("list",list);
    this.appendChild(list);
  }
  /**internal usage*/
  oper(event)
  {
    let list=this.search_cmp("list");
    list.reset();
    let jdata=japi_connectlist();
    window.main_.request(this,jdata);
    window.main_.waiting("ConnectList",10000,Vmp_WebSocketCode.Protocol,"ConnectList timeout");
  }
  uselect(event)
  {
    let list=this.search_cmp("list");
    let id=event.currentTarget.bid();
    let row=list.lines_.get(id)['row'];
    if(vmp_wrap_confirm(`Confirm Close Connection: ${id}`))
    {
      let jdata=vmp_japi_connectclose(id,list.row_getcell(row,2).innerHTML);
      list.reset();
      window.main_.request(this,jdata);
      window.main_.waiting("ConnectClose",10000,Vmp_WebSocketCode.Protocol,"Connection close timeout");
    }
  }
  /**virtual function*/
  req_recv(req,jdata,payload)
  {
    let list=this.search_cmp("list");
    let id=jdata.get_text("id");
    let type=jdata.get_text("type");
    let aux=jdata.get_text("aux");
    let permits=jdata.get_integer_range("permits",Vmp_jrp_min_permits,Vmp_jrp_max_permits);
    let address=jdata.get_text("address");
    let port=jdata.get_integer_range("port",0,65535);
    let k=vmp_cmp_button_img(Vmp_Img.select,id,id);
    k.addEventListener("click",this.uselect.bind(this));
    list.insert(id,[vmp_unicode_str_format_lines(id,32),vmp_unicode_str_format_lines(aux,24),type,permits,vmp_unicode_str_format_lines(address,15),port,k]); 
  }
  /**virtual function*/
  req_close(req)
  {
    window.main_.main();
    let voper;
    if(req.type_ === vmp_japi_type("connectlist"))
      voper="Connect List";
    else
      voper="Connect close";
    if(req.close_status_ === Vmp_Jrp_StatusCode.ok)
      vmp_wrap_alert(`${voper} ok`);
    else
      vmp_wrap_alert(`${voper} error '${req.close_msg_}'`) 
  }
};
/**
  *Component for managing logs
**/
class Vmp_Cmp_Logs extends Vmp_Cmp_Div
{
  /**constructor*/
  constructor()
  {
    super();
    let filelist=vmp_cmp_select();
    filelist.style.setProperty("display","inline-block");
    filelist.style.setProperty("min-width","100px");
    this.appendChild(filelist);
    this.register_cmp("filelist",filelist);
    let update=vmp_cmp_button("update filelist","update");
    update.style.setProperty("display","inline-block");
    update.addEventListener("click",this.oper.bind(this));
    this.appendChild(update);
    let view=vmp_cmp_button("view file","view");
    view.addEventListener("click",this.oper.bind(this));
    view.style.setProperty("display","inline-block");
    this.appendChild(view);
    let textview=vmp_cmp_textview("99%","400px",true);
    this.register_cmp("textview",textview);
    this.appendChild(textview);
  }
  oper(event)
  {
    let rid=event.currentTarget.bid();
    if(rid == "update")
    {
      let jdata=japi_logs();
      window.main_.request(this,jdata);
      window.main_.waiting("Logs list",10000,Vmp_WebSocketCode.Protocol,"Logs list close timeout");
    }
    else
    {
      let value=this.search_cmp("filelist").get_value();
      if(value == "")
        vmp_wrap_alert("update filelist");
      else
      {
        let jdata=japi_logread(value);
        window.main_.request(this,jdata);
        window.main_.waiting("Logs list",10000,Vmp_WebSocketCode.Protocol,"Logs list close timeout");
      }
    }
  }
  /**virtual function*/
  req_recv(req,jdata,payload)
  {
    if(jdata.jtype() == vmp_japi_type("logfiles"))
    {
        let files=jdata.get_args("files");
        let filelist=this.search_cmp("filelist");
        filelist.childRemoveAll();
        for(let f of files)filelist.add_option(f,f);
    }
    else if (jdata.jtype() == vmp_japi_type("logview"))
    {
        let logname=jdata.get_text("logname");
        let view=jdata.get_text("view");
        this.search_cmp("textview").set_text(logname,view);
    }
  }
  /**virtual function*/
  req_close(req)
  {
    window.main_.main();
    let voper;
    if(req.type_ === vmp_japi_type("logs"))
      voper="Logs list";
    else
      voper="Log read";
    if(req.close_status_ === Vmp_Jrp_StatusCode.ok)
      vmp_wrap_alert(`${voper} ok`);
    else
      vmp_wrap_alert(`${voper} error '${req.close_msg_}'`) 
  }
};
/**
  *The default component JRP and session message management class(see Vmp_Cmp_Main)
  *@extends Vmp_Jrp_Manager_I
**/
class Vmp_Cmp_Manager extends Vmp_Jrp_Manager_I
{
  /**constructor*/
  constructor()
  {
    super();
    /**head section*/
    let info=vmp_cmp_info();
    this.register_cmp("info",info);
    let head=vmp_cmp_top('lib/img/ragnu.jpg','System management',info);
    this.appendChild(head);
    /**session section*/
    let session=vmp_cmp_div();
    this.register_cmp("session",session);
    let title=vmp_wrap_createElement("h3");
    title.innerHTML="Request input";
    session.appendChild(title);
    let session_data=vmp_cmp_list({name:'input',order:true,width:'30%'},
                             {name:'response',order:true,width:'30%'},
                             {name:'push',order:true,width:'30%'},
                             {name:'permits',order:true,width:'10%'});
    this.register_cmp("session_data",session_data);
    session.appendChild(session_data);
    title=vmp_wrap_createElement("h3");
    title.innerHTML="Broadcast message";
    session.appendChild(title);
    let session_broad=vmp_cmp_list({name:'input',order:true,width:'30%'},
                                    {name:'',order:false,width:'70%'});
    this.register_cmp("session_broad",session_broad);
    session.appendChild(session_broad);
    /**Request section*/
    let requests=vmp_cmp_div();
    this.register_cmp("requests",requests);
    title=vmp_wrap_createElement("h3");
    title.innerHTML="Request Manager";
    requests.appendChild(title);
    let reqmanager=vmp_cmp_list({name:'rid',order:true,width:'8%'},
                                  {name:'component',order:true,width:'20%'},
                                  {name:'request jdata',order:true,width:'20%'},
                                  {name:'status',order:true,width:'8%'},
                                  {name:'exit code',order:true,width:'20%'},   
                                  {name:'exit message',order:true,width:'20%'},
                                  {name:'kill',width:'8%'});
    this.register_cmp("reqmanager",reqmanager);
    requests.appendChild(reqmanager);
    this.register_cmp("passwd",vmp_wrap_createElement("vmp-passwd",'div'));
    this.register_cmp("users",vmp_wrap_createElement("vmp-user",'div'));
    this.register_cmp("peers",vmp_wrap_createElement("vmp-peer",'div'));
    this.register_cmp("connectstatus",vmp_wrap_createElement("vmp-connectstatus",'div'));
    this.register_cmp("logs",vmp_wrap_createElement("vmp-logs",'div'));
    let body=vmp_cmp_menu();
    this.register_cmp("body",body);
    this.appendChild(body);
  }
  /**Virtual function called on session opening*/
  session_impl()
  {
    let info=this.search_cmp("info");
    info.add_line("user:",this.user_);
    info.add_line("permits:",this.permits_);
    info.add_line("nodetype:",this.nodetype_);
    let body=this.search_cmp("body");
    /**session section*/
    let session=this.search_cmp("session");
    let session_data=this.search_cmp("session_data");
    let session_broad=this.search_cmp("session_broad");
    body.add_section("Session",session);
    for(let r in this.reqdata_)
    {
      let response=vmp_wrap_createElement("div"),push=vmp_wrap_createElement("div");
      let i=0;
      for(let tmp of this.reqdata_[r].response)
      {
        if(i != 0)
          response.appendChild(vmp_wrap_createElement("br"));
        response.appendChild(vmp_wrap_createTextNode(tmp));
        i++; 
      }
      i=0;
      for(let tmp of this.reqdata_[r].push)
      {
        if(i != 0)
          push.appendChild(vmp_wrap_createElement("br"));
        push.appendChild(vmp_wrap_createTextNode(tmp));
        i++; 
      }
      session_data.insert(r,[r,response,push,this.reqdata_[r].permits]);
    }
    for(let tmp of this.bdata_)
      session_broad.insert(tmp,[tmp,'']);
    /**Request section*/
    let requests=this.search_cmp("requests");
    let reqmanager=this.search_cmp("reqmanager");
    body.add_section("Requests",requests);
    if (vmp_japi_type("passwd") in this.reqdata_)
      body.add_section("Password",this.search_cmp("passwd"));
    if (vmp_japi_type("userlist") in this.reqdata_)
      body.add_section("Users",this.search_cmp("users"));
    if (vmp_japi_type("peerlist") in this.reqdata_)
      body.add_section("Peers",this.search_cmp("peers"));
    if (vmp_japi_type("connectlist") in this.reqdata_)
      body.add_section("Connection",this.search_cmp("connectstatus"));
    if (vmp_japi_type("logs") in this.reqdata_)
      body.add_section("Logs",this.search_cmp("logs"));
  }
  /**Virtual function called on request*/
  update_request(req)
  {
    let reqmanager=this.search_cmp("reqmanager");
    if(req.status_ === "open")
    {
      let k=vmp_cmp_button_img(Vmp_Img.close,`kill ${req.rid_}`,req.rid_);
      k.addEventListener("click",this.kill_req.bind(this));
      reqmanager.insert(req.rid_,[req.rid_,req.cmp_.cmpid(),req.jdata_.jtype(),req.status_,req.close_status_,req.close_msg_,k]);
    }
    else
      reqmanager.insert(req.rid_,[req.rid_,req.cmp_.cmpid(),req.jdata_.jtype(),req.status_,req.close_status_,req.close_msg_]);
  }
  /**Internal usage*/
  kill_req(e)
  {
    let rid=event.currentTarget.bid();
    let req=this.table_.get(Number(rid));
    req.kill();
  }
};
try
{
  vmp_wrap_define("vmp-login",Vmp_Cmp_Login,'div');
  vmp_wrap_define("vmp-waiting",Vmp_Cmp_Waiting,'div');
  vmp_wrap_define("vmp-session",Vmp_Cmp_Session,'div');
  vmp_wrap_define("vmp-passwd",Vmp_Cmp_Passwd,'div');
  vmp_wrap_define("vmp-user",Vmp_Cmp_User,'div');
  vmp_wrap_define("vmp-peer",Vmp_Cmp_Peer,'div');
  vmp_wrap_define("vmp-connectstatus",Vmp_Cmp_ConnectStatus,'div');
  vmp_wrap_define("vmp-logs",Vmp_Cmp_Logs,'div');
  vmp_wrap_define("vmp-manager",Vmp_Cmp_Manager,'div');
}
catch(error)
{
  vmp_wrap_alert(error);
}
