/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 05/09/2024
 */

#ifndef VAMPIRIA_VMP_PAIRLIST_H

#define VAMPIRIA_VMP_PAIRLIST_H 1

namespace vampiria { namespace vmp {

//! List of elements composed of a key and data pair
template<typename KEY,typename DATA>
class PairList
{
    private:
        vmp::vector<vmp::pair<KEY,DATA> > list_;/*!<list pair data*/
        
        //!Internal usage
        void swap(vmp_index oldpos,vmp_index pos)
        {
            vmp::pair<KEY,DATA> tmp=list_[oldpos];
            if(oldpos < pos)
            {
                for(vmp_index i=oldpos;i<pos;i++)
                    list_[i]=list_[i+1];   
                list_[pos]=tmp;
            }
            else if(oldpos > pos)
            {
                for(vmp_index i=oldpos;i>pos;i--)
                    list_[i]=list_[i-1];   
                list_[pos]=tmp;
            }
        }
    
    public:
        //! A constructor
        PairList(){}
        
        //! //! A destructor
        ~PairList(){clear();}
        
        //!List clear don't free memory pointer
        void clear(){list_.clear();}
        
        //!List size
        /*!
            @return size data
            @sa list_        
        */
        vmp_size size() {return list_.size();}
        
        //!insert a value in the list,if the value exists change its position and update the value
        /*!
            @param key key list
            @param data data value
            @param pos position to insert data
            @return list size
            @sa list_
        */                  
        vmp_index insert(KEY key,DATA data,vmp_index pos)
        {
            vmp_index oldpos;
            try
            {
                oldpos=position(key);
                list_[oldpos].second=data;
            }
            catch(vmp::exception &x)
            {
                oldpos=list_.size();
                list_.push_back(vmp::pair<KEY,DATA>(key,data));    
            }
            swap(oldpos,pos);
            return list_.size();
        }
        
        //!update a value in the list,if the value does not exist insert it at the end
        /*!
            @param key key list
            @param data data value
            @return list size
            @sa list_
        */  
        vmp_index update(KEY key,DATA data)
        {
            try
            {
                vmp_index pos=position(key);
                list_[pos].second=data;
            }
            catch(vmp::exception &x)
            {
                list_.push_back(vmp::pair<KEY,DATA>(key,data));    
            }
            return list_.size();
        }
        
        //!insert a data at the end, if the value exists it moves to the end
        /*!
            @param key key list
            @param data data value
            @return list size
            @sa list_
        */ 
        vmp_index push(KEY key,DATA data)
        {
            try
            {
                vmp_index oldpos=position(key);
                list_[oldpos].second=data;
                swap(oldpos,list_.size()-1);
            }
            catch(vmp::exception &x)
            {
                list_.push_back(vmp::pair<KEY,DATA>(key,data));    
            }
            return list_.size();
        }
        
        //!Delete the key value from the list, if it does not exist no does anything
        /*!
            @param key key list
            @return list size
            @sa list_
        */ 
        vmp_index cancel(KEY key)
        {
            try
            {
                cancel(position(key));
            }
            catch(vmp::exception &x)
            {
            }
            return list_.size();
        }
        
        //!Delete the value in the index position from the list, if it does not exist no does anything
        /*!
            @param index index of value
            @return list size
            @sa list_
        */ 
        vmp_index cancel(vmp_index index)
        {
            auto e=list_.begin()+index;
            if(e < list_.end())
                list_.erase(e);
            return list_.size();
        }
        
        //! Returns all list keys
        /*!
             @sa list_            
             @return list keys        
        */
        vmp::vector<KEY> all_keys()
        {
            vmp::vector<KEY> ret;
            for(vmp_index i=0;i<list_.size();i++)
                ret.push_back(list_[i].first);
            return ret;
        }
        
        //! Returns all data list
        /*!
            @sa list_            
            @return list data        
        */
        vmp::vector<DATA> all_data()
        {
            vmp::vector<KEY> ret;
            for(vmp_index i=0;i<list_.size();i++)
                ret.push_back(list_[i].second);
            return ret;
        }
        
        //! Returns the index of key position
        /*!
            @param key key list
            @return key position or except in case of failure
        */
        vmp_index position(KEY key)
        {
            for(vmp_index i=0;i<list_.size();i++)
               if(key == list_[i].first)
                   return i;
            vmp::except("vmp::PairList::position() key not found");
            return list_.size();
        }
        
        //! Search a value in the table                
        /*!
            @param key key list
            @param pret data output
            @sa list_
            @return true ok,otherwise false
        */
        vmp_bool search(KEY key,DATA *pret)
        {
            for(vmp_index i=0;i<list_.size();i++)
            {
                if(list_[i].first == key)
                {
                    (*pret)=list_[i].second;
                    return true;
                }
            }
            return false;
        }
        
        //!Returns the key value pair of index index
        /*!
            @param index index of value
            @sa list_
            @return pair key data or except in case of failure
        */
        vmp::pair<KEY,DATA> get(vmp_index index)
        {
            return list_[index];
        }
};

//!Reset the list and free up data storage space. The DATA type must be a pointer and allocated with vmp::malloc_wrap()
/*!
    @param list pointer input pair list
*/
template<class KEY,class DATA>
void pairlist_free_alldata(vmp::PairList<KEY,DATA> *list)
{
    if(list != 0)
    {
        vmp::vector<DATA> data=list->all_data();
        for(vmp_index i=0;i<data.size();i++)
            vmp::free_wrap((void **)&data[i]);
        list->clear(); 
    }
}

//!Reset the pair list and delete up data storage space. The DATA type must be a pointer and allocated with new
/*!
    @param list pointer input pair list
*/
template<class KEY,class DATA>
void pairlist_delete_alldata(vmp::PairList<KEY,DATA> *list)
{
    if(list != 0)
    {
        vmp::vector<DATA> data=list->all_data();
        for(vmp_index i=0;i<data.size();i++)
           delete data[i];
        list->clear();
    }
}

}}

#endif

