/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 26/03/2025
 */

#include "vmp.h"

namespace vampiria { namespace vmp {

vmp::File fopen_wrap(vmp::str pathname,vmp::str mode)
{
    vmp::File file=fopen(pathname.c_str(),mode.c_str());
    if(file == 0)
        vmp::except("'%s' %s",pathname.c_str(),vmp::strerror_wrap(errno).c_str());
    return file;
}

vmp::File fdopen_wrap(vmp_int fd,vmp::str mode)
{
    vmp_assert(_POSIX_C_SOURCE);
    vmp::File ret=fdopen(fd,mode.c_str());
    if(ret == 0)
        vmp::except_errno();
    return ret;
}

vmp_int fgetc_wrap(vmp::File file)
{
    vmp::except_check_pointer((void *) file,"vmp::fgetc_wrap(file=null)");
    return fgetc(file);
}

void fflush_wrap(vmp::File file)
{
    vmp::except_check_pointer((void *) file,"vmp::fflush_wrap(file=null)");
    if(fflush(file) != 0)
        vmp::except_errno();
}

void fseek_wrap(vmp::File file,vmp_int offset,vmp::str whence)
{
    vmp::except_check_pointer((void *) file,"vmp::fseek_wrap(file=null)");
    vmp_int w;
    if(whence == "set")
        w=SEEK_SET;
    else if(whence == "cur")
        w=SEEK_CUR;
    else if(whence == "end")
        w=SEEK_END;
    else
        vmp::except("vmp::fseek(whence='%s') bad value",whence.c_str());
    if(fseek(file,offset,w) == -1)
        vmp::except_errno();
}

vmp_index ftell_wrap(vmp::File file)
{
    vmp::except_check_pointer((void *) file,"vmp::ftell_wrap(file=null)");
    vmp_int ret=ftell(file);
    if(ret == -1)
        vmp::except_errno();
    return (vmp_index) ret;
}

vmp_size fread_wrap(void *ptr,vmp_size size,vmp_size nmemb,vmp::File file)
{
    vmp::except_check_pointer(ptr,"vmp::fread_wrap(ptr=null)");
    vmp::except_check_pointer((void *) file,"vmp::fread_wrap(file=null)");
    clearerr(file);
    vmp_size ritems=fread(ptr,size,nmemb,file);
    if(ferror(file) != 0)
        vmp::except_errno();
    return ritems;
}

vmp_size fwrite_wrap(void *ptr,vmp_size size,vmp_size nmemb,vmp::File file)
{
    vmp::except_check_pointer(ptr,"vmp::fread_wrap(ptr=null)");
    vmp::except_check_pointer((void *) file,"vmp::fwrite_wrap(file=null)");
    vmp_size ritems=fwrite(ptr,size,nmemb,file);
    if(ritems < nmemb)
        vmp::except_errno();
    return ritems;
}

void fclose_wrap(vmp::File *pfile)
{
    if((*pfile) != 0)
    {
        fclose(*pfile);
        (*pfile)=0;
    }
}

static vmp_bool debug_=false;
static vmp::str id_="None";
static vmp::thread::Mutex iomutex_;

void init(vmp::str id,vmp_bool debug)
{
    id_=id;        
    debug_=debug;         
}

vmp_bool is_debug_mode()
{
    return debug_;
}

vmp::str id()
{
    return id_;
}

vmp::str va_wrap(const vmp_char *fmt,va_list ap)
{
    vmp::str ret="";
    #if defined(_XOPEN_SOURCE) || defined(_ISOC99_SOURCE) || defined(_BSD_SOURCE) || _XOPEN_SOURCE >= 500
    #else   
    vmp_assert(_XOPEN_SOURCE || _ISOC99_SOURCE || _BSD_SOURCE || _XOPEN_SOURCE >= 500);   
    #endif
    vmp_int size = 0;
    vmp_char *p = 0;
    va_list tmp;
    va_copy(tmp,ap);
    size = vsnprintf(p,size,fmt,tmp);
    va_end(tmp);
    if (size < 0)
        vmp::except_alloc();
    size++;/*For '\0'*/
    p=(vmp_char *)vmp::malloc_wrap(size);
    va_copy(tmp,ap);
    size = vsnprintf(p,size,fmt,ap);
    va_end(tmp);
    ret=p;
    vmp::free_wrap((void **)&p);
    return ret;
}

void error_process(vmp::str type,vmp::str data)
{
    iomutex_.lock();
    std::cerr << "Process ['" << id_ << "'] " << type << ": " << data << std::endl; 
    iomutex_.unlock();
}

vmp_bool io_forced_unlock()
{
    return iomutex_.forced_unlock();
}

void io_recovery_lock(vmp_bool status)
{
    iomutex_.recovery_lock(status);
}

void debug_s(vmp::str msg)
{
    if(debug_)
        vmp::error_process("Debug",msg);
}

void debug(const vmp_char *fmt,...)
{
    if(debug_)
    {    
        va_list ap;
        va_start(ap,fmt);    
        vmp::str sfmt=vmp::va_wrap(fmt,ap);
        va_end(ap);
        vmp::debug_s(sfmt);     
    } 
}

void info_s(vmp::str msg)
{
    vmp::error_process("Info",msg);
}

void info(const vmp_char *fmt,...)
{
    va_list ap;
    va_start(ap,fmt);    
    vmp::str sfmt=vmp::va_wrap(fmt,ap);
    va_end(ap);
    vmp::info_s(sfmt);
}

void warning_s(vmp::str msg)
{
    vmp::error_process("Warning",msg);
}

void warning(const vmp_char *fmt,...)
{
    va_list ap;
    va_start(ap,fmt);    
    vmp::str sfmt=vmp::va_wrap(fmt,ap);
    va_end(ap);
    vmp::warning_s(sfmt); 
}

void error_s(vmp::str msg)
{
    vmp::error_process("Error",msg); 
}

void error(const vmp_char *fmt,...)
{
    va_list ap;
    va_start(ap,fmt);    
    vmp::str sfmt=vmp::va_wrap(fmt,ap);
    va_end(ap);
    vmp::error_s(sfmt);
}

void error_raw_s(vmp::str msg)
{
    iomutex_.lock();
    std::cerr << msg;
    std::cerr.flush();
    iomutex_.unlock();
}

void error_raw(const vmp_char *fmt,...)
{
    va_list ap;
    va_start(ap,fmt);    
    vmp::str sfmt=vmp::va_wrap(fmt,ap);
    va_end(ap);
    vmp::error_raw_s(sfmt);
}

void output_s(vmp::str msg)
{
    iomutex_.lock();
    std::cout << msg << std::endl;
    iomutex_.unlock();
}

void output(const vmp_char *fmt,...)
{
    va_list ap;
    va_start(ap,fmt);    
    vmp::str sfmt=vmp::va_wrap(fmt,ap);
    va_end(ap);
    output_s(sfmt);
}

void output_raw_s(vmp::str msg)
{
    iomutex_.lock();
    std::cout << msg;
    std::cout.flush(); 
    iomutex_.unlock();
}

void output_raw(const vmp_char *fmt,...)
{
    va_list ap;
    va_start(ap,fmt);    
    vmp::str sfmt=vmp::va_wrap(fmt,ap);
    va_end(ap);
    output_raw_s(sfmt);
}

void output_file_s(vmp::str filename,vmp_bool newfile,vmp::str msg)
{
    output_file(filename,newfile,"%s",msg.c_str());
}

void output_file(vmp::str filename,vmp_bool newfile,const vmp_char *fmt,...)
{
    vmp::File file;
    if(newfile)
        file=vmp::fopen_wrap(filename,"w+");
    else
        file=vmp::fopen_wrap(filename,"a+");
    va_list ap;
    va_start(ap,fmt);    
    vfprintf(file,fmt,ap);
    va_end(ap);
    vmp::fclose_wrap(&file);
}

void output_file_lines(vmp::str filename,vmp_bool newfile,vmp::vector<vmp::str> lines)
{
    vmp::str msg="";
    for(vmp_index i=0;i<lines.size();i++)
    {    
        if(i < lines.size()-1)
            vmp::unicode::str_cwrite(&msg,"%s\n",lines[i]);
        else
            vmp::unicode::str_cwrite(&msg,"%s",lines[i]);
    }
    vmp::output_file(filename,newfile,"%s",msg.c_str());
}

static vmp::str streamcin_="";

vmp::str input()
{
    vmp_byte c;
    vmp_int result;
    while((result=vmp::fd_read_bytes(0,&c,1)) == 1)
    {
        if(c == '\n')
        {
            vmp::str ret=streamcin_;
            streamcin_="";
            if(ret == "")
              return "\n";
            return ret;
        }
        streamcin_+=c;
    }
    if(result == 0)
        vmp::except("eof"); 
    return "";    
}

vmp::vector<vmp::str> input_file(vmp::str filename)
{
    vmp::vector<vmp::str> ret;
    vmp::File file=vmp::fopen_wrap(filename,"r");
    vmp_int c;
    vmp_byte b;
    vmp::str str="";
    while((c=vmp::fgetc_wrap(file)) != EOF)
    {
       b=(vmp_byte)c;
       if(b == '\n')
       {
          if(str.size() != 0)
             ret.push_back(str);
          str="";
       }
       else
          str+=b;
    }
    if(str.size() != 0)
       ret.push_back(str);
    vmp::fclose_wrap(&file);
    return ret;
}

vmp::str input_file_s(vmp::str filename)
{
    vmp::str ret="";
    vmp::File file=vmp::fopen_wrap(filename,"r");
    vmp_int c;
    vmp_byte b;
    while((c=vmp::fgetc_wrap(file)) != EOF)
    {
        b=(vmp_byte)c;
        ret+=b;
    }
    vmp::fclose_wrap(&file);
    return ret;
}

vmp::str input_part_file(vmp::str filename,vmp_int init,vmp_size size)
{
    vmp::str ret="";
    vmp::File file=vmp::fopen_wrap(filename,"r");
    vmp::fseek_wrap(file,init,"set");
    vmp_int c;
    vmp_byte b;
    while((ret.size() < size) && (c=vmp::fgetc_wrap(file)) != EOF)
    {
        b=(vmp_byte)c;
        ret+=b;
    }
    vmp::fclose_wrap(&file);
    return ret;
}

void exit_ok()
{
    exit(EXIT_SUCCESS);
}

void exit_failure_s(vmp::str msg)
{
    vmp::error_process("Failure",msg);
    exit(EXIT_FAILURE); 
}
void exit_failure(const vmp_char *fmt,...)
{
    va_list ap;
    va_start(ap,fmt);    
    vmp::str sfmt=vmp::va_wrap(fmt,ap);
    va_end(ap);
    exit_failure_s(sfmt);
}

vmp::File fd_stream(vmp_int fd) 
{
    return fdopen_wrap(fd,"w");
}

void stream_write(vmp::File f,vmp::str data)
{
    if(f == 0)
       return;
    if(fprintf(f,"%s",data.c_str()) < 0)
       vmp::except_errno();
    vmp::fflush_wrap(f);
}

void stream_close(FILE **f)
{
    vmp::fclose_wrap(f);
}

void copy_file(vmp::str src,vmp::str dst)
{
    vmp::str value=input_file_s(src);
    vmp::output_file_s(dst,true,value);
    
}

void move_file(vmp::str src,vmp::str dst)
{
    vmp::copy_file(src,dst);
    vmp::fs::unlink_wrap(src);
}

}}

