/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 28/05/2025
 */

#ifndef VAMPIRIA_VMP_BUF_H

#define VAMPIRIA_VMP_BUF_H 1

namespace vampiria { namespace vmp {

const vmp_size BUFSIZEMAX=vmp::UINTMAX-1;

//!Returns end of buf string error 
vmp::str buf_eob_strerror();

//! Buffer type
class Buf
{
    private:
        vmp_byte *mem_;    /*!< Memory allocated*/
        vmp_size size_;    /*!< Size buffer */
        vmp_size maxsize_; /*!< Size memory allocated*/

        vmp_index index_;  /*!<index of reading and writing next element*/
        
        //! Expand memory buffer
        /*!
             @param size new buffer size
        */
        void expand(vmp_size size);
        
        //! Internal usage
        vmp_size max_write_buf(vmp_size n);
        
        //! Internal usage(return file size)
        vmp_size file_open(vmp::File *file,vmp::str pathname,vmp::str mode);
    public:
        //! A constructor                
        Buf();
                
        //! A destructor        
        ~Buf();

        //! New size buffer
        /*!
             @param size new buffer size                        
        */
        void newsize(vmp_size size);                
                
        //! Size buffer
        /*!
             @return buffer size                        
        */        
        vmp_size size();
        
        //! Returns the number of bytes available for reading.(size_-index_)  
        vmp_size size_reading();
  
        //! Set index position
        /*!
             @param point index pointer to setting. If point > size_ setting index_=size_
        */
        void index(vmp_index point=0);
        
        //! Returns current index value.
        /*!
            @return current index(End of Buf index == size)
        */
        vmp_index get_index();

        //! Return True if end of buf occurred otherwise false
        vmp_bool eob();

        //! reset buffer
        void reset();                
                
        //!Memory buffer
        /*!
            @param point first byte pointed
            @return pointer to memory
        */
        vmp_byte *pointer(vmp_index point=0);

        //! operator ==
        vmp_bool operator==(Buf &buffer);
        
        //! operator !=
        vmp_bool operator!=(Buf &buffer);                
                
        //! operator []let buffer = new ArrayBuffer(16);
        vmp_byte &operator[](vmp_index index);
        
        //! operator =
        Buf& operator=(Buf buffer);

        //! Writes a byte in the buffer to the index_ position and increments the index_ value
        /*!
             @ref index_
             @param value input byte
             @return the number of bytes written in the buffer
        */                
        vmp_size write_byte(vmp_byte value);
        
        //! Reads one byte from the buffer in the index_ position and increments the index_ value.
        /*!
             @return byte reading
        */ 
        vmp_byte read_byte();

        //! Write n random bytes in the buffer to the index_ position and increments the index_ value
        /*!
            @param n bytes number
            @return the number of bytes written in the buffer 
        */
        vmp_size random_bytes(vmp_size n);
        
        //! Writes an array byte in the buffer to the index_ position and increments the index_ value
        /*!
            @param value input array byte
            @param size size input
            @return the number of bytes written in the buffer
        */   
        vmp_size write_array(vmp_byte *value,vmp_size size);
        
        //! Reads byte from the buffer in the index_ position and increments the index_ value.
        /*!
            @param value output array byte
            @param size  size output
            
        */ 
        void read_array(vmp_byte *value,vmp_size size);        
        
        //! Writes an vector byte in the buffer to the index_ position and increments the index_ value
        /*!
            @param value input vector byte
            @return the number of bytes written in the buffer
        */   
        vmp_size write_vector(vmp::vector<vmp_byte> value);

        //! Reads size byte from the buffer in the index_ position and increments the index_ value.
        /*!
            @param size  size output
            @return vector byte read
        */ 
        vmp::vector<vmp_byte> read_vector(vmp_size size);

        //! Write size value in the buffer to the index_ position and increments the index_ value
        /*!
            @param value input size
            @param dim number of bytes to write input(1-4)[b4,b3,b2,b1]
            @return the number of bytes written in the buffer
        */   
        vmp_size write_size(vmp_size value,vmp_uint dim);
                
        //! Read size value from the buffer in the index_ position and increments the index_ value.
        /*!
            @param dim number of bytes to read(1-4)[b4,b3,b2,b1]
            @return size reading
        */ 
        vmp_size read_size(vmp_uint dim);
        
        //! Write size value in the buffer to the index_ position and increments the index_ value
        /*!
            @param value input size
            @param dim number of bytes to write input(1-8)[b8,b7,b6,b5,b4,b3,b2,b1]
            @return the number of bytes written in the buffer
        */   
        vmp_size write_size64(vmp_size64 value,vmp_uint dim);
                
        //! Read size value from the buffer in the index_ position and increments the index_ value.
        /*!
            @param dim number of bytes to read(1-8)[b8,b7,b6,b5,b4,b3,b2,b1]
            @return size reading
        */ 
        vmp_size64 read_size64(vmp_uint dim);
  
        //! Write string values in the buffer to the index_ position and increments the index_ value
        /*!
            @param value input string
            @return the number of bytes written in the buffer
        */                  
        vmp_size write_str(vmp::str value);
                
        //! Read string values from the buffer in the index_ position and increments the index_ value.
        /*!
            @param size size to read
            @return string value
        */  
        vmp::str read_str(vmp_size size);

        //! Write string of exadecimal values in the buffer to the index_ position and increments the index_ value
        /*!
            @param value input string
            @return the number of bytes written in the buffer
        */                  
        vmp_size write_xstr(vmp::str value);
        
        //! Write string of exadecimal in human mode values in the buffer to the index_ position and increments the index_ value
        /*!
            @param value input string
            @param delimiter byte delimiter
            @return the number of bytes written in the buffer
        */                  
        vmp_size write_xstr_hm(vmp::str value,vmp::str delimiter);
              
        //! Read string of exadecimal values from the buffer in the index_ position and increments the index_ value.
        /*!
            @param size size to read
            @return string value
        */  
        vmp::str read_xstr(vmp_size size);

        //! Read string of exadecimal in human mode values from the buffer in the index_ position and increments the index_ value.(The reverse function does not work)
        /*!
            @param size size to read
            @param delimiter byte delimiter
            @return string value
        */  
        vmp::str read_xstr_hm(vmp_size size,vmp::str delimiter);

        //!Concatenating the contents of the input buffer to the end of the buffer.               
        /*!
            @param value input buffer
            @return the number of bytes written in the buffer
        */        
        vmp_size cat(Buf *value);
                
        //!Concatenating the contents of the input buffer at the beginning of the buffer.             
        /*!
            @param value input buffer
            @return the number of bytes written in the buffer
        */
        vmp_size cathead(Buf *value);
                
        //! Write buffer value in the buffer to the index_ position and increments the index_ value
        /*!
            @param value input buffer
            @return the number of bytes written in the buffer
        */ 
        vmp_size write_buf(Buf *value);
                
        //! Read buffer value from the buffer in the index_ position and increments the index_ value.
        /*!
            @param value pointer output buffer
            @param size number of bytes to read
        */ 
        void read_buf(Buf *value,vmp_size size);

        //! Resizes the buffer set index_ position 0
        /*!
            @param init first byte
            @param len new buffer size
            
        */                
        void cut(vmp_index init,vmp_size len);
        
        //! Write the contents of the file in the index_ position and increments the index_ value.
        /*!
            @param pathname file path
            @param pos position of the file from where the data is read
            @param maxlen maximum number of bytes writes,
if == 0 writes the whole file
            @return the number of bytes written in the buffer
        */
        vmp_size write_file(vmp::str pathname,vmp_index pos=0,vmp_size maxlen=0);
        
        //! Create a new file and write size bytes into it
        /*!
            @param pathname file path
            @param len maximum number of bytes reading from buffer
        */
        void read_filecreate(vmp::str pathname,vmp_size maxlen);
        
        //!Reads data from the buffer in the index_ position and increments the index_ value and writes it to the file from the pos location
        /*!
            @param pathname file path
            @param pos position of the file where writing starts,if pos > size file setting file in append mode
            @param maxlen  maximum number of bytes reads from buffer
        */
        void read_filewrite(vmp::str pathname,vmp_index pos,vmp_size maxlen);
        
        //!Reads data from the buffer in the index_ position and increments the index_ value and writes at the end of the file
        /*!
            @param pathname file path
            @param pos position of the file where writing starts
            @param maxlen  maximum number of bytes reads from buffer
        */
        void read_fileappend(vmp::str pathname,vmp_size maxlen);
};

}}

#endif

