/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 28/05/2025
 */

#include "vmp.h"

namespace vampiria { namespace vmp {

#define BUF_SIZE_INIT    32
#define BUF_EOB_ERROR "End Of Buf Error"

vmp::str buf_eob_strerror()
{
    return BUF_EOB_ERROR;
}

Buf::Buf()
{
    maxsize_=BUF_SIZE_INIT;         
    mem_=(vmp_byte *) vmp::malloc_wrap(maxsize_);
    reset();        
}

Buf::~Buf()
{
    vmp::free_wrap((void **) &mem_);
}

void Buf::expand(vmp_size size)
{
    vmp_uchar *tmp;        
    if(size > maxsize_)
    {
        do
        {
            if(maxsize_ > vmp::BUFSIZEMAX/2)
                maxsize_=vmp::BUFSIZEMAX;
            else
                maxsize_ *= 2;  
        }
        while(size > maxsize_);
        tmp=(vmp_byte *) vmp::malloc_wrap(maxsize_);
        vmp::bzero_wrap((void *)tmp,maxsize_);                
        vmp::memcpy_wrap((void *)tmp,(void *)mem_,size_);
        vmp::free_wrap((void **)&mem_);
        mem_=tmp;
    }
    if(size > size_)
        size_=size;
}

vmp_size Buf::max_write_buf(vmp_size n)
{
    vmp_size bmax=vmp::BUFSIZEMAX - index_;
    if(bmax < n)
        return bmax;
    return n;
}

void Buf::newsize(vmp_size size)
{
    if(size > size_)        
        expand(size);
    else if(size < size_)
    {        
        vmp::bzero_wrap((void *)(&(mem_[size])),(size_-size));
        size_=size;
    }
}

vmp_size Buf::size()
{
    return size_;
}

vmp_size Buf::size_reading()
{
    return (size_-index_);
}

void Buf::index(vmp_index point)
{
    if(point < size_)
        index_=point;
    else
        index_=size_;
}

vmp_index Buf::get_index()
{
    return index_;
}

vmp_bool Buf::eob()
{
    if(index_ == size_)
        return true;
    return false;
}
         
void Buf::reset()
{
    vmp::bzero_wrap((void *)mem_,maxsize_);        
    size_=0;
    index();
}
                
vmp_byte *Buf::pointer(vmp_index point)
{
    if(point < size_)        
        return &(mem_[point]);
    return 0;
}

vmp_bool Buf::operator==(Buf &buffer)
{
    if(size_ == buffer.size())    
    {
        for(vmp_index i=0;i<size_;i++)
            if(mem_[i] != (*(buffer.pointer(i))))
                return false;
        return true;        
    }
    return false;
}

vmp_bool Buf::operator!=(Buf &buffer)
{
    if((*this) == buffer)
        return false;
    return true;
}

vmp_byte &Buf::operator[](vmp_index index)
{
    if(index >= size_)
        vmp::except_s("Overflow Buffer");
    return mem_[index];
}

Buf& Buf::operator=(Buf buffer)
{
    newsize(buffer.size());
    vmp::memcpy_wrap(mem_,buffer.pointer(),buffer.size());
    index(buffer.index_);    
    return (*this);
}

vmp_size Buf::write_byte(vmp_byte value)
{
    vmp_size ret=max_write_buf(1);
    if(ret != 0)
    {
        vmp_index nindex=index_+ret;
        expand(nindex);
        mem_[index_]=value;
        index(nindex);
    }
    return ret;
}

vmp_byte Buf::read_byte()
{
    vmp_byte ret=0x00;
    if(eob())
        vmp::except("%s read byte",BUF_EOB_ERROR);
    ret=mem_[index_];
    index(index_+1);
    return ret;
}

vmp_size Buf::random_bytes(vmp_size n)
{
    vmp_size ret=max_write_buf(n);
    vmp_index nindex=index_+ret;
    expand(nindex);
    vmp::srand_wrap(vmp::time::time_wrap()+vmp::getpid_wrap());
    for(vmp_index i=0;i<ret;i++)
    {
        mem_[i+index_]=(vmp_byte)(vmp::rand_wrap() % 256);
        vmp::srand_wrap(vmp::rand_wrap());
    }
    index(nindex);
    return ret;
}

vmp_size Buf::write_array(vmp_byte *value,vmp_size size)
{
    vmp_size ret=max_write_buf(size);
    vmp_index nindex=index_+ret;      
    expand(nindex);        
    for(vmp_index i=0;i<ret;i++)
        mem_[i+index_]=value[i];
    index(nindex);
    return ret;
}

void Buf::read_array(vmp_byte *value,vmp_size size)
{
    for(vmp_index i=0;i<size;i++)
    {
        try
        {
            value[i]=read_byte();
        }
        catch(vmp::exception &x)
        {
            vmp::except("%s read array size %u",BUF_EOB_ERROR,size);
        }
    }
}

vmp_size Buf::write_vector(vmp::vector<vmp_byte> value)
{
    vmp_size ret=max_write_buf(value.size());
    vmp_index nindex=index_+ret;        
    expand(nindex);        
    for(vmp_index i=0;i<ret;i++)
        mem_[i+index_]=value[i];
    index(nindex);
    return ret;
}

vmp::vector<vmp_byte> Buf::read_vector(vmp_size size)
{
    vmp::vector<vmp_byte> ret;
    for(vmp_index i=0;i<size;i++)
    {
        try
        {
            ret.push_back(read_byte());
        }
        catch(vmp::exception &x)
        {
            vmp::except("%s read vector size %u",BUF_EOB_ERROR,size);
        }
    }
    return ret; 
}

vmp_size Buf::write_size(vmp_size value,vmp_uint dim)
{
    vmp_size ret=max_write_buf(dim);
    vmp_index nindex;
    if(ret == 0)
        return ret;
    else if(ret > 4)
        ret=4;
    nindex=index_+ret;
    expand(nindex);        
    for(vmp_index i=index_;i<nindex;i++)
        mem_[i]=(vmp_byte)((value >> ((nindex-i-1)*8)) & 0xFF);
    index(nindex);
    return ret;   
}
                
vmp_size Buf::read_size(vmp_uint dim)
{
    vmp_size ret=0;
    vmp_uint d;
    vmp_index nindex;
    if(dim == 0)
        return ret;
    else if(dim > 4)
        d=4;
    else
        d=dim;
    nindex=index_+d;
    for(vmp_index i=index_;i<nindex;i++)
    {
        if(i<size_)
           ret+=(vmp_size)(mem_[i] << ((nindex-i-1)*8));       
        else
           vmp::except("%s read size",BUF_EOB_ERROR);;
    }
    index(nindex);    
    return ret;       
} 

vmp_size Buf::write_size64(vmp_size64 value,vmp_uint dim)
{
    vmp_size ret=max_write_buf(dim);
    vmp_index nindex;
    if(ret == 0)
        return ret;
    else if(ret > 8)
        ret=8;
    nindex=index_+ret;
    expand(nindex);        
    for(vmp_index i=index_;i<nindex;i++)
        mem_[i]=(vmp_byte)((value >> ((nindex-i-1)*8)) & 0xFF);
    index(nindex);
    return ret;   
}
                
vmp_size64 Buf::read_size64(vmp_uint dim)
{
    vmp_size ret=0;
    vmp_uint d;
    vmp_index nindex;
    if(dim == 0)
        return ret;
    else if(dim > 8)
        d=8;
    else
        d=dim;
    nindex=index_+d;
    for(vmp_index i=index_;i<nindex;i++)
    {
        if(i<size_)
           ret+=(vmp_size)(mem_[i] << ((nindex-i-1)*8));       
        else
           vmp::except("%s read size",BUF_EOB_ERROR);;
    }
    index(nindex);    
    return ret;       
} 

vmp_size Buf::write_str(vmp::str value)
{
    vmp_size ret=max_write_buf(value.size());
    vmp_size nindex=index_+ret;
    expand(nindex);        
    for(vmp_index i=0;i<value.size();i++)
        mem_[i+index_]=value[i];                
    index(nindex);
    return ret;       
}
                
vmp::str Buf::read_str(vmp_size size)
{
    vmp::str ret="";     
    vmp_index i,nindex;
    for(i=0,nindex=index_;i<size && nindex<size_;i++)
        ret+=mem_[nindex++]; 
    if(i != size)
        vmp::except("%s read string size %d",BUF_EOB_ERROR,size);
    index(nindex);     
    return ret;
} 

vmp_size Buf::write_xstr(vmp::str value)
{
    vmp::vector<vmp_byte> bytes;
    try
    {
        bytes=vmp::unicode::xstr_tobytes(value);
    }
    catch(vmp::exception &x)
    {
        vmp::except("write xstr %s",x.what());
    }
    return write_vector(bytes);
}

vmp_size Buf::write_xstr_hm(vmp::str value,vmp::str delimiter)
{
    vmp::vector<vmp_byte> bytes;
    try
    {
        bytes=vmp::unicode::xstr_tobytes_hm(value,delimiter);
    }
    catch(vmp::exception &x)
    {
        vmp::except("write xstr %s",x.what());
    }
    return write_vector(bytes);
}

vmp::str Buf::read_xstr(vmp_size size)
{
    vmp::vector<vmp_byte> bytes;
    try
    {
        bytes=read_vector(size);
    }
    catch(vmp::exception &x)
    {
        vmp::except("%s read xstr size %u",BUF_EOB_ERROR,size);
    }
    return vmp::unicode::bytes_toxstr(bytes);
}

vmp::str Buf::read_xstr_hm(vmp_size size,vmp::str delimiter)
{
    vmp::vector<vmp_byte> bytes;
    try
    {
        bytes=read_vector(size);
    }
    catch(vmp::exception &x)
    {
        vmp::except("%s read xstr size %u",BUF_EOB_ERROR,size);
    }
    return vmp::unicode::bytes_toxstr_hm(bytes,delimiter);
}

vmp_size Buf::cat(Buf *value)
{
    index(size_);   
    return write_buf(value);
}

vmp_size Buf::cathead(Buf *value)
{
    vmp_size ret=max_write_buf(value->size());
    index();
    vmp_size s=size_;
    vmp_byte tmp[s];
    read_array(tmp,s);
    reset();
    write_buf(value);
    index(ret);
    write_array(tmp,s);
    return ret;   
}

vmp_size Buf::write_buf(Buf *value)
{
    vmp_size ret=max_write_buf(value->size());      
    vmp_index nindex=index_+ret;
    expand(nindex);
    vmp::memcpy_wrap(pointer(index_),value->pointer(),ret);
    index(nindex);
    return ret;
}

void Buf::read_buf(Buf *value,vmp_size size)
{
    vmp_index nindex=index_+size;
    if(nindex > size_)
        vmp::except("%s read buf size %d",BUF_EOB_ERROR,size);
    value->newsize(size);        
    vmp::memcpy_wrap(value->pointer(),pointer(index_),size);        
    index(nindex);
}

void Buf::cut(vmp_index init,vmp_size len)
{
    vmp_index p1,p2;
    for(p2=init,p1=0;(p2 < (init+len) && p2 < size_);p2++,p1++)
        mem_[p1]=mem_[p2];        
    newsize(p1);
    index();        
}

vmp_size Buf::file_open(vmp::File *file,vmp::str pathname,vmp::str mode)
{
    vmp::fs::FileStat stat;
    try
    {
        (*file)=vmp::fopen_wrap(pathname,mode);
        stat.set(pathname);
        vmp::fs::Mode_t_wrap mode_t;
        stat.mode(&mode_t);
        if(!mode_t.isreg())
            vmp::except_s("not regular file");
    }
    catch(vmp::exception &x)
    {
        vmp::except("vmp::Buf::fileopen(pathname=%s) %s",pathname.c_str(),x.what());
    }
    return stat.size();
}

vmp_size Buf::write_file(vmp::str pathname,vmp_index pos,vmp_size maxlen)
{
    vmp::File file;
    vmp_size s=file_open(&file,pathname,"r");
    maxlen=max_write_buf(maxlen);
    try
    {
        if(pos < s)
        {
            s -=pos;
            if((maxlen != 0) && maxlen < s)
                s=maxlen;
            vmp_index nindex=s+index_;
            expand(nindex);
            vmp::fread_wrap((void *)pointer(index_),1,s,file);
            index(nindex);
        }
    }
    catch(vmp::exception &x)
    {
        vmp::fclose_wrap(&file);
        vmp::except_s(x.what());
    }
    vmp::fclose_wrap(&file);
    return s;
}

void Buf::read_filecreate(vmp::str pathname,vmp_size maxlen)
{
    vmp::File file;
    file_open(&file,pathname,"w");
    try
    {
        vmp_size s=size_reading();
        if(s > 0)
        {
            if(maxlen < s)
                s=maxlen;
            vmp_index nindex=index_+s;
            vmp::fwrite_wrap((void *)pointer(index_),1,s,file);
            index(nindex); 
        }   
    }
    catch(vmp::exception &x)
    {
        vmp::fclose_wrap(&file);
        vmp::except_s(x.what());
    }
    vmp::fclose_wrap(&file);
}

void Buf::read_filewrite(vmp::str pathname,vmp_index pos,vmp_size maxlen)
{
    vmp::File file;
    vmp_size fsize=file_open(&file,pathname,"w+");
    try
    {
        vmp_size s=size_reading();
        if(s > 0)
        {
            if(pos < fsize)
                vmp::fseek_wrap(file,pos,"set");
            else
                vmp::fseek_wrap(file,0,"end");
            if(maxlen < s)
                s=maxlen;
            vmp_index nindex=index_+s;
            vmp::fwrite_wrap((void *)pointer(index_),1,s,file);
            index(nindex); 
        }   
    }
    catch(vmp::exception &x)
    {
        vmp::fclose_wrap(&file);
        vmp::except_s(x.what());
    }
    vmp::fclose_wrap(&file);
}

void Buf::read_fileappend(vmp::str pathname,vmp_size maxlen)
{
    vmp::File file;
    file_open(&file,pathname,"a+");
    try
    {
        vmp_size s=size_reading();
        if(s > 0)
        {
            if(maxlen < s)
                s=maxlen;
            vmp_index nindex=index_+s;
            vmp::fwrite_wrap((void *)pointer(index_),1,s,file);
            index(nindex); 
        }
    }
    catch(vmp::exception &x)
    {
        vmp::fclose_wrap(&file);
        vmp::except_s(x.what());
    }
    vmp::fclose_wrap(&file);
}

}}

