/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 30/05/2025
 */

#include <string>
#include <cstring>
#include <vector>
#include <utility>
#include <limits>
#include <memory>
#include <functional>
#include <tuple>
#include <type_traits>
#include <typeinfo>
#include <set>
#include <queue>
#include <memory>
#include <algorithm> 

#include <string.h>
#include <strings.h>
#include <limits.h>

#ifndef VAMPIRIA_VMP_BASETYPES_H

#define VAMPIRIA_VMP_BASETYPES_H 1

namespace vampiria { namespace vmp {

//! string 
typedef std::string str;

//! string size type
typedef size_t str_size;

//! str_size end string
const vmp::str_size str_npos=std::string::npos; 

//! Allocates size bytes and returns a pointer to the allocated memory
/*!
    @param size memory size allocation
    @return a pointer to  the  allocated memory(except error)
*/
void *malloc_wrap(vmp_size size);

//!Frees the memory space pointed  to  by  (*ptr) and set (*ptr)=0
/*!
    @param ptr pointer to pointer free memory
*/
void free_wrap(void **ptr);

//!Erases  the  data  in the n bytes of the memory starting at the location pointed to by s, by writing zeroes (bytes con‐taining '\0') to that area.
/*!
    @param s pointer memory
    @param n memory size
*/
void bzero_wrap(void *s,vmp_size n);

//!Copies n bytes from memory area src to memoryarea dest. 
/*!
    @param dest memory destination
    @param src memory source
    @param n size to copy
*/
void memcpy_wrap(void *dest, const void *src,vmp_size n);

//!Copies the string pointed to by src, including the terminating null byte ('\0'), to the buffer  pointed  to  by  dest
/*!
    @param dest string destination
    @param src string source
    @param n max bytes to copy
*/
void strncpy_wrap(vmp_char *dest, const vmp_char *src,vmp_size n);

//! Creates a string from linux error value
/*!
     @param err error value
     @return the resulting string 
*/
vmp::str strerror_wrap(vmp_int err);

//! vector
template<typename DATA>
using vector=std::vector<DATA>;

//! pair
template<typename DATA1,typename DATA2>
using pair=std::pair<DATA1,DATA2>;

//! make_pair
template<class DATA1,class DATA2>
vmp::pair<DATA1,DATA2> make_pair(DATA1 &&t1,DATA2 &&t2)
{
    return std::make_pair(t1,t2);
}

//! Smart pointer that owns and manages another object through a pointer and disposes of that object when the unique_ptr goes out of scope
template<typename OBJ>
using unique_ptr=std::unique_ptr<OBJ>;

//!Is a general-purpose polymorphic function wrapper. 
template<typename FN>
using function=std::function<FN>;

//!Is a fixed-size collection of heterogeneous values.It is a generalization of vmp::pair
template<class... Types>
using tuple=std::tuple<Types...>;

//!Provides compile-time indexed access to the types of the elements of the vmp::tuple
template<vmp_size I,class T>
using tuple_element=std::tuple_element<I,T>;

//!Provides the nested type type, which is a trivial standard-layout type suitable for use as uninitialized storage for any object whose size is at most Len and whose alignment requirement is a divisor of Align. 
template<vmp_size Len,vmp_size Align=0>
using aligned_storage=std::aligned_storage<Len,Align>;

//!Provides the member constant value equal to the alignment requirement of the type T, as if obtained by an alignof expression. If T is an array type, returns the alignment requirements of the element type. If T is a reference type, returns the alignment requirements of the type referred to.If alignof(T) is not a valid expression, the behavior is undefined. 
template<class T>
using alignment_of=std::alignment_of<T>;

//!The class type_info holds implementation-specific information about a type, including the name of the type and means to compare two types for equality or collating order. This is the class returned by the typeid operator. 
using type_info=std::type_info;

//!Sets are containers that store unique elements following a specific order
template < class T,                        //! set::key_type/value_type
           class Compare = std::less<T>,        //! set::key_compare/value_compare
           class Alloc = std::allocator<T>      //! set::allocator_type
           > 
using set=std::set<T,Compare,Alloc>;

//! see doc c++ std::less
template<class T>
using less=std::less<T>;

//! see doc c++ std::less_equal
template<class T>
using less_equal=std::less_equal<T>;

//! see doc c++ std::greater
template<class T>
using greater=std::greater<T>;

//! see doc c++ std::greater_equal
template<class T>
using greater_equal=std::greater_equal<T>;

//!see doc c++ std::priority queue
template <class T, class Container = vmp::vector<T>,class Compare = std::less<typename Container::value_type> > 
using priority_queue=std::priority_queue<T,Container,Compare>;

//! see doc c++ std::Queue
template <class T, class Container = std::deque<T> >
using queue=std::queue<T,Container>;

//!Takes an input array and returns an ordered array
/*
    @param v input array
    @param reverse false ascending order,true descending
    @param duplicate if false remove duplicate value
    @return ordered carrier
*/
template <class T>
vmp::vector<T> sort(vmp::vector<T> v,vmp_bool reverse=false,vmp_bool duplicate=true)
{
    vmp::vector<T> ret=v;
    if(reverse)
        std::sort(ret.begin(),ret.end(),std::greater<T>());
    else
        std::sort(ret.begin(),ret.end());
    if(!duplicate)
        ret.erase(std::unique(ret.begin(),ret.end()),ret.end());
    return ret;
}

//! Checks if a value is inside the vector
/*!
    @param value value to be verified
    @param values value vector
    @return true if the value is in the vector,otherwise false
*/
template <class T>
vmp_bool invector(T value,vmp::vector<T> values)
{
    for(vmp_index i=0;i<values.size();i++)
        if(value == values[i])
            return true;
    return false;
}

//! Inserts a value into the vector if it is not present
/*!
    @param value value to be inserted
    @param values pointer value vector
    @return if the value was not present and was entered, otherwise false
*/
template <class T>
vmp_bool insinvector(T value,vmp::vector<T> *values)
{
    if(vmp::invector<T>(value,*values))
        return false;
    values->push_back(value);
    return true;
}

//! Delete a value within the vector(first occurrence)
/*!
    @param value value to be deleted
    @param values pointer value vector
    @return if the value was deleted, false if the value was not inside the vector
*/
template <class T>
vmp_bool delinvector(T value,vmp::vector<T> *values)
{
    for(vmp_index i=0;i<values->size();i++)
    {
        if(value == (*values)[i])
        {
            for(vmp_index j=i+1;j<values->size();j++)
                (*values)[j-1]=(*values)[j];
            values->erase(values->end() - 1);
            return true;
        }
    }
    return false;
}

const vmp_int16 INT16MIN=std::numeric_limits<vmp_int16>::min(); /*!< Minimum limit short integer*/
const vmp_int16 INT16MAX=std::numeric_limits<vmp_int16>::max(); /*!< Maximum limit short integer*/
const vmp_int INTMIN=std::numeric_limits<vmp_int>::min(); /*!< Minimum limit integer*/
const vmp_int INTMAX=std::numeric_limits<vmp_int>::max(); /*!< Maximum limit integer*/
const vmp_int64 INT64MIN=std::numeric_limits<vmp_int64>::min(); /*!< Minimum limit short integer 54 bit*/
const vmp_int64 INT64MAX=std::numeric_limits<vmp_int64>::max(); /*!< Maximum limit short integer 64 bit*/
const vmp_uint16 UINT16MIN=std::numeric_limits<vmp_uint16>::min();/*!< Minimum limit unsigned short integer */
const vmp_uint16 UINT16MAX=std::numeric_limits<vmp_uint16>::max(); /*!< Maximum limit unsigned short integer*/
const vmp_uint UINTMIN=std::numeric_limits<vmp_uint>::min(); /*!< Minimum limit unsigned integer */
const vmp_uint UINTMAX=std::numeric_limits<vmp_uint>::max(); /*!< Maximum limit unsigned integer*/
const vmp_uint UINT64MIN=std::numeric_limits<vmp_uint>::min(); /*!< Minimum limit unsigned integer 64 bit*/
const vmp_uint UINT64MAX=std::numeric_limits<vmp_uint>::max(); /*!< Maximum limit unsigned integer 64 bit*/
const vmp_char CHARMIN=std::numeric_limits<vmp_char>::min(); /*!< Minimun limit char*/
const vmp_char CHARMAX=std::numeric_limits<vmp_char>::max(); /*!< Maximum limit char*/
const vmp_uchar UCHARMIN=std::numeric_limits<vmp_uchar>::min(); /*!< Minimun limit unsigned char*/
const vmp_uchar UCHARMAX=std::numeric_limits<vmp_uchar>::max(); /*!< Maximum limit unsigned char*/

const vmp_uint POSIX_ARG_MAX=_POSIX_ARG_MAX; /*!Arguments max length*/
const vmp_uint RANDMAX=RAND_MAX; /*! Random generator max number*/ 

}}

#endif

