/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 13/08/2024
 */

#include "rawnet.h"

namespace vampiria { namespace rawnet {

void empty_rawloop(vmp::Buf *buf,void *arg)
{
}

rawnet::rawnet_t open_dead(vmp_int linktype,vmp_int snaplen)
{
    rawnet_t ret=pcap_open_dead(linktype,snaplen);
    if(ret == 0)
        vmp::except_s("rawnet::open_dead() open error");
    return ret;
}

rawnet::rawnet_t open_offline(vmp::str file)
{
    rawnet::rawnet_t ret;
    vmp_char errbuf[PCAP_ERRBUF_SIZE];
    vmp::bzero_wrap(errbuf,PCAP_ERRBUF_SIZE);
    ret=pcap_open_offline(file.c_str(),errbuf);
    if(ret == 0)
        vmp::except("rawnet::open_offline() error '%s'",errbuf);
    return ret;
}

rawnet::rawnet_t open_live(vmp::str ifname)
{
    vmp_int mtu;
    try
    {
        mtu=net::iface_ipv4_mtu(ifname);
    }
    catch(vmp::exception &x)
    {
        mtu=rawnet::MTU_DEFAULT;
    }
    vmp_char errbuf[PCAP_ERRBUF_SIZE];
    vmp::bzero_wrap(errbuf,PCAP_ERRBUF_SIZE);
    rawnet_t handle=pcap_open_live(ifname.c_str(),mtu,1,500,errbuf);
    if(handle == 0)
        vmp::except("rawnet::open_live() error '%s'",errbuf);
    return handle;
}

vmp_int datalink_rawnet(rawnet::rawnet_t raw)
{
    if(raw == 0)
        vmp::except_s("rawnet::datalink_rawnet() error 'null input pointer'");
    vmp_int ret= pcap_datalink(raw);
    if(ret == PCAP_ERROR_NOT_ACTIVATED)
        vmp::except_s("rawnet::datalink_rawnet() error 'datalink not actived'");
    return ret;
}

void filter_rawnet(rawnet::rawnet_t raw,vmp::str filter)
{
    if(raw != 0)
    {
        struct bpf_program fp;
        if((pcap_compile(raw,&fp,filter.c_str(),0,0) == -1) || (pcap_setfilter(raw,&fp) == -1))
            vmp::except("rawnet::filter_rawnet() error '%s'",pcap_geterr(raw));
    }
}

void raw_loop_cb(vmp_uchar *arg,const struct pcap_pkthdr *pkthdr,const vmp_uchar *packet)
{
    vmp::pair<rawnet::RAWCBLOOP,void *> *cbarg= (vmp::pair<rawnet::RAWCBLOOP,void *> *) arg;
    vmp::Buf buf;
    buf.write_array((vmp_byte *)packet,pkthdr->len);
    buf.index();
    cbarg->first(&buf,cbarg->second);
    buf.reset();
}

void loop_rawnet(rawnet::rawnet_t raw,rawnet::RAWCBLOOP cb,void *arg)
{
    rawnet::RAWCBLOOP callback;
    if(cb == 0)
        callback=rawnet::empty_rawloop;
    else
        callback=cb;
    vmp::pair<rawnet::RAWCBLOOP,void *> cbarg(callback,arg);
    if(pcap_loop(raw,0,rawnet::raw_loop_cb,(vmp_uchar *)&cbarg) == -1)
        vmp::except_s(pcap_geterr(raw));   
}

void breakloop_rawnet(rawnet::rawnet_t raw)
{
    if(raw != 0)
        pcap_breakloop(raw);
}

void inject_rawnet(rawnet::rawnet_t raw,vmp::Buf *buf)
{
    vmp::except_check_pointer((void *)buf,"rawnet::inject_rawnet() null pointer input buffer");
    if(raw != 0)
        if(pcap_inject(raw,buf->pointer(),buf->size()) == -1)
            vmp::except("rawnet::inject_rawnet() error '%s'",pcap_geterr(raw));
}

void close_rawnet(rawnet::rawnet_t *praw)
{
    if((*praw) != 0)
       pcap_close(*praw);
    (*praw)=0;
}

rawnet::dumper_t open_dumper(rawnet::rawnet_t handle,vmp::str wfile)
{
    rawnet::dumper_t ret;
    vmp::except_check_pointer((void *)handle,"rawnet::open_dump() null input handle");
    ret=pcap_dump_open(handle,wfile.c_str());
    if(ret == 0)
        vmp::except("rawnet::open_dump() '%s'",pcap_geterr(handle));
    return ret;
}

void write_dumper(rawnet::dumper_t dumper,vmp::Buf *buf)
{
    if((dumper != 0) && (buf != 0) && (buf->size() !=0))
    {
        pcap_pkthdr pcap_hdr;
        pcap_hdr.caplen = buf->size();
        pcap_hdr.len = pcap_hdr.caplen;
        pcap_dump((vmp_byte *)dumper,&pcap_hdr,buf->pointer());
    }
}

void close_dumper(rawnet::dumper_t *pdumber)
{
    if((*pdumber) != 0)
        pcap_dump_close(*pdumber);
    (*pdumber) = 0;
}

}}

