/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 13/08/2024
 */
 
#ifndef VAMPIRIA_RAWNET_ARPUI_H

#define VAMPIRIA_RAWNET_ARPUI_H 1

namespace vampiria { namespace rawnet {

typedef void (*EVTRAWPKGCB)(event::Cell *cell,vmp::str name,packet::Packet *packet);/*!< Callback to return packages read from scans*/

//! Empty file associated with the callback EVTRAWPKGCB 
void empty_rawpkg_ev(event::Cell *cell,vmp::str name,packet::Packet *packet);

//! Internal usage
class Arp4Evt:public rawnet::EventRawNet
{
    public:
        Arp4Evt();
        ~Arp4Evt();
        
        vmp::str ipsrc_;
        vmp::str macsrc_;
        vmp::vector<vmp::str> targets_;
        vmp::vector<vmp::Buf *> bufs_;
        event::Cell *timer_;
        
        void evt_arp4_active();
        
        vmp::str filter_;
        vmp::Table<vmp::str,event::Cell *> sniffers_;
};

//! Internal usage
class Arp4Timer:public event::EventTimer
{
    public:
        Arp4Timer();
        ~Arp4Timer();
        
        event::Cell *parent_;
};

//! Internal usage
class Arp4Sniffer:public rawnet::EventRawNet
{
    public:
        Arp4Sniffer();
        ~Arp4Sniffer();

        vmp::str name_;
        event::Cell *parent_;
        
        rawnet::EVTRAWPKGCB recv_;
        
};

//! Arp protocol utility.Inject in the network of arp request ipv4 reads the packages generated by targets
class Arp4UI:public event::UI
{
    private:
        packet::PacketHelper *helper_;
        vmp::str ifname_;/*!< Interface name*/
        vmp::str ip_;/*!<ipv4 address associated with ifname*/
        vmp::str mac_;/*!<mac address associated with ifname*/
        
        vmp::utils::Storage<rawnet::Arp4Evt> evts_;/*!<Storage event master*/
        vmp::utils::Storage<rawnet::Arp4Timer> timers_;/*!<Storage event timer*/
        vmp::utils::Storage<rawnet::Arp4Sniffer> sniffers_;/*!<Storage event sniffer*/
        
        rawnet::EVTRAWPKGCB arprecv_;/*!<sniffer callback*/
        event::EVTCB carp_;/*!<close Event callback*/
        
    public:
       //!A Constructor
       /*!
            @param manager input manager
            @param helper  pachet helper used to read packages from sniffers
            @param ifname  interface name(ex. eth0)
            
       */
       Arp4UI(event::Manager *manager,packet::PacketHelper *helper,vmp::str ifname);
       
       //!A Destructor
       ~Arp4UI();
       
       //!Internal usage
       void arp_recv(event::Cell *cell,vmp::Buf *buf);
       
       //!Virtual class implemented
       vmp::str identity(event::Cell *cell);
       
       //!Virtual class implemented
       void close_event(event::Cell *cell);
       
       //!Virtual class implemented
       void free_ref(event::Cell *cell);
       
       //!Set callback event
       /*!
            @param arprecv sniffer read callback
            @param carp close callback 
       */
       void set_event(EVTRAWPKGCB arprecv,event::EVTCB carp);
       
       //!Create a new event for an arp operation
       /*!
           @param ipsrc ip source arp request
           @param macsrc mac source arp request
           @param targets ipv4 target arp request address list
           @return new cell event or except in case of failure
       */
       event::Cell *arp4evt(vmp::str ipsrc,vmp::str macsrc,vmp::vector<vmp::str> targets);

       //! Start packet injection.If timeval = 0.0 performs once,if timeval > 0.0 start loop
       /*!
           @param cell main event cell
           @return void or except in case of failure
       */
       void start(event::Cell *cell,vmp::time::Time timeval=0.0);
       
       //! Stop packet injection
       /*!
           @param cell main event cell
       */
       void stop(event::Cell *cell);
       
       //! Inject a package into the interface
       /*!
           @param cell main event cell
           @param p packet to inject
           @return void or except in case of failure
       */
       void inject(event::Cell *cell,packet::Packet *p);
       
       //! Add a sniffer to main event.Receive packets from ip targets
       /*!
           @param cell main event cell
           @param name sniffer name(param cb name)
           @param filter filter packet
           @return sniffer cell or except in case of failure
       
       */
       event::Cell *add_sniffer(event::Cell *cell,vmp::str name,vmp::str filter);
       
       //! Start sniffer operation
       /*!
           @param cell sniffer event cell
           @return void or except in case of failure
       */
       void start_sniffer(event::Cell *cell);
       
       //!Return the interface name associated with the ui
       /*!
           @sa ifname
           @return interface name
       */
       vmp::str ifname();

       //!Return the ip associated with the interface(see ifname())
       /*!
           @sa ifname
           @return ip address or except if no ip is set in the interface

       */
       vmp::str ifname_ip();

       //!Return the mac address associated with the interface(see ifname())
       /*!
           @sa ifname
           @return mac address
       */
       vmp::str ifname_mac();
       
       //! Return the arp request source ip associated with the main event
       /*!
           @param cell main event cell
           @return source ip or except in case of failure
       */
       vmp::str ipsrc(event::Cell *cell);

       //! Return the arp request source mac associated with the main event
       /*!
           @param cell main event cell
           @return source mac or except in case of failure
       */
       vmp::str macsrc(event::Cell *cell);
       
       //!Return ipv4 target arp request address list associated with event cell
       /*!
           @param cell main event cell
           @return ipv4 address list
       */
       vmp::vector<vmp::str> targets(event::Cell *cell);
};

}}

#endif

