/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 12/02/2020
 */

#include "process.h"

namespace vampiria { namespace process {

StdW::StdW()
{
   refcount_=0;
   name_="";
}

StdW::~StdW()
{
   refcount_=0;
   close();
}

void StdW::open_impl()
{
}

void StdW::write_impl(vmp::str data)
{
}

void StdW::close_impl()
{
}

vmp::str StdW::name()
{
    return name_;
}

vmp_index StdW::refcount()
{
    return refcount_;
}

void StdW::open(vmp::str name)
{
    name_=name;
    open_impl();
}

void StdW::write(vmp::str data)
{
    if(refcount_ > 0)
        write_impl(data);
}

vmp_index StdW::incref()
{
    refcount_++;
    return refcount_;
}

vmp_index StdW::decref()
{
    if(refcount_ > 0)
        refcount_--;
    return refcount_;
}

void StdW::close()
{
    if(refcount_ == 0)
    {
        close_impl(); 
        name_="";
    }
}

StdWOut::StdWOut():StdW()
{
}
       
StdWOut::~StdWOut()
{
}

void StdWOut::write_impl(vmp::str data)
{
    vmp::output_raw("%s",data.c_str());
}

StdWErr::StdWErr():StdW()
{
}
       
StdWErr::~StdWErr()
{
}

void StdWErr::write_impl(vmp::str data)
{
    vmp::error_raw("%s",data.c_str());
}

StdWFile::StdWFile():StdW()
{
    filepath_="";
}
        
StdWFile::~StdWFile()
{
    filepath_="";
}

void StdWFile::open_impl()
{
    try
    {
        vmp::output_file(filepath_,true,"");
    }
    catch(vmp::exception &x)
    {
        vmp::except("process::StdWFile::open filepath '%s' error '%s'",filepath_.c_str(),x.what());
    }
}
        
void StdWFile::write_impl(vmp::str data)
{
    vmp::output_file(filepath_,false,"%s",data.c_str());
}

StdWBox::StdWBox():StdW()
{
}

StdWBox::~StdWBox()
{
    box_=0;
}

void StdWBox::write_impl(vmp::str data)
{
    box_->write_data(data);
}

StdWFactory::StdWFactory()
{
}

StdWFactory::~StdWFactory()
{
   vmp::table_delete_alldata<vmp::str,StdW *>(&stdwref_);
}

process::StdW *StdWFactory::out()
{
    process::StdWOut *out;
    process::StdW *ret;
    vmp::str name="stdout:";
    if(!stdwref_.search(name,&ret))
    {
        out=new process::StdWOut();
        out->open(name);
        ret =(StdW *) out;
        stdwref_.insert(name,ret);
    }
    ret->incref();
    return ret;    
}

process::StdW *StdWFactory::err()
{
    process::StdWErr *err;
    process::StdW *ret;
    vmp::str name="stderr:";
    if(!stdwref_.search(name,&ret))
    {
        err=new StdWErr();
        err->open(name);
        ret =(StdW *) err;
        stdwref_.insert(name,ret);
    }
    ret->incref();
    return ret;    
}

process::StdW *StdWFactory::file(vmp::str filepath)
{
    StdWFile *fout;
    StdW *ret;
    vmp::str name;
    vmp::unicode::str_write(&name,"file:%s",filepath.c_str());
    if(!stdwref_.search(name,&ret))
    {
        fout=new StdWFile();
        fout->filepath_=filepath;
        fout->open(name);
        ret =(StdW *) fout;
        stdwref_.insert(name,ret);
    }
    ret->incref();
    return ret;    
}

       
process::StdW *StdWFactory::box(process::Box *box)
{
    StdWBox *fbox;
    StdW *ret;
    vmp::str name;
    vmp::except_check_pointer((void *)box,"StdWFactory::Box() null input box");
    vmp::unicode::str_write(&name,"box:%s",box->id().c_str());
    if(!stdwref_.search(name,&ret))
    {
        fbox=new StdWBox();
        fbox->box_=box;
        fbox->open(name);
        ret =(StdW *) fbox;
        stdwref_.insert(name,ret);
    }
    ret->incref();
    return ret;
}  

vmp_index StdWFactory::close(process::StdW *stdw,vmp_bool freeref)
{
    vmp::except_check_pointer((void *)stdw,"StdWFactory::close() null input stdw");
    process::StdW *tmp;
    vmp::str name=stdw->name();
    vmp_index refcount=stdw->decref();
    if((refcount == 0) && freeref)
    {
       stdw->close();
       stdwref_.cancel(name,&tmp);
       delete tmp;
    }
    return refcount;
}

}}

