/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 30/03/2025
 */

#ifndef VAMPIRIA_PROCESS_BOX_H

#define VAMPIRIA_PROCESS_BOX_H 1

namespace vampiria { namespace process {

class Box;

//!Callback performed by a process box at runtime(fork process)
/*!
    @param box box process
*/
typedef void (*BOX_ROUTINE)(process::Box *box);

//!Callback performed in the main process when a box process writes to the std_ (output, error)
/*!
    @param box   box process
    @param data  written data
*/
typedef void (*BOX_STD)(process::Box *box,vmp::str data);

//!Callback performed in the main process upon termination of the box process
/*!
    @param pid process id
    @param box box process
    @param status termination status of the box process
*/
typedef void (*BOX_END)(vmp::Pid pid,process::Box *box,vmp_int status);

//!Callback used to free up process memory. Performed at the end of the monitoring process.
/*!
    @param pid process id
    @param box box process
*/
typedef void (*BOX_FREE)(vmp::Pid pid,process::Box *box);

//!Process Box std_(output,error) comunication type(BOX_PIPE = pipe,BOX_PTS=pseudo terminal)
typedef enum box_com{BOX_PIPE,BOX_PTS} BOX_COM;

//!Manages a group of processes (creation, termination, input, output, error)
class BoxManager
{
    private:
       vmp::Table<vmp::str,process::Box *> box_;/*!<Managed processes (process name, process Box)*/
       vmp_bool rootprocess_;/*!<Detect if there is a process with standard keyboard input (only one allowed)*/
       process::BOX_FREE boxfree_;/*!<see process::BOX_FREE*/
    public:
        //!A constructor
        /*!
            @param boxfree free memory box process callback
        */
        BoxManager(process::BOX_FREE boxfree);
       
        //!A destructor
        ~BoxManager();

        //!Add a process box in box manager, it must be done before monitoring and the process box must be set before being added (see process :: Box.set)
        /*!
            @param box box process
            @param is_root root true true if standard input keyboard
            @param ctype see process::Box_COM
            @param user user to run the process("" Does not change the process UID)
        */
        void add_box(process::Box *box,vmp_bool is_root,process::BOX_COM ctype,vmp::str user="");
    
        //!Runs the routine for managing processes
        void ctl_process();
        
        //!Terminate all processes by closing pipes and pseudo terminals.Performed in process::BoxManager_free
        void close_all();
};

//!Box process
class Box
{
    private:
        friend process::BoxManager;
        
        vmp::str id_; /*!<process identification*/
        vmp_bool setting_;/*!<Has it been configured? (see process::Box.set)*/
        vmp::str rootdir_;/*!<root directory where the process will be executed*/
        process::BOX_ROUTINE routine_;/*!<see process::BOX_ROUTINE*/ 
        process::BOX_STD output_;/*!<callback to redirect the process standard output*/
        process::BOX_STD error_;/*!callback to redirect the process standard error*/
        process::BOX_END end_;/*!<see process::BOX_END*/
        
        vmp::Pid pid_;/*!<process pid at runtime*/
        vmp_int stdinr_;/*!<std input read filedescriptor*/
        vmp_int stdinw_;/*!<std input write filedescriptor*/
        FILE *streamw_;/*!<write stream associated with stdinw_*/
        vmp_int stdoutr_;/*!<std output read filedescriptor*/
        vmp_int stdoutw_;/*!<std output write filedescriptor*/
        vmp_int stderrr_;/*!<std error read filedescriptor*/
        vmp_int stderrw_;/*!<std error write filedescriptor*/

        //!Setting standard input.If is_root keyboard setting
        void set_input(vmp_bool is_root);
        
        //!Setting standard output and error
        /*!
            @param ctype see process::BOX_COM
        */
        void set_output(process::BOX_COM ctype);
        
        //!Run process box
        /*!
            @return process id
        */
        vmp::Pid run();
        
        //!Check if the process box is terminated
        /*!
            @return true if the process box is terminated,otherwise false
        */
        vmp_bool end();

        //!Performs callback routines
        void routine();
        
        //!Check if there has been data written in the standard output and in the standard error. If it finds them, it executes the callbacks output_ (stdout) and error_ (stderr)
        void read_data();

    public:
       //! A constructor
       /*!
            @param id process identification
       */
       Box(vmp::str id);

       //!A destructor       
       ~Box();

       //!Setting box process.Must be done before adding in the box manager
       /*!
           @param rootdir root directory where the process will be executed
           @param routine see process::BOX_ROUTINE
           @param output  see process::BOX_STD (standard output)
           @param error   see process::BOX_STD (standard error)
           @param end     see process::BOX_END
       */
       void set(vmp::str rootdir,process::BOX_ROUTINE routine,process::BOX_STD output,process::BOX_STD error,process::BOX_END end);
       
       //! Process box is setting?
       /*!
           @return true is setting,otherwise false
       */
       vmp_bool is_setting();
       
       //! Process box is running?
       /*!
           @return true is running,otherwise false
       */
       vmp_bool is_running();
       
       //!Close std input read filedescriptor
       void close_input_r();
       
       //!Close std input write filedescriptor and relative stream associated
       void close_input_w();

       //!Close std output and error read filedescriptor
       void close_output_r();
       
       //!Close std output and error write filedescriptor
       void close_output_w();

       //!Send a signal to the process box
       /*!
           @param signal signal to send
       */
       void kill(vmp::str signal);
       
       //!Returns process id from box process
       /*!
           @return pid_
       */
       vmp::Pid pid();

       //!Returns root dir from box process
       /*!
           @return rootdir_
       */
       vmp::str rootdir();
       
       //!Returns id process identification from box process
       /*!
           @return id_
       */
       vmp::str id();

       //!Write data in the standard input of the process box
       /*!
           @param data input data
           @return void(except error)
       */
       void write_data(vmp::str data);
};

//!Create a new BoxManager(manipulate sigint and sigterm signals)
/*!
    @param boxfree free memory box process callback
    @return new BoxManager
*/
process::BoxManager *BoxManager_new(process::BOX_FREE boxfree);

//!Finish and free the memory of BoxManager
/*!
    @param manager BoxManager to free
*/
void BoxManager_free(process::BoxManager *manager);

}}

#endif

