/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 25/03/2025
 */

#include "packet.h"

namespace vampiria { namespace packet { namespace websocket {

vmp::str WebSocketCode_reason(vmp_uint code)
{
    switch(code)
    {
        case packet::websocket::WebSocketCode_Normal:
            return "Connection close";
        case packet::websocket::WebSocketCode_Shutdown:
            return "Server is shutting down ";
        case packet::websocket::WebSocketCode_Protocol:
            return "Some error in the protocol has happened"; 
        case packet::websocket::WebSocketCode_Type:
            return "type was not supported";
        case packet::websocket::WebSocketCode_Utf8 :
            return "The message wasn't in UTF8";
        case packet::websocket::WebSocketCode_Policy:
            return "The policy of the server has been broken";
        case packet::websocket::WebSocketCode_Big:
            return "The messages received is too big";
        case packet::websocket::WebSocketCode_Extension:
            return "Mandatory extension missing";
        case packet::websocket::WebSocketCode_Unexpected:
            return "Unexpected happened";
        default:
            return "Unknown";
    }
}

WebSocket_Framing_P::WebSocket_Framing_P():packet::Packet(packet::websocket::P_WEBSOCKET_FRAMING)
{
    //data_[0]
    packet::DataBits *f1=new packet::DataBits(2);
    
    // fin bit
    packet::BitSelect *fin=new packet::BitSelect("fin",15,1);
    fin->insert_code("unset",0);
    fin->insert_code("set",1);
    f1->insert_data(fin);
    
    packet::BitSelect *rsv1=new packet::BitSelect("rsv1",14,1);
    rsv1->insert_code("unset",0);
    rsv1->insert_code("set",1);
    f1->insert_data(rsv1);
    
    packet::BitSelect *rsv2=new packet::BitSelect("rsv2",13,1);
    rsv2->insert_code("unset",0);
    rsv2->insert_code("set",1);
    f1->insert_data(rsv2);
    
    packet::BitSelect *rsv3=new packet::BitSelect("rsv3",12,1);
    rsv3->insert_code("unset",0);
    rsv3->insert_code("set",1);
    f1->insert_data(rsv3);
    
    //opcode data
    packet::BitSelect *opcode=new packet::BitSelect("opcode",8,4);
    opcode->insert_code("continuation",0x00);
    opcode->insert_code("text",0x01);
    opcode->insert_code("bin",0x02);
    opcode->insert_code("nc3",0x03);
    opcode->insert_code("nc4",0x04);
    opcode->insert_code("nc5",0x05);
    opcode->insert_code("nc6",0x06);
    opcode->insert_code("nc7",0x07);
    opcode->insert_code("close",0x08);
    opcode->insert_code("ping",0x09);
    opcode->insert_code("pong",0x0A);
    opcode->insert_code("cb",0x0B);
    opcode->insert_code("cc",0x0C);
    opcode->insert_code("cd",0x0D);
    opcode->insert_code("ce",0x0E);
    opcode->insert_code("cf",0x0F);
    f1->insert_data(opcode);
    //mask bit
    packet::BitSelect *mask=new packet::BitSelect("mask",7,1);
    mask->insert_code("unset",0);
    mask->insert_code("set",1);
    f1->insert_data(mask);
    
    //payload length
    packet::BitSize *payload_len=new packet::BitSize("payload_len",0,7);
    f1->insert_data(payload_len);
    insert_data(f1);
    
    //data[1]
    packet::DataSize64 *payload_ext=new packet::DataSize64("payload_ext",0);
    insert_data(payload_ext);
    
    //data[2]
    packet::DataSize *masking_key=new packet::DataHexSize("masking_key",0);
    insert_data(masking_key);
    
    //data[3]
    packet::DataBuffer *payload=new packet::DataBuffer("payload",0);
    insert_data(payload);
}


WebSocket_Framing_P::~WebSocket_Framing_P()
{
}

void WebSocket_Framing_P::reset()
{
    ((DataBits *) data_[0])->set_data("fin",0);
    ((DataBits *) data_[0])->set_data("rsv1",0);
    ((DataBits *) data_[0])->set_data("rsv2",0);
    ((DataBits *) data_[0])->set_data("rsv3",0);
    set("opcode","continuation");
    ((DataBits *) data_[0])->set_data("mask",0);
    ((DataBits *) data_[0])->set_data("payload_len",0);
    packet::DataSize64 *payload_ext=(packet::DataSize64 *)data_[1];
    payload_ext->set_new_datalen(0);
    packet::DataSize *masking_key=(packet::DataSize *) data_[2];
    masking_key->set_new_datalen(0);
    packet::DataBuffer *payload=(packet::DataBuffer *)data_[3];
    payload->set_new_size(0);
}

void WebSocket_Framing_P::read(vmp::Buf *buf)
{
    vmp_int ret=ret_read(buf,vmp::UINTMAX-1);
    if(ret == -1)
        vmp::except_s("packet::websocket::WebSocket_Frame_P::read() malfomated packet");
    else if(ret == -2)
        vmp::except_s("packet::websocket::WebSocket_Frame_P::read() insufficient reading data");
    else if(ret == -3)
        vmp::except_s("packet::websocket::WebSocket_Frame_P::read() packet too long");
}

void WebSocket_Framing_P::set_payload_size()
{
    packet::DataSize64 *payload_ext=(packet::DataSize64 *)data_[1];
    vmp::Buf *buf=((packet::DataBuffer *) data_[3])->get_data();
    vmp_size s=buf->size();
    vmp::str sstr;
    vmp::unicode::str_write(&sstr,"%u",s);
    if(s <=125)
        set("payload_len",sstr);
    else if (s >= 126 && s < vmp::UINT16MAX)
    {
        set("payload_len","126");
        payload_ext->set_new_datalen(2);
        payload_ext->set_data((vmp_size64)s);
    }
    else
    {
        set("payload_len","127");
        payload_ext->set_new_datalen(8);
        payload_ext->set_data((vmp_size64)s);
    }
}

void WebSocket_Framing_P::set(vmp::str field,vmp::str data)
{
    vmp_index index;
    if(search_.search(field,&index))
    {
        try
        {
            if(field == "payload")
            {
                data_[index]->set(field,data);
                set_payload_size();
            }
            else if(field == "payload_ext")
            {
                vmp_size64 s=vmp::unicode::str_todigit64(data);
                packet::DataSize64 *payload_ext=(packet::DataSize64 *)data_[1];
                if(s < vmp::UINT16MAX)
                {
                    payload_ext->set_new_datalen(2);
                    payload_ext->set_data((vmp_size64)s);
                }
                else
                {
                    payload_ext->set_new_datalen(8);
                    payload_ext->set_data((vmp_size64)s);
                }
            }
            else if(field == "masking_key")
            {
                packet::DataSize *masking_key=(packet::DataSize *)data_[2];
                if(data == "")
                    masking_key->set_new_datalen(0);
                else
                {
                    vmp_size s=vmp::unicode::str_todigit(data);
                    masking_key->set_data(s);
                }   
            }
            else
                 data_[index]->set(field,data);
        }
        catch(vmp::exception &x)
        {
            vmp::except("packet::Packet::set() in packet '%s' check failed to insert data '%s' for field '%s'[%s]",type_.c_str(),data.c_str(),field.c_str(),x.what());
        }
    }
}

void WebSocket_Framing_P::set_payload_raw(vmp_byte *raw,vmp_size size)
{
     packet::DataBuffer *payload=(packet::DataBuffer *)data_[3];
     vmp::Buf *buf=payload->get_data();
     buf->write_array(raw,size);
     set_payload_size();
}

vmp_int WebSocket_Framing_P::ret_read(vmp::Buf *buf,vmp_size maxsize)
{
    vmp_index index=buf->get_index();
    vmp_index ret=2;
    reset();
    vmp_size mask,plen,len;
    try
    {
        data_[0]->read(buf);
        mask=((DataBits *) data_[0])->get_data("mask");
        plen=((DataBits *) data_[0])->get_data("payload_len");
        packet::DataSize64 *payload_ext=(packet::DataSize64 *)data_[1];
        if(plen == 126)
        {
            payload_ext->set_new_datalen(2);
            data_[1]->read(buf);
            ret += 2;
            maxsize -= 2;
        }
        else if(plen == 127)
        {
            payload_ext->set_new_datalen(8);
            data_[1]->read(buf);
            ret += 8;
            maxsize -= 8;
        }
        if(mask == 1)
        {
            packet::DataSize *masking_key=(packet::DataSize *) data_[2];
            masking_key->set_new_datalen(4);
            data_[2]->read(buf);
            ret += 4;
            maxsize -= 1;
        }
        if((plen == 126) || (plen == 127))
        {
            vmp_size64 s=payload_ext->get_data();
            if(s > vmp::UINT64MAX-1)
            {
                buf->index(index);
                return -3;
            }
            len=(vmp_size)s;
        }
        else
            len=plen;
        ret += len;
        if(len > maxsize-2)
        {
            buf->index(index);
            return -3;
        }
        data_[3]->read(buf);        
    }
    catch(vmp::exception &x)
    {
        buf->index(index);
        return -2;
    }
    return ret;
}

vmp_uint64 WebSocket_Framing_P::payload_len()
{
    vmp_uint64 ret;
    vmp_uint s=vmp::unicode::str_todigit(get("payload_len"));
    if((s == 126) || (s == 127))
        ret=((packet::DataSize64 *)data_[1])->get_data();
    else
        ret=(vmp_uint64)s;
    return ret;
}

void WebSocket_Framing_P::get_payload(vmp::Buf *buf)
{
    if(buf != 0)
    {
        buf->index();
        packet::DataBuffer *b=(packet::DataBuffer *)data_[3];
        buf->write_buf(b->get_data());
        buf->index();
    }
}

void WebSocket_Framing_P::add_payload(vmp::Buf *buf)
{
    if(buf != 0)
    {
        packet::DataBuffer *b=(packet::DataBuffer *)data_[3];
        buf->write_buf(b->get_data());
    } 
}

}}}

