/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 25/03/2025
 */

#include "packet.h"

namespace vampiria { namespace packet { namespace websocket {

FramingHelper::FramingHelper()
{
    manager_=0;
    reset();
}

FramingHelper::~FramingHelper()
{
    reset();
}

vmp_uint FramingHelper::parse_error(vmp_uint code,vmp_uint *framecode,vmp::str *frameerr,packet::websocket::WebSocket_Framing_P *p)
{
    (*framecode)=code;
    (*frameerr)=packet::websocket::WebSocketCode_reason((*framecode));
    if(p != 0)
        free_packet(p);
    return 0x100;
}

void FramingHelper::unmask(packet::websocket::WebSocket_Framing_P *p)
{
    if(p != 0)
    {
        if(p->get("mask") == "set")
        {
            vmp::Buf buf,payload,result;
            buf.write_xstr(p->get("masking_key"));
            p->get_payload(&payload);
            for(vmp_index i=0;i<p->payload_len();i++)
            {
                buf.index(i%4);
                result.write_byte(payload.read_byte() ^ buf.read_byte());  
            }
            result.index();
            p->set("mask","unset");
            p->set("masking_key","");
            p->set("payload",result.read_xstr(result.size()));
        }
    }
}

vmp_bool FramingHelper::check_init()
{
    return (manager_ != 0);
}

void FramingHelper::init(packet::websocket::FramingExtManager *manager)
{
    if(check_init())
        vmp::except_s("packet::websocket::FrameHelper::init() already initialized");
    manager_=manager;
}
       
void FramingHelper::reset()
{
    ptype_="";
    if(check_init())
    {
        for(vmp_index i=0;i<exts_.size();i++)
        {
            exts_[i]->ext_->close(exts_[i]);
            exts_[i]->reset();   
        }
        manager_=0;
        while(!squeue_.empty())
        {
            free_buf(squeue_.front());
            squeue_.pop();
        }
        for(vmp_index i=0;i<frames_.size();i++)
            free_packet(frames_[i]);
        frames_.clear();
    } 
}
        
vmp_bool FramingHelper::add_ext(vmp::vector<vmp::str> args,vmp_bool server)
{
    if(!check_init())
        vmp::except_s("packet::websocket::FrameHelper::add_ext() uninitialized");
    vmp::str name;
    vmp_size s;
    vmp_index i;
    if(args.size() > 0)
    {
         name=vmp::unicode::str_trim(args[0]);
         s=exts_.size();
         for(i=0;i<s;i++)
             if(name == exts_[i]->ext_->name_)
                 break;
         if(i == s)
         {
             packet::websocket::FramingExt *ext=manager_->search(name);
             if(ext != 0)
             {
                 packet::websocket::FramingExtData *extdata=storage_.get();
                 if(server)
                 {
                     if(ext->open_server(args,extdata))
                     {      
                         extdata->ext_=ext;
                         exts_.push_back(extdata);
                     }
                     else
                     {
                         extdata->reset();
                         storage_.free(extdata); 
                     }
                 }
                 else
                 {
                     if(ext->open_client(args,extdata))
                     {      
                         extdata->ext_=ext;
                         exts_.push_back(extdata);
                     }
                     else
                     {
                         extdata->reset();
                         storage_.free(extdata);
                         return false;
                     }
                 }
             }   
         }
    }
    return true;
}

vmp_bool FramingHelper::add_exts(vmp::str accepted,vmp_bool server)
{
    vmp::unicode::Strtok strtok(accepted); 	
    vmp::str stmp=vmp::unicode::str_trim(strtok.next(","));
    while(stmp != "")
    {
        vmp::vector<vmp::str> split=vmp::unicode::str_split(stmp,";");
        if(!add_ext(split,server))
           return false; 
        stmp=vmp::unicode::str_trim(strtok.next(","));
    }
    return true;
}

vmp::str FramingHelper::exts_info()
{
    vmp::str ret="";
    for(vmp_index i=0;i<exts_.size();i++)
    {
        if(i == 0)
            vmp::unicode::str_write(&ret,"%s",exts_[i]->accepted_.c_str());
        else
            vmp::unicode::str_cwrite(&ret,", %s",exts_[i]->accepted_.c_str());
    }
    return ret;
}

vmp_uint FramingHelper::parse(vmp::Buf *ibuf,vmp_uint *framecode,vmp::str *frameerr,vmp::Buf *out,vmp_size maxframesize,vmp_size maxframes)
{
    vmp_uint s,size,retcode=0x00;
    vmp_int ret;
    vmp::Buf tmp;
    vmp::Buf *tbuf;
    vmp::str stmp;
    recv_.index(recv_.size());
    while(!tqueue_.empty())
    {
        tbuf=tqueue_.front();
        s=recv_.write_buf(tbuf);
        size=tbuf->size();
        if(s == 0)
            break;
        else if(s == size)
        {
            tqueue_.pop();
            free_buf(tbuf);
        }
        else
        {
            tbuf->cut(s,size-s);
            break;
        }
    }
    if(ibuf != 0)
    {
        if(!tqueue_.empty())
        {
            tbuf=bufs_.get();
            tbuf->write_buf(ibuf);
            tqueue_.push(tbuf);
        }
        else
        {
            s=recv_.write_buf(ibuf);
            size=ibuf->size();
            if(s < size)
            {
                tbuf=bufs_.get();
                tbuf->write_buf(ibuf);
                tbuf->cut(s,size-s);
                tqueue_.push(tbuf);
            }
        }
    }
    recv_.index();
    packet::websocket::WebSocket_Framing_P *p=packets_.get();
    ret=p->ret_read(&recv_,maxframesize);
    switch(ret)
    {
        case -1:
            return parse_error(packet::websocket::WebSocketCode_Protocol,framecode,frameerr,p);
        case -2:
            free_packet(p);
            return 0x00;
        case -3:
            return parse_error(packet::websocket::WebSocketCode_Big,framecode,frameerr,p);
        default:
            recv_.cut(recv_.get_index(),recv_.size_reading());
            break;
    }
    unmask(p);//if masking is setting
    vmp::str fin=p->get("fin");
    vmp::str rsv1=p->get("rsv1");
    vmp::str rsv2=p->get("rsv2");
    vmp::str rsv3=p->get("rsv3");
    vmp::str opcode=p->get("opcode");
    vmp_uint64 plen=p->payload_len();
    if((exts_.size() == 0) && ((rsv1 == "set") || (rsv2 == "set") || (rsv3 == "set")))
        return parse_error(packet::websocket::WebSocketCode_Protocol,framecode,frameerr,p);
    if(opcode == "close")
    {
        if(fin != "set" || plen >=126)
            return parse_error(packet::websocket::WebSocketCode_Protocol,framecode,frameerr,p);
        if(plen < 2)
            return parse_error(packet::websocket::WebSocketCode_Protocol,framecode,frameerr,p);
        p->get_payload(&tmp);
        (*framecode)=tmp.read_size(2);
        stmp=packet::websocket::WebSocketCode_reason(*framecode);
        if(stmp == "Unknown" && ((*framecode) < 3000 || (*framecode) > 4999))
            return parse_error(packet::websocket::WebSocketCode_Protocol,framecode,frameerr,p);
        if(plen == 2)
            (*frameerr)=stmp;
        else
        {
            (*frameerr)=tmp.read_str(tmp.size_reading());
            if(!vmp::unicode::utf8_check(*frameerr))
                return parse_error(packet::websocket::WebSocketCode_Utf8,framecode,frameerr,p);
        }
        free_packet(p);
        return 0x08;
    }
    else if(opcode == "ping")
    {
        if(fin != "set" || plen >=126)
            return parse_error(packet::websocket::WebSocketCode_Protocol,framecode,frameerr,p);
        p->set("opcode","pong");
        tbuf=bufs_.get();
        p->write(tbuf);
        squeue_.push(tbuf);
        free_packet(p);
        return 0x09;
    }
    else if(opcode == "pong")
    {
        if(fin != "set" || plen >=126)
            return parse_error(packet::websocket::WebSocketCode_Protocol,framecode,frameerr,p);
        p->get_payload(&tmp);
        if(pingdata_ != tmp)
            return parse_error(packet::websocket::WebSocketCode_Protocol,framecode,frameerr,p);
        pingdata_.reset();
        free_packet(p);
        return 0x0A;
    }
    else
    {
        if(fin == "unset" && frames_.size() == maxframes)
            return parse_error(packet::websocket::WebSocketCode_Big,framecode,frameerr,p);
        if(opcode == "continuation")
        {
            if(ptype_ == "" || (rsv1 == "set") || (rsv2 == "set") || (rsv3 == "set"))
                return parse_error(packet::websocket::WebSocketCode_Protocol,framecode,frameerr,p);    
        }
        else 
        {
            if(ptype_ != "")
                return parse_error(packet::websocket::WebSocketCode_Protocol,framecode,frameerr,p);  
            ptype_=opcode;
        }
        frames_.push_back(p);
    }
    if(fin == "set")
    {
        for(vmp_index i=exts_.size();i > 0; i--)
        {
            vmp_index k=i-1;
            for(vmp_index j=0; j < frames_.size(); j++)
                exts_[k]->ext_->inframe(frames_[j],exts_[k]);
            exts_[k]->ext_->inframes(frames_,exts_[k]);
        }
        
        rsv1=frames_[0]->get("rsv1");
        rsv2=frames_[0]->get("rsv2");
        rsv3=frames_[0]->get("rsv3");
        if((rsv1 == "set") || (rsv2 == "set") || (rsv3 == "set"))
            return parse_error(packet::websocket::WebSocketCode_Protocol,framecode,frameerr,p);
        
        if(out != 0)
        {
            out->reset();
            for(vmp_index i=0;i<frames_.size();i++)
                frames_[i]->add_payload(out);
            out->index();
        }
        ptype_=frames_[0]->get("opcode");
        for(vmp_index i=0;i<frames_.size();i++)
            free_packet(frames_[i]);
        frames_.clear();
        if(ptype_ == "text")
        {
            retcode=0x01;
            if(out != 0)
            {
                stmp=out->read_str(out->size());
                out->index();
                if(!vmp::unicode::utf8_check(stmp))
                    return parse_error(packet::websocket::WebSocketCode_Utf8,framecode,frameerr,0);
            }
        }
        else if(ptype_ == "bin")
            retcode=0x02;
        else
            return parse_error(packet::websocket::WebSocketCode_Type,framecode,frameerr,0); 
        ptype_="";
    }
    return retcode;
}

void FramingHelper::send_packet(vmp::Buf *buf,vmp::str opcode,vmp_size maxframesize,vmp_size maxframes)
{
    vmp::except_check_pointer((void *) buf,"packet::websocket::send_text(buf=Null)");
    if(buf->size() > (maxframesize*maxframes))
        vmp::except_s("packet::websocket::send_text() buf size too big");
    buf->index();
    vmp::vector<packet::websocket::WebSocket_Framing_P *> send;
    packet::websocket::WebSocket_Framing_P *p;
    if(buf->size() != 0)
    {
        vmp_index i=0;
        while(!buf->eob())
        {
            p=packets_.get();
            if(i == 0)
                p->set("opcode",opcode);
            else
                p->set("opcode","continuation");
            p->set("rsv1","unset");
            p->set("rsv2","unset");
            p->set("rsv3","unset");
            p->set("mask","unset");
            if(buf->size_reading() > maxframesize)
            {
                p->set_payload_raw(buf->pointer(i),maxframesize);
                i+=maxframesize;
                buf->index(i);
                p->set("fin","unset");
            }
            else
            {
                p->set_payload_raw(buf->pointer(i),buf->size_reading());
                //non escce bug
                i+=buf->size_reading();
                buf->index(i);
                p->set("fin","set");
            }
            send.push_back(p);
            i++;
        }
        for(vmp_index i=0;i<exts_.size();i++)
        {
            exts_[i]->ext_->outframes(send,exts_[i]);
            for(vmp_index j=0;j<send.size();j++)
                exts_[i]->ext_->outframe(send[j],exts_[i]);
        }
        vmp::Buf *out;
        for(vmp_index i=0;i<send.size();i++)
        {
            out=bufs_.get();
            p->write(out);
            squeue_.push(out);
            free_packet(p);
        }
        send.clear();
    }
}

void FramingHelper::send_ping(vmp_size bodylen)
{
    packet::websocket::WebSocket_Framing_P *p=packets_.get();
    vmp::Buf *buf=bufs_.get();
    p->set("fin","set");
    p->set("opcode","ping");
    p->set("mask","unset");
    pingdata_.random_bytes(bodylen);
    p->set_payload_raw(pingdata_.pointer(),pingdata_.size());
    pingdata_.index();
    p->write(buf);
    packet_free(p);
    squeue_.push(buf);
}

vmp::Buf *FramingHelper::next_send()
{
    vmp::Buf *ret=0;
    if(!squeue_.empty())
    {
        ret=squeue_.front();
        squeue_.pop();
    }
    return ret;
}


vmp::Buf *FramingHelper::frame_close(vmp_uint code,vmp::str reason)
{
    packet::websocket::WebSocket_Framing_P *p=packets_.get();
    vmp::Buf *buf=bufs_.get();
    p->set("fin","set");
    p->set("opcode","close");
    buf->write_size(code,2);
    buf->write_str(reason);
    buf->index();
    p->set("payload",buf->read_xstr(buf->size()));
    buf->reset();
    p->write(buf);
    packet_free(p);
    return buf;
}        

void FramingHelper::free_packet(packet::websocket::WebSocket_Framing_P *p)
{
    if(p != 0)
    {
        p->reset();
        packets_.free(p);
    }
}

void FramingHelper::free_buf(vmp::Buf *buf)
{
    if(buf != 0)
    {
        buf->reset();
        bufs_.free(buf);
    }
}

}}}

