/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 03/09/2024
 */

#include "packet.h"

namespace vampiria { namespace packet { namespace http {

vmp::str HttpCode_reason(vmp_uint code)
{
    switch (code)
    {
        case 100: return "Continue";
        case 101: return "Switching Protocols";
        case 102: return "Processing";
        case 103: return "Early Hints";
        case 200: return "OK";
        case 201: return "Created";
        case 202: return "Accepted";
        case 203: return "Non-Authoritative Information";
        case 204: return "No Content";
        case 205: return "Reset Content";
        case 206: return "Partial Content";
        case 207: return "Multi-Status";
        case 208: return "Already Reported";
        case 226: return "IM Used";
        case 300: return "Multiple Choices";
        case 301: return "Moved Permanently";
        case 302: return "Found";
        case 303: return "See Other";
        case 304: return "Not Modified";
        case 305: return "Use Proxy";
        case 307: return "Temporary Redirect";
        case 308: return "Permanent Redirect";
        case 400: return "Bad Request";
        case 401: return "Unauthorized";
        case 402: return "Payment Required";
        case 403: return "Forbidden";
        case 404: return "Not Found";
        case 405: return "Method Not Allowed";
        case 406: return "Not Acceptable";
        case 407: return "Proxy Authentication Required";
        case 408: return "Request Timeout";
        case 409: return "Conflict";
        case 410: return "Gone";
        case 411: return "Length Required";
        case 412: return "Precondition Failed";
        case 413: return "Payload Too Large";
        case 414: return "URI Too Long";
        case 415: return "Unsupported Media Type";
        case 416: return "Range Not Satisfiable";
        case 417: return "Expectation Failed";
        case 418: return "I'm a teapot";
        case 422: return "Unprocessable Entity";
        case 423: return "Locked";
        case 424: return "Failed Dependency";
        case 426: return "Upgrade Required";
        case 428: return "Precondition Required";
        case 429: return "Too Many Requests";
        case 431: return "Request Header Fields Too Large";
        case 451: return "Unavailable For Legal Reasons";
        case 500: return "Internal Server Error";
        case 501: return "Not Implemented";
        case 502: return "Bad Gateway";
        case 503: return "Service Unavailable";
        case 504: return "Gateway Time-out";
        case 505: return "HTTP Version Not Supported";
        case 506: return "Variant Also Negotiates";
        case 507: return "Insufficient Storage";
        case 508: return "Loop Detected";
        case 510: return "Not Extended";
        case 511: return "Network Authentication Required";
        default:break;
    }
    return "";
}

vmp_bool HttpCode_isInformational(vmp_uint code) 
{ 
    return (code >= 100 && code < 200);
}

vmp_bool HttpCode_isSuccessful(vmp_uint code)    
{ 
    return (code >= 200 && code < 300);
}

vmp_bool HttpCode_isRedirection(vmp_uint code)   
{ 
    return (code >= 300 && code < 400);
}

vmp_bool HttpCode_isClientError(vmp_uint code)   
{ 
    return (code >= 400 && code < 500); 
}

vmp_bool HttpCode_isServerError(vmp_uint code)   
{ 
    return (code >= 500 && code < 600); 
}

vmp_bool HttpCode_isError(vmp_uint code)         
{ 
    return (code >= 400);
} 

vmp::str uri_encode(vmp::str url)
{
    vmp::vector<vmp_byte> specials={';','(',')','/','?',':','#','*','@','=','[',']','&',' ',',','"','<','>','#','%','|','{','}','\\','\'','^','~','!','`','-','+','$'};
    vmp::str ret="",hex;
    for(vmp_index i=0;i<url.size();i++)
    {
        if(vmp::invector<vmp_byte>(url[i],specials))
        {    
            vmp::unicode::str_write(&hex,"%c",url[i]);
            vmp::unicode::str_cwrite(&ret,"%%%s",vmp::unicode::str_toxstr(hex,"").c_str());
        }
        else
            vmp::unicode::str_cwrite(&ret,"%c",url[i]);
    }
    return ret;
}

vmp::str uri_decode(vmp::str url)
{
   vmp::str ret="",hex;
   for(vmp_index i=0;i<url.size();i++)
   {
       if(url[i] == '%')
       {
           vmp::unicode::str_write(&hex,"%c%c",url[i+1],url[i+2]);
           vmp::unicode::str_cwrite(&ret,"%s",vmp::unicode::xstr_tostr(hex,"").c_str());
           i=i+2;
       }
       else
           vmp::unicode::str_cwrite(&ret,"%c",url[i]);
   }
   return ret;
}

UriHelper::UriHelper()
{
    regex_scheme="[a-zA-Z][a-zA-Z0-9+.-]+";
    regex_host="([-a-zA-Z0-9._~!$&\"*+,;=:]|(%[a-zA-Z0-9]{2}))+";
    vmp::str segment="([-a-zA-Z0-9._~!$&\"*+,;=:@]|(%[a-zA-Z0-9]{2}))+";
    vmp::unicode::str_write(&regex_path,"\\/(%s(\\/%s)*)?",segment.c_str(),segment.c_str());
    regex_userinfo="([-a-zA-Z0-9._~!$&\"*+,;=]|(%[a-zA-Z0-9]{2}))+";
    regex_params="([-a-zA-Z0-9._~!$\"*+,;]|(%[a-zA-Z0-9]{2}))+";
    regex_urlipv6="\\[[0-9a-fA-F:]*\\](\\:[0-9]{1,5})?";
    reset();    
}

UriHelper::UriHelper(vmp::str uri)
{
    read(uri);
}
        
UriHelper::~UriHelper()
{
    reset();
}

void UriHelper::reset()
{
    type_="*";//*,abs,path,authority
    scheme_="";
    user_="";
    password_="";
    host_="";
    port_=0;
    path_="";
    query_.clear();
    fragment_.clear();
}

void UriHelper::read(vmp::str uri)
{
    reset();
    vmp::vector<vmp::str> split=vmp::unicode::str_split(uri,"://");
    vmp::str authority,trailer,tmp;
    if(uri == "")
        vmp::except("packet::http::UriHelper::read(uri='')");
    if(split.size() == 2)
    {
        type_="abs";
        scheme_=split[0];
        if(!vmp::unicode::str_regex_matching(scheme_,regex_scheme))
            vmp::except("packet::http::UriHelper::read(uri='%s') bad value",uri.c_str());
        split=vmp::unicode::str_split(split[1],"/");
        if(split.size() == 1)
        {
            authority=split[0];
            trailer="";
        }
        else
        {
            authority=split[0];
            trailer="";
            for(vmp_index i=1;i<split.size();i++)
                vmp::unicode::str_cwrite(&trailer,"/%s",split[i].c_str());
        }
    }
    else if(split[0][0] == '/')
    {
        type_="path";
        authority="";
        trailer=split[0];
    }
    else
    {
        type_="authority";
        authority=split[0];
        trailer="";
    }
    if(authority != "")
    {
        split=vmp::unicode::str_split(authority,"@");
        if(split.size() == 2)
        {
            tmp=split[1];
            split=vmp::unicode::str_split(split[0],":");
            user_=split[0];
            if(split.size() == 2)
            {    
                 password_=split[1];
                 if(!vmp::unicode::str_regex_matching(password_,regex_userinfo))
                     vmp::except("packet::http::UriHelper::read(uri='%s') bad value",uri.c_str());
            }
            if(!vmp::unicode::str_regex_matching(user_,regex_userinfo))
                 vmp::except("packet::http::UriHelper::read(uri='%s') bad value",uri.c_str());
            
        }
        else
            tmp=split[0];
        vmp::str port="";
        if(vmp::unicode::str_regex_matching(tmp,regex_urlipv6))
        {
            vmp::unicode::Strtok strtok(tmp);
            host_=strtok.next("]")+']';
            strtok.next(":");
            port=strtok.next(" ");
        }
        else
        {
            vmp::unicode::Strtok strtok(tmp);
            host_=strtok.next(":");
            port=strtok.next(" ");
            if(!vmp::unicode::str_regex_matching(host_,regex_host))
                vmp::except("packet::http::UriHelper::read(uri='%s') bad value",uri.c_str());
        }
        try
        {
            if(port != "")
                port_=vmp::unicode::str_todigit_range(port,1,65535);
            else
                port_=80;
        }
        catch(vmp::exception &x)
        {
            vmp::except("packet::http::UriHelper::read(uri='%s') bad value",uri.c_str());
        }
        
    }
    if(trailer != "")
    {
        vmp::str query,fragment;
        split=vmp::unicode::str_split(trailer,"#");
        if(split.size() == 2)
            fragment=split[1];
        else
            fragment="";
        split=vmp::unicode::str_split(split[0],"?");
        if(split.size() == 2)
            query=split[1];
        else
            query="";
        path_=split[0];
        if(!vmp::unicode::str_regex_matching(path_,regex_path))
             vmp::except("packet::http::UriHelper::read(uri='%s') bad value",uri.c_str());
        vmp::vector<vmp::str> values;
        if(query != "")
        {
            split=vmp::unicode::str_split(query,"&");
            for(vmp_index i=0;i<split.size();i++)
            {
                values=vmp::unicode::str_split(split[i],"=");
                if(!vmp::unicode::str_regex_matching(values[0],regex_params))
                     vmp::except("packet::http::UriHelper::read(uri='%s') bad value",uri.c_str());
                if(values.size() == 2)
                {
                    if(!vmp::unicode::str_regex_matching(values[1],regex_params))
                        vmp::except("packet::http::UriHelper::read(uri='%s') bad value",uri.c_str());
                    tmp=values[1];
                }
                else
                    tmp="";
                query_.push(values[0],tmp);
            }
        }
        if(fragment != "")
        {
            split=vmp::unicode::str_split(fragment,"&");
            for(vmp_index i=0;i<split.size();i++)
            {
                values=vmp::unicode::str_split(split[i],"=");
                if(!vmp::unicode::str_regex_matching(values[0],regex_params))
                    vmp::except("packet::http::UriHelper::read(uri='%s') bad value",uri.c_str());
                if(values.size() == 2)
                {
                    if(!vmp::unicode::str_regex_matching(values[1],regex_params))
                         vmp::except("packet::http::UriHelper::read(uri='%s') bad value",uri.c_str());
                    tmp=values[1];
                }
                else
                    tmp="";
                fragment_.push(values[0],tmp);
            }
        }
    }
}

//! Setting in encoded
void UriHelper::init_abs(vmp::str scheme,vmp::str host,vmp_uint port,vmp::str path)
{
    reset();
    type_="abs";
    scheme_=scheme;
    host_=host;
    port_=port;
    path_=path;
        
} 

void UriHelper::init_authority(vmp::str host,vmp_uint port)
{
    reset();
    type_="autority";
    host_=host;
    port_=port;
}
   
void UriHelper::init_path(vmp::str path)
{
    reset();
    path_=path;
    type_="path";
}     

void UriHelper::set_userinfo(vmp::str user,vmp::str password)
{
    if((type_ == "authority") || (type_ == "abs"))
    {
        user_=user;
        password_=password;
    }
    else
        vmp::except("packet::http::UriHelper::set_userinfo() not associated with type=%s",type_.c_str()); 
}
        
vmp::str UriHelper::type()
{
    return type_;
}

vmp::str UriHelper::scheme()
{
    return scheme_;
}

vmp::str UriHelper::user()
{
    return user_;
}
        
vmp::str UriHelper::password()
{
    return password_;
}

vmp::str UriHelper::host()
{
    return host_;
}

vmp_index UriHelper::port()
{
    return port_;
}

vmp::str UriHelper::path()
{
    return path_;
}

vmp::PairList<vmp::str,vmp::str> *UriHelper::query()
{
    return &query_;
}
        
vmp::PairList<vmp::str,vmp::str> *UriHelper::fragment()
{
    return &fragment_;
}

void UriHelper::print_authority(vmp::str *ret)
{
    if(user_ != "")
    {
        if(!vmp::unicode::str_regex_matching(user_,regex_userinfo))
            vmp::except("packet::http::UriHelper::print()[user_='%s'] bad value",user_.c_str());
        vmp::unicode::str_cwrite(ret,"%s",user_.c_str());
        if(password_ != "")
        {
            if(!vmp::unicode::str_regex_matching(password_,regex_userinfo))
                vmp::except("packet::http::UriHelper::print()[password='%s'] bad value",password_.c_str());    
            vmp::unicode::str_cwrite(ret,":%s",password_.c_str());
        }
        vmp::unicode::str_cwrite(ret,"@");
    }
    if(!vmp::unicode::str_regex_matching(host_,regex_host))
        vmp::except("packet::http::UriHelper::print()[host='%s'] bad value",host_.c_str());
    vmp::unicode::str_cwrite(ret,"%s",host_.c_str());
    if(port_ > 65535)
        vmp::except("packet::http::UriHelper::print(port='%u') bad value",port_);
    if(port_ != 0)
        vmp::unicode::str_cwrite(ret,":%u",port_);
}

void UriHelper::print_path(vmp::str *ret)
{
    if(!vmp::unicode::str_regex_matching(path_,regex_path))
        vmp::except("packet::http:UriHelper::print()[path='%s'] bad value",path_.c_str());
    vmp::unicode::str_cwrite(ret,"%s",path_.c_str());
}

void UriHelper::print_params(vmp::str *ret)
{
    vmp::vector<vmp::str> keys=query_.all_keys();
    vmp::str name,value;
    vmp_size s=keys.size();
    if(s != 0)
    { 
        vmp::unicode::str_cwrite(ret,"?");
        for(vmp_index i=0;i<s;i++)
        {
            name=keys[i];
            value=query_.get(i).second;
            if(!vmp::unicode::str_regex_matching(name,regex_params))
                vmp::except("packet::http::UriHelper::print() query [name='%s'] bad value",name.c_str());
            if((value != "") && !vmp::unicode::str_regex_matching(value,regex_params))
                vmp::except("packet::http::UriHelper::print() query [value='%s'] in [name='%s'] bad value",value.c_str(),name.c_str());
            if(i == s-1)
                vmp::unicode::str_cwrite(ret,"%s=%s",name.c_str(),value.c_str());
            else
                vmp::unicode::str_cwrite(ret,"%s=%s&",name.c_str(),value.c_str());
        }
    }
    keys=fragment_.all_keys();
    s=keys.size();
    if(s != 0)
    { 
        vmp::unicode::str_cwrite(ret,"#");
        for(vmp_index i=0;i<s;i++)
        {
            name=keys[i];
            value=fragment_.get(i).second;
            if(!vmp::unicode::str_regex_matching(name,regex_params))
                vmp::except("packet::http::UriHelper::print() fragment [name='%s'] bad value",name.c_str());
            if((value != "") && !vmp::unicode::str_regex_matching(value,regex_params))
                vmp::except("packet::http::UriHelper::print() fragment [value='%s'] in [name='%s'] bad value",value.c_str(),name.c_str());
            if(i == s-1)
                vmp::unicode::str_cwrite(ret,"%s=%s",name.c_str(),value.c_str());
            else
                vmp::unicode::str_cwrite(ret,"%s=%s&",name.c_str(),value.c_str());
        }
    }
}

vmp::str UriHelper::print()
{
    vmp::str ret;
    if(type_ == "abs")
    {
        if(!vmp::unicode::str_regex_matching(scheme_,regex_scheme))
            vmp::except("packet::http::UriHelper::print()[scheme_='%s'] bad value",scheme_.c_str());
        vmp::unicode::str_write(&ret,"%s://",scheme_.c_str());
        print_authority(&ret);
        if(path_ != "/")
            print_path(&ret);
        print_params(&ret);
    }
    else if(type_ == "autority")
        print_authority(&ret);
    else if(type_ == "path")
    {
        print_path(&ret);
        print_params(&ret);
    }
    else
        ret="*";
    return ret;
}

void UriHelper::set_requesturi(packet::Packet *packet)
{ 
    vmp::except_check_pointer((void *) packet,"packet::http::UriHelper::set_requesturi(packet=null)");
    ((*packet)[packet::http::P_HTTP_REQUEST])->set("uri",print());
}

HeadersHelper::HeadersHelper()
{
    regex_name="[-!#$#&,*+.0-9a-zA-Z^`_{}|~]+";
    regex_value="[-0-9a-zA-Z./!\"#$&,*+^`=_{}|~()<>@,;:\\s\\t\\[\\]\\\\]+";
    reset();
}

HeadersHelper::HeadersHelper(vmp::str headers)
{
    read(headers);
}

HeadersHelper::~HeadersHelper()
{
    reset();
}

vmp::PairList<vmp::str,vmp::str> *HeadersHelper::headers()
{
    return &headers_;
}

void HeadersHelper::reset()
{
    headers_.clear();
}
        
void HeadersHelper::read(vmp::str headers)
{
    reset();
    vmp::vector<vmp::str> split=vmp::unicode::str_split(headers,"\r\n"),tsplit;
    for(vmp_index i=0;i<split.size();i++)
    {
        tsplit=vmp::unicode::str_split(split[i],":");
        tsplit[1]=vmp::unicode::str_trim(tsplit[1]);
        if((tsplit.size() < 2) || (!vmp::unicode::str_regex_matching(tsplit[0],regex_name)))
            vmp::except("packet::http::HeadersHelper::read(headers='%s') bad value",headers.c_str());
        vmp::str value=tsplit[1];
        for(vmp_index i=2;i<tsplit.size();i++)
            vmp::unicode::str_cwrite(&value,":%s",tsplit[i].c_str());
        vmp::unicode::str_regex_matching(value,regex_value);
        headers_.push(tsplit[0],value);
    }
}

vmp::str HeadersHelper::print()
{
    vmp::str ret="";
    vmp::pair<vmp::str,vmp::str> tmp;
    for(vmp_index i=0;i<headers_.size();i++)
    {
        tmp=headers_.get(i);
        if(!vmp::unicode::str_regex_matching(tmp.first,regex_name))
            vmp::except("packet::http::HeadersHelper::print() [name='%s'] bad value",tmp.first.c_str());
        if(!vmp::unicode::str_regex_matching(tmp.second,regex_value))
            vmp::except("packet::http::HeadersHelper::print() [value='%s'] in [name='%s'] bad value",tmp.second.c_str(),tmp.first.c_str());
        vmp::unicode::str_cwrite(&ret,"%s: %s\r\n",tmp.first.c_str(),tmp.second.c_str());
    }
    return ret;
}

void HeadersHelper::set_requestheaders(packet::Packet *packet)
{
    vmp::except_check_pointer((void *) packet,"packet::http::UriHelper::set_requestheaders(packet=null)");
    ((*packet)[packet::http::P_HTTP_REQUEST])->set("headers",print());
}

void HeadersHelper::set_responseheaders(packet::Packet *packet)
{
    vmp::except_check_pointer((void *) packet,"packet::http::UriHelper::set_responseheaders(packet=null)");
    ((*packet)[packet::http::P_HTTP_RESPONSE])->set("headers",print());
}

}}}

