/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 *Date last update: 05/08/2025
 */

#include "net.h"

namespace vampiria { namespace net {

SocksUI::SocksUI(event::Manager *manager):event::UI(manager)
{
    common_=new net::SocksCommon(manager); 
}

SocksUI::~SocksUI()
{
    delete common_;    
}

vmp::str SocksUI::identity(event::Cell *cell)
{
    vmp::str ret="";
    net::EventSocks *socks=cell->event<net::EventSocks>();
    net::Address *local=socks->evt_connection_localaddr(),*peer;
    net::Address *address=socks->evt_socks_address();
    vmp_int type=socks->evt_connection_type();
    if(type == net::CONN_TCPLISTEN)
        vmp::unicode::str_write(&ret,"SocksUI_Listen:[%s:%s]",local->host().c_str(),local->service().c_str());
    else
    {
        peer=socks->evt_connection_peeraddr();
        if(address->host() == "")
            vmp::unicode::str_write(&ret,"SocksUI:[%s:%s]->[%s:%s]->init",peer->host().c_str(),peer->service().c_str(),local->host().c_str(),local->service().c_str());
        else
            vmp::unicode::str_write(&ret,"SocksUI:[%s:%s]->[%s:%s]->%s:[%s:%s]",peer->host().c_str(),peer->service().c_str(),local->host().c_str(),local->service().c_str(),socks->evt_socks_command_str().c_str(),address->host().c_str(),address->service().c_str());
    }
    return ret;
}

void SocksUI::close_event(event::Cell *cell)
{
    net::EventSocks *socks=cell->event<net::EventSocks>();   
    socks->evt_socks_close();
}

void SocksUI::free_ref(event::Cell *cell)
{
    net::EventSocks *socks=cell->event<net::EventSocks>();   
    socks->evt_socks_free();
    sref_.free(socks);
}

event::Event *SocksUI::child_event_new(event::Cell *cell)
{
    return (event::Event *) sref_.get();  
}

void SocksUI::set_tcp_event(net::EVTCBACCEPT accept,event::EVTCB lclose,event::EVTCB close)
{
    common_->set_tcp_event(accept,lclose,close);
}

void SocksUI::set_timeout(vmp::time::Time ctimeout)
{
    common_->set_timeout(ctimeout);
}

void SocksUI::set_auths(net::SOCKSCBNOAUTH noauth,net::SOCKSCBUSERPASS userpass)
{
    common_->set_auths(noauth,userpass);
}

void SocksUI::set_connect_event(net::SOCKSCBREQUEST cmdconnect,event::EVTCBBUF crecv)
{
    common_->set_connect_event(cmdconnect,crecv);
}

void SocksUI::set_bind_event(net::SOCKSCBREQUEST cmdbind,event::EVTCBBUF brecv)
{
    common_->set_bind_event(cmdbind,brecv);
}

void SocksUI::set_udpassociate_event(net::SOCKSCBREQUEST cmdudpassociate,net::EVTCBRECVFROM udprecv,vmp_size udpmaxsize)
{
    common_->set_udpassociate_event(cmdudpassociate,udprecv,udpmaxsize);
}

event::Cell *SocksUI::new_server(net::Address *local,vmp_size backlog)
{
    event::Cell *cell=0;
    manager_->lock();
    net::EventSocks *socks=sref_.get();
    try
    {
        cell=socks->evt_socks_listen(this,local,backlog,common_);
    }
    catch(vmp::exception &x)
    {
        sref_.free(socks);
        manager_->unlock();
        vmp::except("event::SocksUI::new_server() '%s'",x.what());
    }
    manager_->unlock();
    return cell;
}

void SocksUI::reply_error(event::Cell *cell,vmp_byte result,vmp::str msg)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventSocks *socks=cell->event<net::EventSocks>();
    socks->evt_socks_reply_error(result,msg);
    manager->unlock();
}

void SocksUI::reply_connect(event::Cell *cell,net::Address *laddress)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventSocks *socks=cell->event<net::EventSocks>();
    socks->evt_socks_reply_connect(laddress);
    manager->unlock();
}

void SocksUI::reply_bind_listen(event::Cell *cell,net::Address *listen)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventSocks *socks=cell->event<net::EventSocks>();
    socks->evt_socks_reply_bind_listen(listen);
    manager->unlock();
}

void SocksUI::reply_bind_connect(event::Cell *cell,net::Address *connect)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventSocks *socks=cell->event<net::EventSocks>();
    socks->evt_socks_reply_bind_connect(connect);
    manager->unlock();
}

void SocksUI::reply_udpassociate(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventSocks *socks=cell->event<net::EventSocks>();
    socks->evt_socks_reply_udpassociate();
    manager->unlock();
}

void SocksUI::send(event::Cell *cell,vmp::Buf *buf)
{
    event::Manager *manager=cell->get_manager();
    buf->index();
    manager->lock();
    net::EventSocks *socks=cell->event<net::EventSocks>();
    socks->evt_connection_send(buf);
    manager->unlock();
}

void SocksUI::sendTo(event::Cell *cell,vmp::Buf *buf,net::Address *peer)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventSocks *socks=cell->event<net::EventSocks>();
    socks->evt_connection_sendTo(buf,peer);
    manager->unlock();
}

net::Address *SocksUI::localaddr(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventSocks *socks=cell->event<net::EventSocks>();
    net::Address *ret=socks->evt_connection_localaddr();
    manager->unlock();
    return ret;
}

net::Address *SocksUI::peeraddr(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventSocks *socks=cell->event<net::EventSocks>();
    net::Address *ret=socks->evt_connection_peeraddr();
    manager->unlock();
    return ret; 
}

vmp_int SocksUI::connection_type(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventSocks *socks=cell->event<net::EventSocks>();
    vmp_int ret=socks->evt_connection_type();
    manager->unlock();
    return ret;
}

vmp::str SocksUI::connection_strtype(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventSocks *socks=cell->event<net::EventSocks>();
    vmp::str ret=socks->evt_connection_strtype();
    manager->unlock();
    return ret;
}

vmp_byte SocksUI::cstatus(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventSocks *socks=cell->event<net::EventSocks>();
    vmp_byte ret=socks->evt_socks_cstatus();
    manager->unlock();
    return ret;
}

vmp::str SocksUI::userid(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventSocks *socks=cell->event<net::EventSocks>();
    vmp::str ret=socks->evt_socks_userid();
    manager->unlock();
    return ret;
}

vmp_uint SocksUI::permits(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventSocks *socks=cell->event<net::EventSocks>();
    vmp_uint ret=socks->evt_socks_permits();
    manager->unlock();
    return ret;
}

vmp_byte SocksUI::socks_command(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventSocks *socks=cell->event<net::EventSocks>();
    vmp_byte ret=socks->evt_socks_command();
    manager->unlock();
    return ret;
}

vmp::str SocksUI::socks_command_str(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventSocks *socks=cell->event<net::EventSocks>();
    vmp::str ret=socks->evt_socks_command_str();
    manager->unlock();
    return ret;
}

net::Address *SocksUI::socks_address(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventSocks *socks=cell->event<net::EventSocks>();
    net::Address *ret=socks->evt_socks_address();
    manager->unlock();
    return ret;
}

net::Address *SocksUI::socks_udpbind(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventSocks *socks=cell->event<net::EventSocks>();
    net::Address *ret=socks->evt_socks_udpbind();
    manager->unlock();
    return ret;
}

}}

