/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>evt_reply_error
 * Date last update: 04/11/2024
*/
 
#include "net.h"
 
namespace vampiria { namespace net {
 
void socks_accept(event::Cell *cell,event::Cell *child)
{
    event::Manager *manager=cell->get_manager();
    net::EventSocks *socks=cell->event<net::EventSocks>();
    net::EventSocks *csocks=child->event<net::EventSocks>();
    manager->lock();
    csocks->common_=socks->common_;
    manager->unlock();
    csocks->common_->accept_(cell,child);
    manager->lock();
    csocks->ctime_=manager->cell_get_timeout(child);
    manager->cell_timewait(child,csocks->common_->ctimeout_);
    manager->unlock();
}
 
void socks_init(event::Cell *cell,vmp::Buf *buf)
{
    event::Manager *manager=cell->get_manager();    
    vmp_byte res;
    vmp_uint permits;
    vmp::vector<vmp_byte> auths;
    net::Address address;
    vmp::str err;
    vmp::Buf send;
    net::EventSocks *socks=cell->event<net::EventSocks>();
    try
    {
        auths=net::socks5_clientinit_get(buf);    
    }
    catch(vmp::exception &x)
    {
        try
        {
            buf->index();
            net::socks4_request_get(buf,&address);
            res=net::socks4_result_failed;
            net::socks4_reply(&send,res,&address);
            err="recv socks4 request";
        }
        catch(vmp::exception &x4)
        {
            err=x.what();
            res=net::socks5_result_failure;
            net::socks5_serverinit(&send,net::socks5_auth_noaccetable);
        }
        manager->lock();
        socks->evt_connection_send(&send);
        manager->cell_close_err_spec(cell,"net::EventSocks",-1,err);
        manager->unlock();
        return;
        
    }
    res=net::socks5_auth_noaccetable;
    manager->lock();
    for(vmp_index i=0;i<auths.size();i++)
    {
        if(vmp::invector<vmp_byte>(auths[i],socks->common_->auths_))
        {
            switch(auths[i])
            {
                case net::socks5_auth_noauth:
                    manager->unlock();
                    try
                    {
                        permits=socks->common_->noauth_(cell);
                    }
                    catch(vmp::exception &x)
                    {
                        vmp::error("net::socks_init()[socks->common_->noauth_] %s",x.what());
                    }
                    manager->lock();
                    if(permits > 0)
                    {    
                        socks->permits_=permits;
                        res=net::socks5_auth_noauth;
                        socks->recv_=net::socks_request;
                    }
                    break;
                case net::socks5_auth_user:
                    res=net::socks5_auth_user;
                    socks->recv_=net::socks_userauth;
                    break;
                default:
                    break;
            }    
        }
        if(res != net::socks5_auth_noaccetable)
            break;
    }
    net::socks5_serverinit(&send,res);
    socks->evt_connection_send(&send);
    if(res == net::socks5_auth_noaccetable)
        manager->cell_close_err_spec(cell,"net::EventSocks",-1,"Bad authentication");    
    manager->unlock();
}

void socks_userauth(event::Cell *cell,vmp::Buf *buf)
{
    event::Manager *manager=cell->get_manager();
    vmp::Buf send;
    vmp::str user,password;
    vmp_uint permits=0;
    vmp_byte res=net::socks5_result_ok;
    net::EventSocks *socks=cell->event<net::EventSocks>();
    try
    {
        net::socks5_userauth_get(buf,&user,&password);
    }
    catch(vmp::exception &x)
    {
        net::socks5_userauth_reply(&send,net::socks5_result_failure);
        manager->lock();
        socks->evt_connection_send(&send);
        manager->cell_close_err_spec(cell,"net::EventSocks",-1,x.what());
        manager->unlock();
        return;
    }
    permits=socks->common_->userpass_(cell,user,password);
    manager->lock();
    if(permits > 0)
    {
        socks->userid_=user;
        socks->permits_=permits;
        socks->recv_=net::socks_request;
    }
    else
        res=net::socks5_result_refused;
    net::socks5_userauth_reply(&send,res);
    socks->evt_connection_send(&send);
    if(res != net::socks5_result_ok)
        manager->cell_close_err_spec(cell,"net::EventSocks",(vmp_int)res,net::socks_msg_result(res));
    manager->unlock();
}

void socks_request(event::Cell *cell,vmp::Buf *buf)
{
    event::Manager *manager=cell->get_manager();
    vmp::Buf send;
    net::EventSocks *socks=cell->event<net::EventSocks>();
    net::EventConnection *udp,*parent;
    event::Cell *associate;
    net::Address bind;
    vmp_bool exec=false;
    manager->lock();
    try
    {
        socks->command_=net::socks5_request_get(buf,&(socks->address_));
        socks->cstatus_=net::evtsocks_status_request;
        socks->recv_=net::socks_wait_recv;
    }
    catch(vmp::exception &x)
    {
        socks->evt_socks_reply_error(net::socks5_result_failure,x.what());
        manager->unlock();
        return;   
    }
    switch(socks->command_)
    {
        case socks_cd_connect:
            manager->unlock();
            socks->common_->cmdconnect_(cell,&(socks->address_));
            break;
        case socks_cd_bind:
            manager->unlock();
            socks->common_->cmdbind_(cell,&(socks->address_));
            break;
        case socks_cd_udpassociate:
            udp=socks->common_->cref_.get();
            try
            {
                parent=socks->parent_->event<net::EventConnection>();
                bind.set(parent->evt_connection_localaddr()->ip());
                associate=udp->evt_connection_udpbind(socks->common_,&bind,socks->common_->udpmaxsize_,net::socks_udp_recv,0);
                manager->cell_addsub(cell,"udpbind",associate);
                exec=true;
            }
            catch(vmp::exception &x)
            {
                socks->common_->cref_.free(udp);
                socks->evt_socks_reply_error(net::socks5_result_failure,x.what());
            }
            manager->unlock();
            if(exec)
                socks->common_->cmdudpassociate_(cell,&(socks->address_));
            break;
        default:
            socks->evt_socks_reply_error(net::socks5_result_badcmd);
            manager->unlock();
            break;
    }
}

void socks_wait_recv(event::Cell *cell,vmp::Buf *buf)
{
    event::Manager *manager=cell->get_manager();
    net::EventSocks *socks=cell->event<net::EventSocks>();
    manager->lock();
    socks->evt_socks_reply_error(net::socks5_result_failure,0);
    manager->unlock();
}

void socks_default_cmd(event::Cell *cell,net::Address *address)
{
    event::Manager *manager=cell->get_manager();
    net::EventSocks *socks=cell->event<net::EventSocks>();
    manager->lock();
    socks->evt_socks_reply_error(net::socks5_result_badcmd);
    manager->unlock();
}

void socks_udp_recv(event::Cell *cell,vmp::Buf *buf,net::Address *peer)
{
    event::Manager *manager=cell->get_manager();
    vmp_bool exec=false;
    manager->lock();
    net::EventConnection *udp=cell->event<net::EventConnection>();
    net::EventSocks *socks=udp->master_->event<net::EventSocks>();
    if(buf->size() == 0)
        manager->cell_close_err_spec(cell,"net::EventSocks",0,"recv empty package");
    else if(buf->size() > udp->udpmaxsize_)
        manager->cell_close_err_spec(cell,"net::EventSocks",0,"recv package too large");
    else if((socks->address_.ip() == peer->ip()) && (socks->address_.service() == peer->service()))
        exec=true;
    manager->unlock();
    if(exec)
    {
        net::Address address;
        vmp::Buf data;
        buf->index();
        net::socks5_udp_get(buf,&(socks->udplastfrag_),&address,&data);
        data.index();
        socks->common_->udprecv_(cell->master(),&data,&address);
    }
}

void socks_udp_send(event::Cell *cell,vmp::Buf *buf,net::Address *peer)
{
    event::Manager *manager=cell->get_manager();
    vmp::Buf out,send;
    vmp::str msg;
    net::EventConnection *udp=manager->cell_searchsub(cell,"udpbind")->event<net::EventConnection>();
    net::EventSocks *evt=cell->event<net::EventSocks>();
    vmp_int ret=net::proxy_udp_send(udp,buf,peer,&evt->address_,&msg);
    if(ret == 0)
        manager->cell_close_err_spec(cell,"net::EventSocks",0,msg);
    else if (ret == -1)
        manager->cell_close_err_spec(cell,"net::EventSocks",vmp::get_errno(),vmp::value_errno());  
}

}}
 
