/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 18/07/2025
*/
 
#include "net.h"

namespace vampiria { namespace net {

vmp::str socks_msg_result(vmp_byte result)
{
    switch(result)
    {
        case net::socks5_result_ok:
            return "Socks5(Reply request granted)";
        case net::socks5_result_failure:
            return "Socks5(General SOCKS server failure)";
        case net::socks5_result_notallowed:
            return "Socks5(Connection not allowed by ruleset)";
        case net::socks5_result_netunreachable:
            return "Socks5(Network unreachable)";
        case net::socks5_result_hostunreachable:
            return "Socks5(Host unreachable)";
        case net::socks5_result_refused:
            return "Socks5(Connection refused)";
        case net::socks5_result_ttlexpired:
            return "Socks5(Ttl expired)";
        case net::socks5_result_badcmd:
            return "Socks5(Command not supported)";
        case net::socks5_result_badatype:
            return "Socks5(Address type not supported)";
        case net::socks4_result_granted:
            return "Socks4(Reply request granted)";
        case net::socks4_result_failed:
            return "Socks4(Reply request rejected or failed)";
        case net::socks4_result_reject:
            return "Socks4(Reply request rejected because SOCKS server cannot connect to identd on the client)";
        case net::socks4_result_rejectuserid:
            return "Socks4(Reply request rejected because the client program and identd report different user-ids)";
        case net::socks_internal_badauth:
            return "Internal(Recv bad authentication code)";
        case net::socks_internal_badaddr:
            return "Internal(Reply bad address)";
        case net::socks_internal_accessdenied:
            return "Internal(Access Denied)";
    }
    return "Undefined";
}

void socks4_writeaddress(vmp::Buf *out,net::Address *address)
{
    vmp_uint service=vmp::unicode::str_todigit_range(address->service(),0,65535);
    out->write_size(service,2);
    if(address->is_onlyhost())
    {
        vmp::str host=address->host();
        if(net::is_ipv4_raw(host))
        {
            net::Address tmp;
            tmp.set(host);
            net::ip_to_buf(&tmp,out);
        }
        else if(address->is_ipv6())
            vmp::except_s("address='ipv6 not supported from sock4'");
        else
        {
            net::Address tmp;
            tmp.set("0.0.0.255");
            net::ip_to_buf(&tmp,out);
            out->write_str(host);
        }
    }
    else if(address->is_ipv4())
        net::ip_to_buf(address,out);
    else if(address->is_ipv6())
        vmp::except_s("address='ipv6 not supported from sock4'");
    else
    {
        vmp::str host=address->host();
        net::Address tmp;
        tmp.set("0.0.0.255");
        net::ip_to_buf(&tmp,out);
        out->write_str(host);
    }
}

void socks4_request(vmp::Buf *out,vmp_byte cd,net::Address *address,vmp::str userid)
{
    vmp::except_check_pointer((void *)out,"net::socks4_request(out=null)");
    vmp::except_check_pointer((void *)address,"net::socks4_request(address=null)");
    out->write_byte(0x04);
    out->write_byte(cd);
    try
    {
        net::socks4_writeaddress(out,address);
        if(address->is_ipv4() || (address->is_onlyhost() && net::is_ipv4_raw(address->host())))
            out->write_str(userid);
    }
    catch(vmp::exception &x)
    {
        vmp::except("net::socks4_request(address='%s')",x.what());
    }
    out->write_byte(0x00);
}

vmp_byte socks4_request_get(vmp::Buf *input,net::Address *address,vmp::str *userid)
{
    vmp_byte cd;
    vmp::except_check_pointer((void *)input,"net::socks4_request_get(input=null)");
    vmp::except_check_pointer((void *)address,"net::socks4_request_get(address=null)");
    try
    {
        if(input->read_byte() != 0x04)
            vmp::except("invalid version number");
        cd=input->read_byte();
        vmp_uint service=input->read_size(2);
        vmp::str sservice;
        vmp::unicode::str_write(&sservice,"%u",service);
        net::Address tmp;
        net::ipv4_from_buf(&tmp,input);
        vmp::vector<vmp::str> ip=vmp::unicode::str_split(tmp.ip(),".");
        vmp::str uid="";
        vmp_byte b;
        while((b=input->read_byte()) != 0x00)
            vmp::unicode::str_cwrite(&uid,"%c",b);
        //socks 4a
        if((ip[0] == "0") && (ip[1] == "0") && (ip[2] == "0") && (ip[3] != "0"))
            address->set(uid,sservice,true);      
        else
        {
            address->set(tmp.ip(),sservice);
            if(userid != 0)
                (*userid)=uid;    
        }
    }
    catch(vmp::exception &x)
    {
        vmp::except("net::socks4_request_get() %s",x.what());
    }
    return cd;
}

void socks4_reply(vmp::Buf *out,vmp_byte cd,net::Address *address)
{
    vmp::except_check_pointer((void *)out,"net::socks4_reply(out=null)");   
    vmp::except_check_pointer((void *)address,"net::socks4_reply(address=null)");
    out->write_byte(0x00);
    out->write_byte(cd);
    try
    {
        net::socks4_writeaddress(out,address);
    }
    catch(vmp::exception &x)
    {
        vmp::except("net::socks4_reply(address='%s')",x.what());
    }
}

vmp_byte socks4_reply_get(vmp::Buf *input,net::Address *address)
{
    vmp::except_check_pointer((void *)input,"net::socks4_reply_get(input=null)");
    vmp::except_check_pointer((void *)address,"net::socks4_reply_get(address=null)");
    if(input->read_byte() != 0x00)
        vmp::except("net::socks4_reply_get() bad version number");
    vmp_byte cd=input->read_byte();
    vmp_uint service=input->read_size(2);
    vmp::str sservice;
    vmp::unicode::str_write(&sservice,"%u",service);
    net::ipv4_from_buf(address,input);
    address->set(address->host(),sservice);
    return cd;
}

void socks5_writeaddress(vmp::Buf *out,net::Address *address)
{
    if(address->is_onlyhost())
    {
        vmp::str host=address->host();
        net::Address tmp;
        if(net::is_ipv4_raw(host))
        {
            out->write_byte(net::socks5_atype_ipv4);
            tmp.set(host);
            net::ip_to_buf(&tmp,out);
        }
        else if(address->is_ipv6())
        {
            out->write_byte(net::socks5_atype_ipv6);
            tmp.set(host);
            net::ip_to_buf(&tmp,out);
        }
        else
        {
            out->write_byte(net::socks5_atype_domain);
            out->write_size(host.size(),1);
            out->write_str(host);
        }
    }
    else if(address->is_ipv4())
    {    
        out->write_byte(net::socks5_atype_ipv4);
        net::ip_to_buf(address,out);
    }   
    
    else if(address->is_ipv6())
    {
        out->write_byte(net::socks5_atype_ipv6);
        net::ip_to_buf(address,out);
    }
    else
    {
        vmp::str host=address->host();
        out->write_byte(net::socks5_atype_domain);
        out->write_size(host.size(),1);
        out->write_str(host);
    }
    vmp_uint service=vmp::unicode::str_todigit_range(address->service(),0,65535);
    out->write_size(service,2);
}

void socks5_readaddress(vmp::Buf *input,net::Address *address)
{
    net::Address tmp;
    vmp::str host;
    vmp_byte atype=input->read_byte();
    vmp_size s;
    switch(atype)
    {
        case net::socks5_atype_ipv4:
            net::ipv4_from_buf(&tmp,input);
            break;
        case net::socks5_atype_ipv6:
            net::ipv6_from_buf(&tmp,input);
            break;
        case net::socks5_atype_domain:
            s=input->read_size(1);
            host=input->read_str(s);
            break;
        default:
            vmp::except("bad atype address"); 
    }
    vmp_uint service=input->read_size(2);
    vmp::str sservice;
    vmp::unicode::str_write(&sservice,"%u",service);
    if(tmp.is_ipv4() || tmp.is_ipv6())
        address->set(tmp.ip(),sservice);
    else
        address->set(host,sservice,true);
}

void socks5_clientinit(vmp::Buf *out,vmp::vector<vmp_byte> auths)
{
    vmp::except_check_pointer((void *)out,"net::socks5_clientinit(out=null)");
    out->write_byte(0x05);
    out->write_size(auths.size(),1);
    out->write_vector(auths);
}

vmp::vector<vmp_byte> socks5_clientinit_get(vmp::Buf *input)
{
    if(input->read_byte() != 0x05)
        vmp::except("net::socks5_clientinit_get() bad version number");
    vmp_size nmethods=input->read_size(1);
    vmp::vector<vmp_byte> ret=input->read_vector(nmethods);
    return ret;
}

void socks5_serverinit(vmp::Buf *out,vmp_byte auth)
{
    vmp::except_check_pointer((void *)out,"net::socks5_serverinit(out=null)");
    out->write_byte(0x05);
    out->write_byte(auth); 
}

vmp_byte socks5_serverinit_get(vmp::Buf *input)
{
    if(input->read_byte() != 0x05)
        vmp::except("net::socks5_serverinit_get() bad version number");
    return input->read_byte();
}

void socks5_request(vmp::Buf *out,vmp_byte cd,net::Address *address)
{
    vmp::except_check_pointer((void *)out,"net::socks5_request(out=null)");
    vmp::except_check_pointer((void *)address,"net::socks5_request(address=null)"); 
    out->write_byte(0x05);
    out->write_byte(cd);
    out->write_byte(0x00);
    net::socks5_writeaddress(out,address);
}

vmp_byte socks5_request_get(vmp::Buf *input,net::Address *address)
{
    vmp::except_check_pointer((void *)input,"net::socks5_request_get(input=null)");
    vmp::except_check_pointer((void *)address,"net::socks5_request_get(address=null)");
    if(input->read_byte() != 0x05)
        vmp::except("net::socks5_request_get() bad version number");
    vmp_byte cd=input->read_byte();
    input->read_byte();
    try
    {
        socks5_readaddress(input,address);
    }
    catch(vmp::exception &x)
    {
        vmp::except("net::socks5_request_get() %s",x.what());
    }
    return cd;
}

void socks5_reply(vmp::Buf *out,vmp_byte cd,net::Address *address)
{
    vmp::except_check_pointer((void *)out,"net::socks5_reply(out=null)");
    vmp::except_check_pointer((void *)address,"net::socks5_request(address=null)");
    out->write_byte(0x05);
    out->write_byte(cd);
    out->write_byte(0x00);
    net::socks5_writeaddress(out,address);
}

vmp_byte socks5_reply_get(vmp::Buf *input,net::Address *address)
{
    vmp::except_check_pointer((void *)input,"net::socks5_reply_get(input=null)");
    vmp::except_check_pointer((void *)address,"net::socks5_reply_get(address=null)");
    net::Address tmp;
    vmp::str host;
    if(input->read_byte() != 0x05)
        vmp::except("net::socks5_reply_get() bad version number");
    vmp_byte cd=input->read_byte();
    input->read_byte();
    try
    {
        socks5_readaddress(input,address);
    }
    catch(vmp::exception &x)
    {
        vmp::except("net::socks5_reply_get() %s",x.what());
    }
    return cd;
}

void socks5_udp(vmp::Buf *out,vmp_byte frag,net::Address *address,vmp::Buf *data)
{
    vmp::except_check_pointer((void *)out,"net::socks5_udp(out=null)");
    vmp::except_check_pointer((void *)address,"net::socks5_udp(address=null)");
    vmp::except_check_pointer((void *)data,"net::socks5_udp(data=null)");
    out->write_byte(0x00);
    out->write_byte(0x00);
    if(frag > 255)
        vmp::except("net::socks5_udp(frag=%u) bad value",frag);
    out->write_byte(frag);
    net::socks5_writeaddress(out,address);
    out->write_buf(data);
}

void socks5_udp_get(vmp::Buf *input,vmp_byte *frag,net::Address *address,vmp::Buf *data)
{
    vmp::except_check_pointer((void *)input,"net::socks5_udp_get(input=null)");
    vmp::except_check_pointer((void *)frag,"net::socks5_udp_get(frag=null)");
    vmp::except_check_pointer((void *)address,"net::socks5_udp_get(address=null)");
    vmp::except_check_pointer((void *)data,"net::socks5_udp_get(data=null)");
    input->read_vector(2);
    (*frag)=input->read_byte();
    try
    {
        socks5_readaddress(input,address);
    }
    catch(vmp::exception &x)
    {
        vmp::except("net::socks5_reply_get() %s",x.what());
    }
    input->read_buf(data,input->size_reading());
}

void socks5_userauth(vmp::Buf *out,vmp::str user,vmp::str password)
{
    vmp::except_check_pointer((void *)out,"net::socks5_userauth(out=null)");
    out->write_byte(0x01);
    out->write_size(user.size(),1);
    out->write_str(user);
    out->write_size(password.size(),1);
    out->write_str(password);
}

void socks5_userauth_get(vmp::Buf *input,vmp::str *user,vmp::str *password)
{
    vmp::except_check_pointer((void *)input,"net::socks5_userauth_get(input=null)");
    vmp::except_check_pointer((void *)user,"net::socks5_userauth_get(user=null)");
    vmp::except_check_pointer((void *)password,"net::socks5_userauth_get(password=null)");
    vmp_size s;
    vmp_byte b=input->read_byte();
    if(b != 0x01)
        vmp::except("net::socks5_userauth_get() bad subnegotiation version");
    s=input->read_size(1);
    (*user)=input->read_str(s);
    s=input->read_size(1);
    (*password)=input->read_str(s);   
}

void socks5_userauth_reply(vmp::Buf *out,vmp_byte status)
{
    vmp::except_check_pointer((void *)out,"net::socks5_userauth_reply(out=null)");
    out->write_byte(0x01);
    out->write_byte(status);   
}

vmp_byte socks5_userauth_reply_get(vmp::Buf *input)
{
    vmp::except_check_pointer((void *)input,"net::socks5_userauth_reply_get(input=null)");
    vmp_byte b=input->read_byte();
    if(b != 0x01)
        vmp::except("net::socks5_userauth_reply_get() bad subnegotiation version");
    return input->read_byte();
}

Proxy::Proxy(vmp::str type,net::Address *address)
{
    type_=type;
    address_.copy(address);
    user_="";
    password_="";
}
        
Proxy::~Proxy()
{
    type_="";
    address_.reset();
    user_="";
    password_="";
}

void Proxy::set_userauth(vmp::str user,vmp::str password)
{
    user_=user;
    password_=password;
}

vmp::str Proxy::type()
{
    return type_;
}

net::Address *Proxy::address()
{
    return &address_;
}

vmp::str Proxy::user()
{
    return user_;
}

vmp::str Proxy::password()
{
    return password_;
}

ProxyChain::ProxyChain(event::Manager *manager,vmp::utils::Logger *logger):event::UI(manager)
{
    logger_=logger;
    set_timeout(0.0);
}

ProxyChain::~ProxyChain()
{
    reset();
    logger_=0;
}

void ProxyChain::set_timeout(vmp::time::Time timeout)
{
    if(timeout == 0.0)
        timeout_=5.0;
    else
        timeout_=timeout;
}

vmp::time::Time ProxyChain::get_timeout()
{
    return timeout_;
}

void ProxyChain::close_event(event::Cell *cell)
{
    net::EventConnection *evt=cell->event<net::EventConnection>();
    evt->evt_connection_close();
}
        
void ProxyChain::free_ref(event::Cell *cell)
{
    net::EventConnection *evt=cell->event<net::EventConnection>();
    evt->evt_connection_free();
    cref_.free(evt);
}

vmp_index ProxyChain::replace(net::Proxy *proxy,vmp_index i)
{
    net::Proxy *tmp;
    if(i < chain_.size())
    {
        tmp=chain_[i];
        chain_[i]=proxy;
        delete tmp;
        return i;
    }
    chain_.push_back(proxy);
    return chain_.size()-1;
}

vmp_index ProxyChain::insert(net::Proxy *proxy,vmp_index i)
{
    vmp_size s=chain_.size();
    if(i<s)
    {
        chain_.push_back(chain_[s-1]);
        for(vmp_index j=s-1;j>i;j--)
            chain_[j]=chain_[j-1];
        chain_[i]=proxy;
        return i;
    }
    chain_.push_back(proxy);
    return chain_.size()-1;
}

void ProxyChain::reset()
{
    vmp::vector_delete_alldata<net::Proxy*>(&chain_);
}

vmp_size ProxyChain::size()
{
    return chain_.size();
}

vmp_index ProxyChain::push_socks4(net::Address *address)
{
    net::Proxy *proxy=new Proxy("socks4",address);
    chain_.push_back(proxy);
    return chain_.size()-1;
}

vmp_index ProxyChain::insert_socks4(net::Address *address,vmp_index i)
{
    net::Proxy *proxy=new Proxy("socks4",address);
    return insert(proxy,i);
}

vmp_index ProxyChain::replace_socks4(net::Address *address,vmp_index i)
{
    net::Proxy *proxy=new Proxy("socks4",address);
    return replace(proxy,i);
}

vmp_index ProxyChain::push_socks5(net::Address *address)
{
    net::Proxy *proxy=new Proxy("socks5",address);
    chain_.push_back(proxy);
    return chain_.size()-1;
}

vmp_index ProxyChain::insert_socks5(net::Address *address,vmp_index i)
{
    net::Proxy *proxy=new Proxy("socks5",address);
    return insert(proxy,i);
}

vmp_index ProxyChain::replace_socks5(net::Address *address,vmp_index i)
{
    net::Proxy *proxy=new Proxy("socks5",address);
    return replace(proxy,i);
}

vmp_index ProxyChain::push_socks5_userauth(net::Address *address,vmp::str user,vmp::str password)
{
    net::Proxy *proxy=new Proxy("socks5_userauth",address);
    proxy->set_userauth(user,password);
    chain_.push_back(proxy);
    return chain_.size()-1;  
}

vmp_index ProxyChain::insert_socks5_userauth(net::Address *address,vmp_index i,vmp::str user,vmp::str password)
{
    net::Proxy *proxy=new Proxy("socks5_userauth",address);
    proxy->set_userauth(user,password);
    return insert(proxy,i);
}

vmp_index ProxyChain::replace_socks5_userauth(net::Address *address,vmp_index i,vmp::str user,vmp::str password)
{
    net::Proxy *proxy=new Proxy("socks5_userauth",address);
    proxy->set_userauth(user,password);
    return replace(proxy,i);
}

net::Proxy *ProxyChain::get(vmp_index i)
{
    if(i < chain_.size())
        return chain_[i];
    return 0;
}

vmp_size ProxyChain::cancel(vmp_index i)
{
    vmp_size s=chain_.size();
    if(i < s)
    {
        delete chain_[i];
        for(vmp_index j=i;j<s-1;j++)
            chain_[i]=chain_[i+1];
        chain_.pop_back();
    }
    return chain_.size();
}

event::Cell *ProxyChain::tcpconnect(net::EventConnection *evt,event::UI *ui,event::EVTCB close)
{
    vmp::str err;
    event::Cell *cell;
    net::Proxy *proxy=chain_[evt->proxyindex_];
    net::Address *addr=proxy->address();
    vmp::str type=proxy->type();
    vmp_int socket;
    try
    {
        socket=net::proxychain_connect(type,addr);  
    }
    catch(vmp::exception &x)
    {
        vmp::unicode::str_write(&err,"net::ProxyChain::tcpconnect() target %s:%s error '%s'",evt->peer_.host().c_str(),evt->peer_.service().c_str(),x.what());
        if(logger_ != 0)
            logger_->write_s(vmp::utils::LOG_ERR,err);
        vmp::except_s(err);
    }
    cell=evt->evt_new(ui,socket,net::proxychain_tcp_connect,net::proxychain_close);
    cell->writing_=true;
    evt->proxyoper_=net::socks_cd_connect;
    evt->proxyclose_=close;
    if(logger_ != 0)
        logger_->write(vmp::utils::LOG_INFO,"net::ProxyChain::tcpconnect(cellid=%u) target %s:%s connection start",evt->cell_->id(),evt->peer_.host().c_str(),evt->peer_.service().c_str());
    return cell;
}

event::Cell *ProxyChain::tcpbind(net::EventConnection *evt,net::Address *ref,event::UI *ui,event::EVTCB close)
{
    vmp::str err;
    event::Cell *cell;
    net::Proxy *proxy=chain_[evt->proxyindex_];
    net::Address *addr=proxy->address();
    vmp::str type=proxy->type();
    vmp_int socket;
    try
    {
        socket=net::proxychain_connect(type,addr);  
    }
    catch(vmp::exception &x)
    {
        vmp::unicode::str_write(&err,"net::ProxyChain::tcpbind() ref %s:%s error '%s'",ref->host().c_str(),ref->service().c_str(),x.what());
        if(logger_ != 0)
            logger_->write_s(vmp::utils::LOG_ERR,err);
        vmp::except_s(err);
    }
    cell=evt->evt_new(ui,socket,net::proxychain_tcp_connect,net::proxychain_close);
    cell->writing_=true;
    evt->proxyoper_=net::socks_cd_bind;
    evt->proxyclose_=close;
    if(logger_ != 0)
        logger_->write(vmp::utils::LOG_INFO,"net::ProxyChain::tcpconnect(cellid=%u) ref %s:%s connection start",evt->cell_->id(),ref->host().c_str(),ref->service().c_str());
    return cell;
}

event::Cell *ProxyChain::udpassociate(net::EventConnection *evt,event::UI *ui,event::EVTCB close)
{
    vmp::str err;
    event::Cell *cell;
    net::Proxy *proxy=chain_[evt->proxyindex_];
    if(!((size() == 1) && ((proxy->type() == "socks5") || (proxy->type() == "socks5_userauth"))))
        vmp::except("net::ProxyChain::udpassociate() target %s:%s bad proxy config",evt->peer_.host().c_str(),evt->peer_.service().c_str());
    net::Address *addr=proxy->address();
    vmp::str type=proxy->type();
    vmp_int socket;
    try
    {
        socket=net::proxychain_connect(type,addr);     
    }
    catch(vmp::exception &x)
    {
        vmp::unicode::str_write(&err,"net::ProxyChain::udpassociate() target %s:%s error '%s'",evt->peer_.host().c_str(),evt->peer_.service().c_str(),x.what());
        if(logger_ != 0)
            logger_->write_s(vmp::utils::LOG_ERR,err);
        vmp::except_s(err);
    }
    cell=evt->evt_new(ui,socket,net::proxychain_tcp_connect,net::proxychain_close);
    cell->writing_=true;
    evt->proxyoper_=net::socks_cd_udpassociate;
    evt->proxyclose_=close;
    if(logger_ != 0)
        logger_->write(vmp::utils::LOG_INFO,"net::ProxyChain::udpassociate(cellid=%u) target %s:%s connection start",evt->cell_->id(),evt->peer_.host().c_str(),evt->peer_.service().c_str());
    return cell;
}

void ProxyChain::socks5_request(net::EventConnection *evt)
{
    vmp::Buf out;
    net::Address address;
    if(evt->proxyoper_ == net::socks_cd_udpassociate)
    {
        net::EventConnection *udp=cref_.get();
        address.set(evt->evt_connection_localaddr()->ip());
        event::Cell *cell=udp->evt_connection_udpbind(this,&address,evt->evt_connection_udpmaxsize(),net::proxychain_udp_recv,0);
        manager_->cell_addsub(evt->cell_,"udpassociate",cell);
        net::socks5_request(&out,evt->proxyoper_,udp->evt_connection_localaddr());
        evt->proxystatus_="socks5_udpassociate";
    }
    else
    {
        if(evt->proxyindex_ < (size()-1))
        {
            net::socks5_request(&out,net::socks_cd_connect,chain_[(evt->proxyindex_)+1]->address());
            evt->proxystatus_="socks5_connect";
        }
        else if(evt->proxyoper_ == net::socks_cd_connect)
        {    
            net::socks5_request(&out,evt->proxyoper_,&(evt->peer_));
            evt->proxystatus_="socks5_connect";
        }
        else if(evt->proxyoper_ == net::socks_cd_bind)
        {    
            net::socks5_request(&out,evt->proxyoper_,&(evt->bindservice_));
            evt->proxystatus_="socks5_bind";
        }
    }
    evt->evt_connection_send(&out);
}

vmp_bool ProxyChain::next(net::EventConnection *evt,vmp::Buf *buf)
{
    if(evt->proxyindex_ == size())
    {
        if(logger_ != 0)
        {
           switch(evt->proxyoper_)
           {
               case net::socks_cd_connect:
                   logger_->write(vmp::utils::LOG_INFO,"net::ProxyChain::tcpconnect(cellid=%u) connection established",evt->cell_->id());
                   break;
               case net::socks_cd_bind:
                   logger_->write(vmp::utils::LOG_INFO,"net::ProxyChain::tcpbind(cellid=%u) connection established",evt->cell_->id());
                   break;
               case socks_cd_udpassociate:
                   logger_->write(vmp::utils::LOG_INFO,"net::ProxyChain::udpassociate(cellid=%u) connection established",evt->cell_->id());
                   break;
           }
        }
        return true;
    }
    vmp::Buf out;
    vmp_byte res;
    vmp::str err;
    net::Address address,tmp;
    vmp::str type=chain_[evt->proxyindex_]->type();
    if(evt->proxystatus_=="init")
    {
        if(type == "socks4")
        {
            if(evt->proxyindex_ < (size()-1))
            {
                net::socks4_request(&out,net::socks_cd_connect,chain_[(evt->proxyindex_)+1]->address());
                evt->proxystatus_="socks4_connect";
            }
            else if(evt->proxyoper_ == net::socks_cd_connect)
            {
                net::socks4_request(&out,evt->proxyoper_,&(evt->peer_));    
                evt->proxystatus_="socks4_connect";
            }    
            else if(evt->proxyoper_ == net::socks_cd_bind)
            {        
                net::socks4_request(&out,evt->proxyoper_,&(evt->bindservice_)); 
                evt->proxystatus_="socks4_bind";
            }
        }
        if(type == "socks5")
        {
            vmp::vector<vmp_byte> auths;
            auths.push_back(net::socks5_auth_noauth);
            net::socks5_clientinit(&out,auths);
            evt->proxystatus_="socks5_init";
        } 
        else if(type == "socks5_userauth")
        {
            vmp::vector<vmp_byte> auths;
            auths.push_back(net::socks5_auth_user);
            net::socks5_clientinit(&out,auths);
            evt->proxystatus_="socks5_userauth_init";
        }
        evt->evt_connection_send(&out);
    }
    else if(evt->proxystatus_=="socks4_connect")
    {
        if((res=net::socks4_reply_get(buf,&address)) == net::socks4_result_granted)
        {
            if(evt->proxyindex_ == (size()-1))
                tmp.copy(&(evt->peer_));
            else
                tmp.copy(chain_[(evt->proxyindex_)+1]->address());
            if((address.host() != tmp.host()) && (address.service() != tmp.service()))
                net::proxychain_error(evt->cell_,"net::ProxyChain",(vmp_int) net::socks_internal_badaddr,net::socks_msg_result(net::socks_internal_badaddr),evt->proxyindex_);
            else
            {
                (evt->proxyindex_)++;
                evt->proxystatus_="init";
                return next(evt);    
            }
        }
        else
            net::proxychain_error(evt->cell_,"net::ProxyChain",(vmp_int) res,net::socks_msg_result(res),evt->proxyindex_);
    }
    else if(evt->proxystatus_=="socks4_bind")
    {
        if((res=socks4_reply_get(buf,&evt->bindaddr_)) == net::socks4_result_granted)
        {
            event::Manager *manager=evt->cell_->get_manager();
            manager->unlock();
            try
            {
                evt->bind_(evt->cell_);
            }
            catch(vmp::exception &x)
            {
            }
            manager->lock();
            evt->proxystatus_="socks4_bind2";
        }
        else
            net::proxychain_error(evt->cell_,"net::ProxyChain",(vmp_int) res,net::socks_msg_result(res),evt->proxyindex_);
    }
    else if(evt->proxystatus_=="sock4_bind2")
    {
         if((res=socks4_reply_get(buf,&evt->peer_)) == net::socks4_result_granted)
         {
             (evt->proxyindex_)++;
             evt->proxystatus_="init";
             return next(evt);
         }
         else
            net::proxychain_error(evt->cell_,"net::ProxyChain",(vmp_int) res,net::socks_msg_result(res),evt->proxyindex_);
    }
    else if(evt->proxystatus_=="socks5_init")
    {
        res=net::socks5_serverinit_get(buf);
        if(res == net::socks5_auth_noauth)
             socks5_request(evt);
        else
            net::proxychain_error(evt->cell_,"net::ProxyChain",(vmp_int) net::socks_internal_badauth,net::socks_msg_result(net::socks_internal_badauth),evt->proxyindex_);
    }
    else if(evt->proxystatus_=="socks5_userauth_init")
    {
        res=net::socks5_serverinit_get(buf);
        if(res == net::socks5_auth_user)
        {
            socks5_userauth(&out,chain_[evt->proxyindex_]->user(),chain_[evt->proxyindex_]->password());
            evt->evt_connection_send(&out);
            evt->proxystatus_="socks5_userauth_auth";
        }
        else
            net::proxychain_error(evt->cell_,"net::ProxyChain",(vmp_int) net::socks_internal_badauth,net::socks_msg_result(net::socks_internal_badauth),evt->proxyindex_);
    }
    else if(evt->proxystatus_=="socks5_connect")
    {
        res=net::socks5_reply_get(buf,&address);
        if(res != net::socks5_result_ok)
            net::proxychain_error(evt->cell_,"net::ProxyChain",(vmp_int) res,net::socks_msg_result(res),evt->proxyindex_);
        else
        {
            if(evt->proxyindex_ == (size()-1))
                tmp.copy(&(evt->peer_));
            else
                tmp.copy(chain_[(evt->proxyindex_)+1]->address());
            if((address.host() != tmp.host()) && (address.service() != tmp.service()))
                net::proxychain_error(evt->cell_,"net::ProxyChain",(vmp_int) net::socks_internal_badaddr,net::socks_msg_result(net::socks_internal_badaddr),evt->proxyindex_);
            else
            {
                (evt->proxyindex_)++;
                evt->proxystatus_="init";
                return next(evt);    
            }
        }
    }
    else if(evt->proxystatus_=="socks5_bind")
    {
        if((res=net::socks5_reply_get(buf,&evt->bindaddr_)) == net::socks5_result_ok)
        {
            event::Manager *manager=evt->cell_->get_manager();
            manager->unlock();
            try
            {
                evt->bind_(evt->cell_);
            }
            catch(vmp::exception &x)
            {
            }
            manager->lock();
            evt->proxystatus_="socks5_bind2";
        }
        else
            net::proxychain_error(evt->cell_,"net::ProxyChain",(vmp_int) res,net::socks_msg_result(res),evt->proxyindex_);
        
    }
    else if(evt->proxystatus_=="socks5_bind2")
    {
        if((res=net::socks5_reply_get(buf,&evt->peer_)) == net::socks5_result_ok)
        {
            (evt->proxyindex_)++;
            evt->proxystatus_="init";
            return next(evt);      
        }
        else
            net::proxychain_error(evt->cell_,"net::ProxyChain",(vmp_int) res,net::socks_msg_result(res),evt->proxyindex_);
        
    }
    else if(evt->proxystatus_=="socks5_udpassociate")
    {
        res=net::socks5_reply_get(buf,&address);
        if(res != net::socks5_result_ok)
            net::proxychain_error(evt->cell_,"net::ProxyChain",(vmp_int) res,net::socks_msg_result(res),evt->proxyindex_);
        else
        {
            event::Cell *udp=manager_->cell_searchsub(evt->cell_,"udpassociate");
            net::EventConnection *evtudp=udp->event<net::EventConnection>();
            evtudp->peer_.copy(&address);
            evt->send_=net::proxychain_udp_send;
            evt->cell_->read_=net::proxychain_tcp_recv_empty;
            (evt->proxyindex_)++;
            evt->proxystatus_="init";
            return next(evt);    
        }
    }
    else if(evt->proxystatus_=="socks5_userauth_auth")
    {
        res=socks5_userauth_reply_get(buf);
        if(res != 0x00)
            net::proxychain_error(evt->cell_,"net::ProxyChain",(vmp_int) net::socks_internal_accessdenied,net::socks_msg_result(net::socks_internal_accessdenied),evt->proxyindex_);
        else
            socks5_request(evt);
    }
    return false;
}

}}

