/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 04/08/2025
 */
 
#include "net.h"
 
namespace vampiria { namespace net {

SocksCommon::SocksCommon(event::Manager *manager):UI(manager)
{
    accept_=net::empty_accept_ev;
    close_=event::empty_ev;
    lclose_=event::empty_ev;
    
    ctimeout_=10.0;
    
    set_auths(0,0);
    
    cmdconnect_=net::socks_default_cmd;
    crecv_=event::empty_buf_ev;
    
    cmdbind_=net::socks_default_cmd;
    brecv_=event::empty_buf_ev;
    
    cmdudpassociate_=net::socks_default_cmd;
    udprecv_=net::empty_recvfrom_ev;
    udpmaxsize_=net::UDP_MSIZE_DEFAULT;
}

SocksCommon::~SocksCommon()
{
}

void SocksCommon::close_event(event::Cell *cell)
{
    net::EventConnection *conn=cell->event<net::EventConnection>();   
    conn->evt_connection_close();
}

void SocksCommon::free_ref(event::Cell *cell)
{
    net::EventConnection *conn=cell->event<net::EventConnection>();
    conn->evt_connection_free();
    cref_.free(conn);
}

void SocksCommon::set_tcp_event(net::EVTCBACCEPT accept,event::EVTCB lclose,event::EVTCB close)
{
    if(accept_ != 0)
        accept_=accept;
    if(lclose != 0)
        lclose_=lclose;
    if(close != 0)
        close_=close;
}

void SocksCommon::set_timeout(vmp::time::Time ctimeout)
{
    if(ctimeout != 0.0)
        ctimeout_=ctimeout;
}

void SocksCommon::set_auths(net::SOCKSCBNOAUTH noauth,net::SOCKSCBUSERPASS userpass)
{
    if(noauth == 0)
    {
        noauth_=net::empty_socksnoauth_ev;
        vmp::delinvector<vmp_byte>(net::socks5_auth_noauth,&auths_);
    }
    else
    {
        noauth_=noauth;
        vmp::insinvector<vmp_byte>(net::socks5_auth_noauth,&auths_);
    }
    if(userpass == 0)
    {
        userpass_=net::empty_socksuserpass_ev;
        vmp::delinvector<vmp_byte>(net::socks5_auth_user,&auths_);
    }
    else
    {
        userpass_=userpass;
        vmp::insinvector<vmp_byte>(net::socks5_auth_user,&auths_);
    }
}

void SocksCommon::set_connect_event(net::SOCKSCBREQUEST cmdconnect,event::EVTCBBUF crecv)
{
    if(cmdconnect != 0)
        cmdconnect_=cmdconnect;
    if(crecv != 0)
        crecv_=crecv;
}

void SocksCommon::set_bind_event(net::SOCKSCBREQUEST cmdbind,event::EVTCBBUF brecv)
{
    if(cmdbind != 0)
        cmdbind_=cmdbind;
    if(brecv != 0)
        brecv_=brecv;
}

void SocksCommon::set_udpassociate_event(net::SOCKSCBREQUEST cmdudpassociate,net::EVTCBRECVFROM udprecv,vmp_size udpmaxsize)
{
    if(cmdudpassociate != 0)
        cmdudpassociate_=cmdudpassociate;
    if(udprecv != 0)
        udprecv_=udprecv;
    if(udpmaxsize != 0)
        udpmaxsize_=udpmaxsize;
}

EventSocks::EventSocks():net::EventConnection()
{
    evt_socks_reset();
}
 
EventSocks::~EventSocks()
{
}

void EventSocks::evt_socks_reset()
{
    common_=0;
    cstatus_=net::evtsocks_status_init;
    ctime_=0.0;
    userid_="";
    permits_=0;
    command_=0x00;
    address_.reset();
}

event::Cell *EventSocks::evt_socks_listen(event::UI *ui,net::Address *local,vmp_uint backlog,net::SocksCommon *common)
{
    vmp::except_check_pointer((void *) common,"evt_socks_listen(common=NULL)");
    common_=common;
    return evt_connection_tcplisten(ui,local,backlog,net::socks_accept,common_->lclose_,net::socks_init,common_->close_);
}

void EventSocks::evt_socks_reply_error(vmp_byte result,vmp::str msg)
{
    event::Manager *manager=cell_->get_manager();
    vmp::Buf send;
    net::Address address;
    address.set("0.0.0.0");
    net::socks5_reply(&send,result,&address);
    if(msg == "")
        msg=net::socks_msg_result(result);
    evt_connection_send(&send);
    manager->cell_close_err_spec(cell_,"net::EventSocks",(vmp_int)result,msg);
}

void EventSocks::evt_socks_reply_connect(net::Address *laddress)
{
    vmp::Buf send;
    recv_=common_->crecv_;
    event::Manager *manager=cell_->get_manager();
    manager->cell_timewait(cell_,ctime_);
    net::socks5_reply(&send,net::socks5_result_ok,laddress);
    evt_connection_send(&send);
    cstatus_=net::evtsocks_status_established;
}

void EventSocks::evt_socks_reply_bind_listen(net::Address *listen)
{
    vmp::except_check_pointer((void *) listen,"evt_socks_reply_bind_listen(listen=NULL)");
    vmp::Buf send;
    net::socks5_reply(&send,net::socks5_result_ok,listen);
    evt_connection_send(&send);
    cstatus_=net::evtsocks_status_bind;
}

void EventSocks::evt_socks_reply_bind_connect(net::Address *connect)
{
    vmp::except_check_pointer((void *) connect,"evt_socks_reply_bind_connect(listen=NULL)");
    vmp::Buf send;
    recv_=common_->brecv_;
    event::Manager *manager=cell_->get_manager();
    manager->cell_timewait(cell_,ctime_);
    net::socks5_reply(&send,net::socks5_result_ok,connect);
    evt_connection_send(&send);
    cstatus_=net::evtsocks_status_established;
}

void EventSocks::evt_socks_reply_udpassociate()
{
    vmp::Buf send;
    event::Manager *manager=cell_->get_manager();
    manager->cell_timewait(cell_,ctime_);
    net::socks5_reply(&send,net::socks5_result_ok,evt_socks_udpbind());
    evt_connection_send(&send);    
    send_=net::connection_send_noimpl;
    sendto_=net::socks_udp_send;
    cstatus_=net::evtsocks_status_established;
}

vmp_byte EventSocks::evt_socks_cstatus()
{
    return cstatus_;
}

vmp::str EventSocks::evt_socks_userid()
{
    return userid_; 
}

vmp_uint EventSocks::evt_socks_permits()
{
    return permits_;
}

vmp_byte EventSocks::evt_socks_command()
{
    return command_;
}

vmp::str EventSocks::evt_socks_command_str()
{
    switch(command_)
    {
        case socks_cd_connect:
            return "connect";
        case socks_cd_bind:
            return "bind";
        case socks_cd_udpassociate:
            return "udpassociate";
    }
    return "undefined";
}

net::Address *EventSocks::evt_socks_address()
{
    return &address_;
}

net::Address *EventSocks::evt_socks_udpbind()
{
    event::Manager *manager=cell_->get_manager();
    event::Cell *sub=manager->cell_searchsub(cell_,"udpbind");
    if(sub == 0)
        vmp::except_s("net::EventSocks::evt_socks_udpbind() non-udp request");
    net::EventConnection *udp=sub->event<net::EventConnection>();
    return udp->evt_connection_localaddr();
}

void EventSocks::evt_socks_close()
{
    cstatus_=net::evtsocks_status_close;
    evt_connection_close();
}
       
void EventSocks::evt_socks_free()
{
    evt_socks_reset();
    evt_connection_free();
}

}}
 
