/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it>
 * Email:<vasta@ragnu.it>
 * Date last update: 28/10/2024
*/
 
#include "net.h"

namespace vampiria { namespace net {

vmp::str conn_strtype(vmp_int conntype)
{
    switch(conntype)
    {
        case net::CONN_NONE:
            return "NONE";
        case net::CONN_UDP4:
            return "UDP4";
        case net::CONN_UDP6:
            return "UDP6";
        case net::CONN_UDPCONNECT:
            return "UDPCONNECT";
        case net::CONN_UDPPROXY:
            return "UDPPROXY";
        case net::CONN_TCPLISTEN:
            return "TCPLISTEN";
        case net::CONN_TCPSERVER:
            return "TCPSERVER";
        case net::CONN_TCPCLIENT:
            return "TCPCLIENT";
        case net::CONN_TCPPROXY:
            return "TCPPROXY";
        case net::CONN_TCPBIND:
            return "TCPBIND";
        case CONN_TCPBINDPROXY:
            return "TCPBINDPROXY";
        default:
            vmp::except("net::EventConnection::evt_connection_strtype() conntype_=%d bad value",conntype);
    }
    return "";
}

EventConnection::EventConnection():event::Event()
{
    parent_=0;
    evt_connection_reset();
}

EventConnection::~EventConnection()
{
}

void EventConnection::evt_connection_reset()
{
    if(parent_ != 0)
    {
        parent_->get_manager()->cell_release(parent_);
        parent_=0;
    }
    conntype_=net::CONN_NONE;
    local_.reset();
    peer_.reset();
    udpmaxsize_=0;
    proxy_=0;
    proxywait_=0;
    proxyindex_=0;
    proxyoper_=0x00;
    proxystatus_="";
    udplastfrag_=0x00;
    
    proxyclose_=event::empty_ev;
    
    send_=net::connection_send_noimpl;
    sendto_=net::connection_send_noimpl;
    
    connect_=event::empty_ev;
    acptevent_=net::empty_accept_ev;
    cchild_=event::empty_ev;
    recv_=event::empty_buf_ev;
    recvfrom_=net::empty_recvfrom_ev;
    
    bind_=event::empty_ev;
    bindaddr_.reset();
    bindservice_.reset();
}

net::Socket EventConnection::evt_connection_udpbindset(vmp_int conntype,net::EVTCBRECVFROM recvfrom,vmp_size udpmaxsize)
{
    vmp_int socket;
    conntype_=conntype;
    if(recvfrom != 0)
        recvfrom_=recvfrom;
    udpmaxsize_=udpmaxsize;
    try
    {
        if(conntype_ == net::CONN_UDP4)
        {   
            local_.set("0.0.0.0");
            peer_.set("0.0.0.0");
            socket=net::socket_datagram(AF_INET,false);
            if(socket == -1)
                vmp::except_errno();
        }
        else
        {
            local_.set("::");
            peer_.set("::");
            socket=net::socket_datagram(AF_INET6,false);
            if(socket == -1)
                vmp::except_errno();
        }
    }
    catch(vmp::exception &x)
    {
        evt_connection_reset();
        vmp::except("net::EventConnection::evt_connection_bindset() %s",x.what());
    }
    return socket;
}

event::Cell *EventConnection::evt_connection_udpbind(event::UI *ui,net::Address *local,vmp_size udpmaxsize,net::EVTCBRECVFROM recvfrom,event::EVTCB close)
{
    vmp_int iptype;
    net::Socket socket;
    vmp::except_check_pointer((void *) local,"evt_connection_udpbind(local=NULL)");
    try
    {
        iptype=local->iptype();     
    }
    catch(vmp::exception &x)
    {
        iptype=PF_UNSPEC;
    }
    if(iptype == PF_UNSPEC)
        vmp::except_s("net::evt_connection_udpbind(local='Invalid address')"); 
    if(iptype == AF_INET)
        socket=evt_connection_udpbindset(net::CONN_UDP4,recvfrom,udpmaxsize);
    else
        socket=evt_connection_udpbindset(net::CONN_UDP6,recvfrom,udpmaxsize);
    try
    {
        local_.copy(local);
        if(net::socket_bind(socket,&local_) == -1)
            vmp::except_errno();
        net::socket_addrlocal(socket,&local_);
        if(local_.is_ipv4())
            sendto_=net::connection_udp_sendto4;
        else if(local_.is_ipv6())
            sendto_=net::connection_udp_sendto6;
    }
    catch(vmp::exception &x)
    {
        net::socket_close(&socket);
        evt_connection_reset();
        vmp::except("net::EventConnection::evt_connection_udpbind() %s",x.what());
    }
    return evt_new(ui,socket,net::udp_read,close);
}

event::Cell *EventConnection::evt_connection_udp4(event::UI *ui,vmp_size udpmaxsize,net::EVTCBRECVFROM recvfrom,event::EVTCB close)
{
    net::Socket socket=evt_connection_udpbindset(net::CONN_UDP4,recvfrom,udpmaxsize);
    sendto_=net::connection_udp_sendto4_getaddr;
    return evt_new(ui,socket,net::udp_read,close);
}

event::Cell *EventConnection::evt_connection_udp6(event::UI *ui,vmp_size udpmaxsize,net::EVTCBRECVFROM recvfrom,event::EVTCB close)
{
    net::Socket socket=evt_connection_udpbindset(net::CONN_UDP6,recvfrom,udpmaxsize);
    sendto_=net::connection_udp_sendto6_getaddr;
    return evt_new(ui,socket,net::udp_read,close);
}

event::Cell *EventConnection::evt_connection_udp(event::UI *ui,net::Address *peer,vmp_size udpmaxsize,event::EVTCB connect,event::EVTCBBUF recv,event::EVTCB close,net::ProxyChain *proxy)
{
    net::Socket socket;
    event::Cell *cell;
    vmp::except_check_pointer((void *) peer,"evt_connection_udp(peer=NULL)");
    if(connect != 0)
        connect_=connect;
    if(recv != 0)
        recv_=recv;
    udpmaxsize_=udpmaxsize;
    proxy_=proxy;
    if((proxy_ == 0) || (proxy_->size() == 0))
    {
        vmp_int iptype;
        try
        {
            iptype=peer->iptype();     
        }
        catch(vmp::exception &x)
        {
            iptype=PF_UNSPEC;
        }
        if((iptype == PF_UNSPEC) || (peer->service() == "0"))
            vmp::except_s("net::evt_connection_udp(peer='Invalid address')");
        try
        {
            peer_.copy(peer);
            if(peer_.is_ipv4())
            {   
                local_.set("0.0.0.0");
                socket=net::socket_datagram(AF_INET,false);
                if(socket == -1)
                    vmp::except_errno();
                send_=net::connection_udp_sendto4_getaddr_s;
            }
            else
            {
                local_.set("::");
                socket=net::socket_datagram(AF_INET6,false);
                if(socket == -1)
                    vmp::except_errno();
                send_=net::connection_udp_sendto6_getaddr_s;
            }
            conntype_=net::CONN_UDPCONNECT;
        }
        catch(vmp::exception &x)
        {
            evt_connection_reset();
            vmp::except("net::EventConnection::evt_connection_udp() %s",x.what());
        }
        cell=evt_new(ui,socket,net::udp_connect_read,close);
    }
    else
    {
        peer_.copy(peer);
        conntype_=net::CONN_UDPPROXY;
        send_=net::connection_tcp_send;
        try
        {
            cell=proxy_->udpassociate(this,ui,close);
        }
        catch(vmp::exception &x)
        {
            evt_connection_reset();
            vmp::except_s(x.what());
        }
    }
    return cell;
}

void EventConnection::evt_connection_udp_connect(event::Cell *cell)
{
    if(conntype_==net::CONN_UDPCONNECT)
        connect_(cell);
}

event::Cell *EventConnection::evt_connection_tcplisten(event::UI *ui,net::Address *local,vmp_uint backlog,net::EVTCBACCEPT acptevent,event::EVTCB svlcevent,event::EVTCBBUF recv,event::EVTCB svcevent)
{
    net::Socket socket;
    vmp::except_check_pointer((void *)local,"evt_connection_tcplisten(local=NULL)");
    if(acptevent != 0)
        acptevent_=acptevent;
    if(recv != 0)
        recv_=recv;
    if(svcevent != 0)
        cchild_=svcevent;
    vmp_int iptype; 
    try
    {
        iptype=local->iptype();
    }
    catch(vmp::exception &x)
    {
        iptype=PF_UNSPEC; 
    }
    if(iptype== PF_UNSPEC)
        vmp::except("evt_connection_tcplisten(local='invalid address')");
    try
    {
        socket=net::socket_stream(iptype,false);
        if(socket == -1)
            vmp::except_errno();
        local_.copy(local);
        net::socket_bind(socket,&local_);
        net::socket_addrlocal(socket,&local_);
        net::socket_listen(socket,backlog);
        conntype_=net::CONN_TCPLISTEN;
        if(local_.is_ipv4())
            peer_.set("0.0.0.0");
        else
            peer_.set("::");
    } 
    catch(vmp::exception &x)
    {
        evt_connection_reset();
        vmp::except("evt_connection_tcplisten %s",x.what());   
    }
    return evt_new(ui,socket,net::connection_tcp_accept,svlcevent);
}

event::Cell *EventConnection::evt_connection_tcpclient(event::UI *ui,net::Address *peer,event::EVTCB connect,event::EVTCBBUF recv,event::EVTCB close,net::ProxyChain *proxy)
{
    net::Socket socket;
    event::Cell *cell;
    vmp::except_check_pointer((void *) peer,"evt_connection_tcpclient(peer=NULL)");
    if(connect != 0)
        connect_=connect;
    if(recv != 0)
        recv_=recv;
    send_=net::connection_tcp_send;
    proxy_=proxy;
    if((proxy_ == 0) || (proxy_->size() == 0))
    {
        vmp_int iptype;
        try
        {
            iptype=peer->iptype();     
        }
        catch(vmp::exception &x)
        {
            iptype=PF_UNSPEC;
        }
        if((iptype == PF_UNSPEC) || (peer->service() == "0"))
            vmp::except_s("net::evt_connection_tcpclient(peer='Invalid aLast UDP fragment received, used for proxiesddress')");
        try
        {
            peer_.copy(peer);
            socket=net::socket_stream(iptype,false);
            if(socket == -1)
                vmp::except_errno();
            if(socket_connect(socket,&peer_) == -1)
                vmp::except_errno();
            conntype_=net::CONN_TCPCLIENT;
        }
        catch(vmp::exception &x)
        {
            evt_connection_reset();
            vmp::except("net::EventConnection::evt_connection_tcpclient() %s",x.what());
        }
        cell=evt_new(ui,socket,net::connection_tcp_connect,close);
        cell->writing_=true;
    }
    else
    {
        peer_.copy(peer);
        conntype_=net::CONN_TCPPROXY;
        try
        {
            cell=proxy_->tcpconnect(this,ui,close);
        }
        catch(vmp::exception &x)
        {
            evt_connection_reset();
            vmp::except_s(x.what());
        }
    }
    return cell;
}

event::Cell *EventConnection::evt_connection_tcpbind(event::UI *ui,net::Address *bindref,net::Address *bindservice,event::EVTCB bind,event::EVTCB connect,event::EVTCBBUF recv,event::EVTCB close,vmp::time::Time bindwait,net::ProxyChain *proxy)
{
    net::Socket socket;
    event::Cell *cell;
    vmp::except_check_pointer((void *)bindref,"evt_connection_tcpbind(bindref=NULL)");
    vmp::except_check_pointer((void *)bindservice,"evt_connection_tcpbind(bindservice=NULL)");
    bindservice_.copy(bindservice);
    event::Manager *manager=ui->manager();
    if(bind != 0)
        bind_=bind;
    if(connect != 0)
        connect_=connect;
    if(recv != 0)
        recv_=recv;
    proxy_=proxy;
    if((proxy_ == 0) || (proxy_->size() == 0))
    {
        vmp_int iptype; 
        try
        {
            iptype=bindref->iptype();
        }
        catch(vmp::exception &x)
        {
            iptype=PF_UNSPEC; 
        }
        if(iptype== PF_UNSPEC)
            vmp::except("evt_connection_tcpbind(ref='invalid address')");
        try
        {
            socket=net::socket_stream(iptype,false);
            if(socket == -1)
                vmp::except_errno();
            bindaddr_.set(bindref->ip(),"0");
            net::socket_bind(socket,&bindaddr_);
            net::socket_addrlocal(socket,&bindaddr_);
            net::socket_listen(socket,1);
            conntype_=net::CONN_TCPBIND;
            if(bindaddr_.is_ipv4())
                peer_.set("0.0.0.0");
            else
                peer_.set("::");
        }
        catch(vmp::exception &x)
        {
            evt_connection_reset();
            vmp::except("evt_connection_tcpbind %s",x.what());   
        }
        cell=evt_new(ui,socket,net::connection_tcp_bind,close);
        manager->cell_timewait(cell,bindwait);
    }
    else
    {
        conntype_=net::CONN_TCPBINDPROXY;
        send_=net::connection_tcp_send;
        try
        {
            cell=proxy_->tcpbind(this,bindref,ui,close);
        }
        catch(vmp::exception &x)
        {
            evt_connection_reset();
            vmp::except_s(x.what());
        }
    }
    return cell;
}

void EventConnection::evt_connection_tcpbind_bind()
{
    if(conntype_==net::CONN_TCPBIND)
        bind_(cell_);
}

void EventConnection::evt_connection_sendTo(vmp::Buf *buf,net::Address *peer,vmp_bool forcesend)
{
    event::Manager *manager=cell_->get_manager();
    if(manager->cell_update(cell_) || forcesend)
        sendto_(cell_,buf,peer);   
}

void EventConnection::evt_connection_send(vmp::Buf *buf,vmp_bool forcesend)
{
    event::Manager *manager=cell_->get_manager();
    if(manager->cell_update(cell_) || forcesend)
        send_(cell_,buf,&peer_);
}

vmp_int EventConnection::evt_connection_socket()
{
    return fd_;
}

event::Cell *EventConnection::evt_connection_parent()
{
    if(conntype_ == net::CONN_TCPSERVER)
        return parent_;
    vmp::except("In Connection type '%s' not parent associated",evt_connection_strtype().c_str());
    return 0;
}

vmp_int EventConnection::evt_connection_type()
{
    return conntype_;
}

vmp_bool EventConnection::evt_connection_isclient()
{
    return (conntype_ == net::CONN_TCPCLIENT) || (conntype_ == net::CONN_TCPPROXY);
}

vmp_bool EventConnection::evt_connection_isserver()
{
    return (conntype_ == net::CONN_TCPSERVER);
}

vmp::str EventConnection::evt_connection_strtype()
{
    return net::conn_strtype(conntype_);
}

net::Address *EventConnection::evt_connection_localaddr()
{
    return &local_;
}

net::Address *EventConnection::evt_connection_peeraddr()
{
    return &peer_;
}

net::Address *EventConnection::evt_connection_bindaddr()
{
    return &bindaddr_;
}

net::Address *EventConnection::evt_connection_bindservice()
{
    return &bindservice_;
}

net::ProxyChain *EventConnection::evt_connection_proxychain()
{
    return proxy_;
}

vmp_size EventConnection::evt_connection_udpmaxsize()
{
    return udpmaxsize_;
}

vmp_byte EventConnection::evt_connection_udplastfrag()
{
    return udplastfrag_;
}

void EventConnection::evt_connection_close()
{
    net::socket_close(&fd_);
    evt_close();
}

void EventConnection::evt_connection_free()
{
    evt_connection_reset();
    evt_free();
}

}}

