/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 30/06/2025
*/

#include "jrp.h"

namespace vampiria { namespace jrp { namespace misc {

Passwd::Passwd(crypto::Ctx_Peer_Web *ctx):jrp::misc::JrpApi_I()
{
    vmp::except_check_pointer((void *) ctx,"jrp::misc::Passwd(ctx=Null)");
    addtype(json::japi("passwd"));
    ctx_=ctx;
}

Passwd::~Passwd()
{
    ctx_=0;
}

void Passwd::register_common(jrp::JrpCommon *common,vmp::str permits)
{
    if(common !=0)
        common->add_reqdata(json::japi("passwd"),{},{},permits);
}

void Passwd::management_impl(jrp::JrpReq *jreq,vmp::Buf *payload)
{
    if(jreq != 0)
    {
        try
        {
            json::JData jdata;
            jdata.set(jreq->jdata_root());
            vmp::str user=jrp::sessionid(jreq->cell());
            vmp::str oldpassword=jdata.get_text("oldpassword");
            vmp::str password=jdata.get_text("password");
            vmp_bool md5_password=jdata.get_bool("md5_password");
            if(ctx_->verify_acl_user(user,oldpassword,true,md5_password) != 0)
            {
                ctx_->change_password(user,password,md5_password);
                jreq->close(jrp::status_ok);
            }
            else
                jreq->close(jrp::status_accessdenied);
        }
        catch(vmp::exception &x)
        {
            jreq->close(jrp::status_input_bad);
        }
    }
}

User::User(crypto::Ctx_Peer_Web *ctx):jrp::misc::JrpApi_I()
{
    vmp::except_check_pointer((void *) ctx,"jrp::misc::User(ctx=Null)");
    addtype(json::japi("userlist"));
    addtype(json::japi("useradd"));
    addtype(json::japi("usermod"));
    addtype(json::japi("userpasswd"));
    addtype(json::japi("userdel"));
    ctx_=ctx;
}

User::~User()
{
    ctx_=0;
}

void User::register_common(jrp::JrpCommon *common,vmp::str permits)
{
    if(common !=0)
    {
        common->add_reqdata(json::japi("userlist"),{},{json::japi("userinfo")},permits);
        common->add_reqdata(json::japi("useradd"),{},{json::japi("userinfo")},permits);
        common->add_reqdata(json::japi("usermod"),{},{json::japi("userinfo")},permits);
        common->add_reqdata(json::japi("userpasswd"),{},{json::japi("userinfo")},permits);
        common->add_reqdata(json::japi("userdel"),{},{json::japi("userinfo")},permits);
    }
}

void User::management_impl(jrp::JrpReq *jreq,vmp::Buf *payload)
{
    if(jreq != 0)
    {
        vmp::str jtype=jreq->jdata_type();
        json::JData jdata;
        vmp_int ret=jrp::status_ok;
        vmp::str retvalue="";
        try
        {
            if(jtype == json::japi("useradd"))
            {
                jdata.set(jreq->jdata_root());
                vmp::str user=jdata.get_text("user");
                vmp::str password=jdata.get_text("password");
                vmp_uint permits=jdata.get_integer_range("permits",0,vmp::INTMAX);
                vmp_bool only_localhost=jdata.get_bool("only_localhost");
                vmp_bool md5_password=jdata.get_bool("md5_password");
                vmp::vector<vmp::str> users=ctx_->list_acl_users();
                if(vmp::invector<vmp::str>(user,users))
                {
                    ret=jrp::status_err;
                    retvalue="User already in the system";
                }
                else if(permits > sessionpermits(jreq->cell()))
                {
                    ret=jrp::status_err;
                    retvalue="User permissions too high";
                }    
                else
                    ctx_->add_acl_user(user,password,permits,only_localhost,md5_password);
            }
            else if(jtype == json::japi("usermod"))
            {
                jdata.set(jreq->jdata_root());
                vmp::str user=jdata.get_text("user");
                vmp_uint permits=jdata.get_integer_range("permits",0,vmp::INTMAX);
                vmp_bool only_localhost=jdata.get_bool("only_localhost");
                vmp::vector<vmp::str> users=ctx_->list_acl_users();
                if(!vmp::invector<vmp::str>(user,users))
                {
                    ret=jrp::status_err;
                    retvalue="User does not exist";
                }
                else if(permits > sessionpermits(jreq->cell()))
                {
                    ret=jrp::status_err;
                    retvalue="User permissions too high";
                }
                else
                    ctx_->change_permits(user,permits,only_localhost);
            }
            else if(jtype == json::japi("userpasswd"))
            {
                jdata.set(jreq->jdata_root());
                vmp::str user=jdata.get_text("user");
                vmp::str password=jdata.get_text("password");
                vmp_bool md5_password=jdata.get_bool("md5_password");
                vmp::vector<vmp::str> users=ctx_->list_acl_users();
                if(!vmp::invector<vmp::str>(user,users))
                {
                    ret=jrp::status_err;
                    retvalue="User does not exist";
                }
                else
                {
                    vmp_uint permits=ctx_->get_permits(user);
                    if(permits > sessionpermits(jreq->cell()))
                    {
                        ret=jrp::status_err;
                        retvalue="User permissions too high";
                    }
                    else
                        ctx_->change_password(user,password,md5_password);
                }
            }
            else if(jtype == json::japi("userdel"))
            {
                jdata.set(jreq->jdata_root());
                vmp::str user=jdata.get_text("user");
                vmp::vector<vmp::str> users=ctx_->list_acl_users();
                if(!vmp::invector<vmp::str>(user,users))
                {
                    ret=jrp::status_err;
                    retvalue="User does not exist";
                }
                else
                {
                    vmp_uint permits=ctx_->get_permits(user);
                    if(permits > sessionpermits(jreq->cell()))
                    {
                        ret=jrp::status_err;
                        retvalue="User permissions too high";
                    }
                    else
                        ctx_->del_acl_user(user);
                }
            }
        }
        catch(vmp::exception &x)
        {
            ret=jrp::status_input_bad;
        }
        vmp::vector<vmp::str> users=ctx_->list_acl_users();
        json::Json json;
        vmp_uint permits;
        vmp_bool only_localhost;
        for(vmp_index i=0;i<users.size();i++)
        {
            permits=ctx_->get_permits(users[i]);
            only_localhost=ctx_->get_only_localhost(users[i]);
            json::japi_userinfo(json.json_new(),users[i],permits,only_localhost);
            jreq->response(json.root());
        }
        jreq->close(ret,retvalue);
    }
}

Peer::Peer(crypto::Ctx_Peer *ctx):jrp::misc::JrpApi_I()
{
    vmp::except_check_pointer((void *) ctx,"jrp::misc::Peer(ctx=Null)");
    ctx_=ctx;
    addtype(json::japi("peerlist"));
    addtype(json::japi("peeradd"));
    addtype(json::japi("peermod"));
    addtype(json::japi("peerdel"));
}
        
Peer::~Peer()
{
    ctx_=0;
}
        
void Peer::register_common(jrp::JrpCommon *common,vmp::str permits)
{
    if(common !=0)
    {
        common->add_reqdata(json::japi("peerlist"),{},{json::japi("peerinfo")},permits);
        common->add_reqdata(json::japi("peeradd"),{},{json::japi("peerinfo")},permits);
        common->add_reqdata(json::japi("peermod"),{},{json::japi("peerinfo")},permits);
        common->add_reqdata(json::japi("peerdel"),{},{json::japi("peerinfo")},permits);
    }
}
        
void Peer::management_impl(jrp::JrpReq *jreq,vmp::Buf *payload)
{
    if(jreq != 0)
    {
        vmp::str jtype=jreq->jdata_type();
        json::JData jdata;
        vmp_int ret=jrp::status_ok;
        vmp::str retvalue="";
        try
        {
            if(jtype == json::japi("peeradd"))
            {
                jdata.set(jreq->jdata_root());
                vmp::str fng=jdata.get_text("fingerprint");
                vmp::str subject=jdata.get_text("subject");
                vmp_uint permits=jdata.get_integer_range("permits",0,vmp::INTMAX);
                vmp::vector<vmp::str> fngs=ctx_->fingerprints_acl_peer();
                if(vmp::invector<vmp::str>(fng,fngs))
                {
                    ret=jrp::status_err;
                    retvalue="Fingerptint already in the system";
                }
                else if(permits > sessionpermits(jreq->cell()))
                {
                    ret=jrp::status_err;
                    retvalue="Peer permissions too high";
                }
                else
                    ctx_->add_acl_peer(fng,subject,permits);
            }
            else if(jtype == json::japi("peermod"))
            {
                jdata.set(jreq->jdata_root());
                vmp::str fng=jdata.get_text("fingerprint");
                vmp_uint permits=jdata.get_integer_range("permits",0,vmp::INTMAX);
                vmp::vector<vmp::str> fngs=ctx_->fingerprints_acl_peer();
                if(!vmp::invector<vmp::str>(fng,fngs))
                {
                    ret=jrp::status_err;
                    retvalue="Fingerprint does not exist";
                }
                else if(permits > sessionpermits(jreq->cell()))
                {
                    ret=jrp::status_err;
                    retvalue="Peer permissions too high";
                }
                else
                    ctx_->change_permits(fng,permits);
            }
            else if(jtype == json::japi("peerdel"))
            {
                jdata.set(jreq->jdata_root());
                vmp::str fng=jdata.get_text("fingerprint");
                vmp::vector<vmp::str> fngs=ctx_->fingerprints_acl_peer();
                if(!vmp::invector<vmp::str>(fng,fngs))
                {
                    ret=jrp::status_err;
                    retvalue="Fingerprint does not exist";
                }
                else
                {
                    vmp_uint permits=ctx_->verify_acl_peer(fng);
                    if(permits > sessionpermits(jreq->cell()))
                    {
                        ret=jrp::status_err;
                        retvalue="User permissions too high";
                    }
                    else
                        ctx_->del_acl_peer(fng);
                }
            } 
        }
        catch(vmp::exception &x)
        {
            ret=jrp::status_input_bad;
        }
        vmp::vector<vmp::str> fngs=ctx_->fingerprints_acl_peer();
        json::Json json;
        vmp::str subject;
        vmp_uint permits;
        for(vmp_index i=0;i<fngs.size();i++)
        {
            subject=ctx_->get_acl_subject(fngs[i]);
            permits=ctx_->verify_acl_peer(fngs[i]);
            json::japi_peerinfo(json.json_new(),fngs[i],subject,permits);
            jreq->response(json.root());
        }
        jreq->close(ret,retvalue);
    }    
}
ConnectStatus::ConnectStatus(jrp::JrpUI *jrpui,jrp::WssJrpUI *wssjrpui):jrp::misc::JrpApi_I()
{
    jrpui_=jrpui;
    wssjrpui_=wssjrpui;
    addtype(json::japi("connectlist"));
    addtype(json::japi("connectclose"));
}
        
ConnectStatus::~ConnectStatus()
{
    jrpui_=0;
    wssjrpui_=0;
}

void ConnectStatus::register_common(jrp::JrpCommon *common,vmp::str permits)
{
    if(common !=0)
    {
        common->add_reqdata(json::japi("connectlist"),{},{json::japi("connectinfo")},permits);
        common->add_reqdata(json::japi("connectclose"),{},{json::japi("connectinfo")},permits);
    }
}

void ConnectStatus::management_impl(jrp::JrpReq *jreq,vmp::Buf *payload)
{
    if(jreq != 0)
    {
        vmp::str jtype=jreq->jdata_type();
        json::JData jdata;
        vmp_int ret=jrp::status_ok;
        vmp::str retvalue="";
        event::Cell *cell;
        try
        {
            if(jtype == json::japi("connectclose"))
            {
                jdata.set(jreq->jdata_root());
                vmp::str id=jdata.get_text("id");
                vmp::str type=jdata.get_text("type");
                if(type == "p2p")
                {
                    if(jrpui_ != 0)
                    {
                        cell=jrpui_->connect_search(id);
                        if(cell == 0)
                            vmp::except("id field '%s' not found",id.c_str());
                        crypto::ssl_shutdown(cell,"jrp::ConnectStatus");
                    }
                }
                else if(type == "wss")
                {
                    if(wssjrpui_ != 0)
                    {
                        cell=wssjrpui_->connect_search(id);
                        if(cell == 0)
                            vmp::except("id field '%s' not found",id.c_str());
                        crypto::ssl_shutdown(cell,"jrp::ConnectStatus");
                    }
                }
                else
                    vmp::except("text field '%s' bad value",type.c_str());
            }
        }
        catch(vmp::exception &x)
        {
            retvalue=x.what();
            ret=jrp::status_input_bad;
        }
        json::Json json;
        net::Address *addr;
        if(jrpui_ != 0)
        {
            vmp::vector<vmp::str> ids=jrpui_->connect_list();
            for(vmp_index i=0;i<ids.size();i++)
            {
                cell=jrpui_->connect_search(ids[i]);
                if(cell->is_active())
                {
                    addr=jrpui_->peeraddr(cell);
                    json::japi_connectinfo(json.json_new(),ids[i],"p2p",jrpui_->peer_subject(cell),jrpui_->peer_permits(cell),addr->host(),vmp::unicode::str_todigit(addr->service()));
                    jreq->response(json.root());
                }
            }
        }
        if(wssjrpui_ != 0)
        {
            vmp::vector<vmp::str> ids=wssjrpui_->connect_list();
            for(vmp_index i=0;i<ids.size();i++)
            {
                cell=wssjrpui_->connect_search(ids[i]);
                if(cell->is_active())
                {
                    addr=wssjrpui_->peeraddr(cell);   json::japi_connectinfo(json.json_new(),ids[i],"wss","",wssjrpui_->permits(cell),addr->host(),vmp::unicode::str_todigit(addr->service()));
                    jreq->response(json.root());
                }
            }
        }
        jreq->close(ret,retvalue); 
    }
}

Logs::Logs()
{
    addtype(json::japi("logs"));
    addtype(json::japi("logread"));
}

Logs::~Logs()
{
}

void Logs::set_logdir(vmp::str key,vmp::str dirpath)
{
    dirs_.update(key,&dirpath);
}

void Logs::register_common(jrp::JrpCommon *common,vmp::str permits)
{
    if(common !=0)
    {
        common->add_reqdata(json::japi("logs"),{},{json::japi("logfiles")},permits);
        common->add_reqdata(json::japi("logread"),{},{json::japi("logview")},permits);
    }
}

void Logs::management_impl(jrp::JrpReq *jreq,vmp::Buf *payload)
{
    if(jreq != 0)
    {
        vmp::str jtype=jreq->jdata_type();
        json::JData jdata;
        vmp_int ret=jrp::status_ok;
        vmp::str retvalue="";
        json::Json json;
        if(jtype == json::japi("logs"))
        {
            vmp::vector<vmp::str> keys=dirs_.all_keys(),files,listen;
            vmp::str dirpath,preamble,file;
            for(vmp_index i=0;i<keys.size();i++)
            {
                dirs_.search(keys[i],&dirpath);
                try
                {
                    listen=vmp::sort<vmp::str>(vmp::fs::listendir(dirpath));
                    if(keys[i] == "")
                        preamble="";
                    else
                        vmp::unicode::str_write(&preamble,"%s/",keys[i].c_str());
                    for(vmp_index j=0;j<listen.size();j++)
                    {
                        file=vmp::fs::union_path(dirpath,listen[j]);
                        if(vmp::fs::isrfile(file) && vmp::unicode::str_regex_matching(listen[j],".*\\.log(\\.[0-9]*)?"))
                        {
                            vmp::unicode::str_write(&file,"%s%s",preamble.c_str(),listen[j].c_str());
                            files.push_back(file);   
                        }
                    }
                }
                catch(vmp::exception &x)
                {
                }
            }
            json::japi_logfiles(json.root(),files);
            jreq->response(json.root());
        }
        else
        {
            try
            {
                jdata.set(jreq->jdata_root());
                vmp::str logname=jdata.get_text("logname");
                vmp::str key,dirpath="",file="";
                vmp::vector<vmp::str> split=vmp::unicode::str_split(logname,"/");
                if(split.size() == 1)
                {
                    if(split[0] == "" || !dirs_.search("",&dirpath))
                        ret=jrp::status_resource_notfound;
                    else
                        file=split[0];   
                }
                else if(split.size() == 2)
                {
                    if(!dirs_.search(split[0],&dirpath))
                        ret=jrp::status_resource_notfound;
                    else
                        file=split[1];
                }
                else
                    ret=jrp::status_resource_notfound;
                if(dirpath != "")
                {
                    vmp::unicode::str_write(&file,"%s/%s",dirpath.c_str(),file.c_str());
                    if(vmp::fs::isrfile(file))
                    {
                        file=vmp::input_file_s(file);
                        vmp_size s=file.size();
                        if(s > 1000000)
                        {
                            file=vmp::unicode::str_sub(file,s-1000000,1000000);
                            vmp::unicode::str_write(&file,"(... truncated)\n%s",file.c_str());
                        }
                        json::japi_logview(json.root(),logname,file);
                        jreq->response(json.root());
                    }
                    else
                        ret=jrp::status_resource_notfound;
                }
            }
            catch(vmp::exception &x)
            {
                retvalue=x.what();
                ret=jrp::status_input_bad;
            }
        }
        jreq->close(ret,retvalue);
    }
}

}}}

