/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 02/07/2025
*/

#ifndef VAMPIRIA_JRP_MISC_STATUS_H

#define VAMPIRIA_JRP_MISC_STATUS_H 1

namespace vampiria { namespace jrp { namespace misc {

//!  A class used to manage the state of the dynamic system managed by a process.
//!  These are a json data table representing state variables
class Status:public jrp::misc::JrpApi_I
{
    private:
        vmp::thread::Mutex mutex_;/*!<Internal usage mutual exclusion */
        json::Json root_;/*!<State memory*/
        json::JTable jtable_;/*!<Internal usage*/
        vmp::Table<vmp::str,jrp::JrpReq *> jreqs_;/*!<Json request table to send status update to*/
        vmp::Table<vmp::str,vmp::vector<vmp_index> >limits_;/*!<Restrict the permissions for keys(regular expression) to send to users*/
        vmp::str dirpath_;/*!<Directory used to store status files*/
        vmp::str filecache_;/*!<Internal usage*/
        
        //!Internal usage
        void cache();
        
        //!Internal usage
        vmp_bool islimit(vmp::vector<vmp::str> accepted,jrp::JrpReq *jreq);
        
        //!Internal usage
        void broadcast(json::JsonObj *jdata,vmp::str jkey);
    public:
        //! A Constructor
        /*!
            @param dirpath Directory used to store status files
            @param cached if set to true on startup restores state saved in cache file, otherwise initializes an empty state
            @return except in case of failure
        */
        Status(vmp::str dirpath,vmp_bool cached=true);
        
        //!A Destructor
        ~Status();
        
        //! Return the root json parser
        /*!
            @ref root_
            @return pointer json parser
            
        */
        json::Json *root();
        
        //!Restrict the permissions for keys to send to users
        /*!
            @ref limits_
            @param jkey regular expression of the jdata key
            @param permits accepted permission range
        */
        void set_limit(vmp::str jkey,vmp::str permits);
        
        //! Writes the State to a backup file
        void backup();
        
        //! Return table keys that have the jtype variable
        /*!
            @param jtype datatype, if == "" all the keys are returned
            @return outputs keys or except in case of failure
        */
        vmp::vector<vmp::str> keys(vmp::str jtype="");

        //!Update or insert the value of a status variable
        /*!
            @param jkey key of the variable
            @param jdata json data variable
            @return void or except in case of failure
        */
        void update(vmp::str jkey,json::JsonObj *jdata);

        //!Update or insert the value of a status variable
        /*!
            @param jstatus a json data of type json::japi("statusupdate")
            @return void or except in case of failure
        */
        void update_from_japi(json::JsonObj *jstatus);

        //! Delete a variable from the state
        /*!
            @param jkey key of the variable
        */
        void cancel(vmp::str jkey);

        //! Delete a variable from the state
        /*!
            @param jstatus a json data of type json::japi("statuscancel")
            @return void or except in case of failure
        */
        void cancel_from_japi(json::JsonObj *jstatus);

        //! Search for a status variable
        /*!
            @param jkey key of the variable
            @param jdata output data
            @return True if the variable exists, otherwise false or except in case of failure

        */
        vmp_bool search(vmp::str jkey,json::JsonObj *jdata);

        //! Open an external monitoring request
        /*!
            @param jreq jrp request associated
            @return void or except in case of failure
        */
        void request_open(jrp::JrpReq *jreq);
        
        //! Close an external monitoring request(send jrp close to associated peer)
        /*!
            @param jreqkey Key associated with the request to be closed
            @param status closing state 
            @param msg closing state in string format
            @return void or except in case of failure
        */
        void request_close(vmp::str jreqkey,vmp_int status,vmp::str msg="");
        
        //! virtual function
        void register_common(jrp::JrpCommon *common,vmp::str permits="*");
        
        //! virtual function
        void management_impl(jrp::JrpReq *jreq,vmp::Buf *payload);
        
        //! virtual function
        void management_kill_impl(jrp::JrpReq *jreq);
};

//!Manages data sent in response to a status request json::japi("statusupdate") and json::japi("statuscancel")
class StatusGet
{
    private:
        vmp::str key_;/*!<Key of the updated  or cancel object*/
        json::JsonObj data_;/*!Value of the updated object*/
    public:
        //!A constructor
        StatusGet();
        
        //!A destructor
        ~StatusGet();
        
        //! Reset data
        void reset();
        
        //! Returns key of the updated  or cancel object
        /*!
            @sa key_
            @return key data
        */
        vmp::str key();
        
        //! The value of the updated object
        /*!
            @sa data_
            @return the value of the updated object
        */
        json::JsonObj *data();
        
        //! Check if the input request and type json::japi("statusupdate") and the key matching the keyregex in input.If the verification is ok the data are stored
        /*!
            @param keyregex regular expression to verify the key
            @param jdata json data in input
            @return true if the verification is ok,otherwise false
        */
        vmp_bool management_update(vmp::str keyregex,jrp::JrpReq *jreq,json::JsonObj *jdata);
        
        //! Check if the input request and type json::japi("statuscancel") and the key matching the keyregex in input.If the verification is ok the data are stored
        /*!
            @param keyregex regular expression to verify the key
            @param jreq jrp request associated
            @param jdata json data in input
            @return true if the verification is ok,otherwise false
        */
        vmp_bool management_cancel(vmp::str keyregex,jrp::JrpReq *jreq,json::JsonObj *jdata);
};

}}}

#endif

