/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 02/07/2025
*/

#include "jrp.h"

namespace vampiria { namespace jrp { namespace misc {

Status::Status(vmp::str dirpath,vmp_bool cached):jrp::misc::JrpApi_I()
{
    try
    {
        addtype(json::japi("status"));
        json::JData jdata;
        dirpath_=dirpath;
        filecache_=vmp::fs::union_path(dirpath_,"cache.json");
        if(!vmp::fs::isdir(dirpath_))
            vmp::fs::mkdir_wrap(dirpath_,0700);
        if(cached)
        {
            if(vmp::fs::isfile(filecache_))
            {
                try
                {
                    root_.parse_from_file(filecache_);
                    jdata.set(root_.root());
                    if(jdata.jtype() == json::japi_type_table())
                        jdata.get_table("table",&jtable_);
                    else
                        cached=false;
                }
                catch(vmp::exception &x)
                {
                    
                    cached=false;
                }
            }
            else
                cached=false;
        }
        if(!cached)
        {
            json::japi_table(root_.json_new());
            jdata.set(root_.root());
            jdata.get_table("table",&jtable_);
            cache();
        }
    }
    catch(vmp::exception &x)
    {
        vmp::except("jrp::misc::Status() %s",x.what());
    }
    
}

Status::~Status()
{
     root_.json_new();
     dirpath_="";
     filecache_="";
     vmp::vector<jrp::JrpReq *> data=jreqs_.all_data();
     for(vmp_index i=0;i<data.size();i++)
     {   
        data[i]->release();
        data[i]->close(jrp::status_killed);
     }
     jreqs_.clear();
     limits_.clear();
}

void Status::cache()
{
    root_.json_to_file(filecache_);
    vmp::fs::chmod_wrap(filecache_,0600);
}

vmp_bool Status::islimit(vmp::vector<vmp::str> accepted,jrp::JrpReq *jreq)
{
    vmp::vector<vmp_index> plist;
    for(vmp_index i=0;i<accepted.size();i++)
    {
        limits_.search(accepted[i],&plist);
        if(!vmp::invector<vmp_index>(sessionpermits(jreq->cell()),plist))
            return true;
    }
    return false;
}

void Status::broadcast(json::JsonObj *jdata,vmp::str jkey)
{
    cache();
    vmp::vector<jrp::JrpReq *> data=jreqs_.all_data();
    vmp::vector<vmp::str> limits=limits_.all_keys(),accepted;
    for(vmp_index i=0;i<limits.size();i++)
    {
        if(vmp::unicode::str_regex_matching(jkey,limits[i]))
            accepted.push_back(limits[i]);
    }
    for(vmp_index i=0;i<data.size();i++)
    {    
        try
        {
            if(!islimit(accepted,data[i]))
                data[i]->response(jdata);
        }
        catch(vmp::exception &x)
        {
        }
    }
}

json::Json *Status::root()
{
    return &root_;
}

void Status::set_limit(vmp::str jkey,vmp::str permits)
{
    vmp::vector<vmp_index> plist=vmp::unicode::str_toindex_list(permits,jrp::min_permits,jrp::max_permits);
    limits_.update(jkey,&plist);
}

void Status::backup()
{
    mutex_.lock();
    vmp::time::Localtime lt;
    vmp::str tmp,year,month,day,filename;
    vmp::unicode::str_write(&tmp,"%04u",lt.year());
    year=vmp::fs::union_path(dirpath_,tmp);
    month=vmp::fs::union_path(year,lt.month());
    vmp::unicode::str_write(&tmp,"%02u",lt.mday());
    day=vmp::fs::union_path(month,tmp);
    try
    {
        if(!vmp::fs::isdir(year))
            vmp::fs::mkdir_wrap(year,0700);
        if(!vmp::fs::isdir(month))
            vmp::fs::mkdir_wrap(month,0700);
        if(!vmp::fs::isdir(day))
            vmp::fs::mkdir_wrap(day,0700);
        vmp::unicode::str_write(&tmp,"%02u:%02u:%02u.json",lt.hour(),lt.min(),lt.sec());
        filename=vmp::fs::union_path(day,tmp);
        root_.json_to_file(filename);
        vmp::fs::chmod_wrap(filename,0600);
    }
    catch(vmp::exception &x)
    {
        mutex_.unlock();
        vmp::except("jrp::misc::Status::backup() %s",x.what());
    }
    mutex_.unlock();
}

vmp::vector<vmp::str> Status::keys(vmp::str jtype)
{
    vmp::vector<vmp::str> tmp,ret;
    json::JsonObj jobj;
    json::JData jdata;
    mutex_.lock();
    try
    {
        if(jtype == "")
            ret=jtable_.keys();
        else
        {
            tmp=jtable_.keys();
            for(vmp_index i=0;i<tmp.size();i++)
            {
                jtable_.search(tmp[i],&jobj);
                jdata.set(&jobj);
                if(jdata.jtype() == jtype)
                   ret.push_back(tmp[i]);
            }
        }
    }
    catch(vmp::exception &x)
    {
        mutex_.unlock();
        vmp::except("jrp::misc::Status::keys() %s",x.what());
    }
    mutex_.unlock();
    return ret;
}

void Status::update(vmp::str jkey,json::JsonObj *jdata)
{
    mutex_.lock();
    try
    {
        jtable_.insert(jkey,jdata);
        json::Json json;
        json::japi_status_update(json.root(),jkey,jdata);
        broadcast(json.root(),jkey);
    }
    catch(vmp::exception &x)
    {
        mutex_.unlock();
        vmp::except("jrp::misc::Status::update() %s",x.what());
    }
    mutex_.unlock();
}

void Status::update_from_japi(json::JsonObj *jstatus)
{
    vmp::except_check_pointer((void *) jstatus,"jrp::misc::Status::update_from_japi(jstatus=Null)");
    mutex_.lock();
    try
    {
        json::JData jdata;
        jdata.set(jstatus);
        if(jdata.jtype() != json::japi("statusupdate"))
            vmp::except("");
        json::JsonObj objdata;
        jdata.get_subtype("data",&objdata);
        vmp::str jkey=jdata.get_text("key");
        jtable_.insert(jkey,&objdata);
        broadcast(jstatus,jkey);
    }
    catch(vmp::exception &x)
    {
        mutex_.unlock();
        vmp::except("jrp::misc::Status::update_from_japi(jstatus='Bad json data object')");    
    }
    mutex_.unlock();
}

void Status::cancel(vmp::str jkey)
{
    mutex_.lock();
    try
    {
        if(jtable_.cancel(jkey))
        {
            json::Json json;
            json::japi_status_cancel(json.root(),jkey);
            broadcast(json.root(),jkey);
        }
    }
    catch(vmp::exception &x)
    {
        mutex_.unlock();
        vmp::except("jrp::misc::Status::cancel() %s",x.what());
    }
    mutex_.unlock();
}

void Status::cancel_from_japi(json::JsonObj *jstatus)
{
    vmp::except_check_pointer((void *) jstatus,"jrp::misc::Status::cancel_from_japi(jstatus=Null)");
    mutex_.lock();
    try
    {
        json::JsonObj jdata;
        json::JData jd;
        jd.set(jstatus);
        if(jd.jtype() != json::japi("statuscancel"))
            vmp::except("");
        vmp::str jkey=jd.get_text("key");
        if(jtable_.cancel(jkey))
            broadcast(jstatus,jkey);
    }
    catch(vmp::exception &x)
    {
        mutex_.unlock();
        vmp::except("jrp::misc::Status::update_from_japi(jstatus='Bad json data object')");    
    }
    mutex_.unlock();
}

vmp_bool Status::search(vmp::str jkey,json::JsonObj *jdata)
{
    vmp::except_check_pointer((void *) jdata,"jrp::misc::Status::search(jdata=Null)");
    vmp_bool ret=false;
    mutex_.lock();
    ret=jtable_.search(jkey,jdata);
    mutex_.unlock();
    return ret;
}

void Status::request_open(jrp::JrpReq *jreq)
{
    mutex_.lock();
    vmp::except_check_pointer((void *) jreq,"jrp::misc::Status::request_open(jreq=Null)");
    try
    {
        jreqs_.insert(jreq->key(),jreq);
        jreq->alloc();
    }
    catch(vmp::exception &x)
    {
        mutex_.unlock();
        vmp::except("jrp::misc::Status:request_open() %s",x.what());
    }
    try
    {    
        vmp::vector<vmp::str> keys=jtable_.keys();
        json::Json list,tmp; 
        json::JsonObj array,obj;
        list.root()->add_object_array("",&array);
        vmp_index j=0;
        vmp::vector<vmp::str> limits=limits_.all_keys(),accepted;
        for(vmp_index i=0;i<keys.size();i++)
        {
            accepted.clear();
            for(vmp_index j=0;j<limits.size();j++)
            {
                if(vmp::unicode::str_regex_matching(keys[i],limits[j]))
                    accepted.push_back(limits[j]);
            }
            if(!islimit(accepted,jreq))
            {
                jtable_.search(keys[i],&obj);
                json::japi_status_update(tmp.json_new(),keys[i],&obj);
                array.push_array_obj_ex(tmp.root());
                if(j == 9)
                {
                    jreq->response(&array);
                    list.json_new()->add_object_array("",&array);
                    j=0;
                }
                else
                    j++;
            }
        }
        if(j != 0)
            jreq->response(&array);
    }
    catch(vmp::exception &x)
    {
        jrp::JrpReq *tmp;
        jreqs_.cancel(jreq->key(),&tmp);
        tmp->release();
        mutex_.unlock();
        vmp::except("jrp::misc::Status:request_open() %s",x.what());
    }
    mutex_.unlock();
}

//send close
void Status::request_close(vmp::str jreqkey,vmp_int status,vmp::str msg)
{
    mutex_.lock();
    try
    {
        jrp::JrpReq *tmp;
        jreqs_.cancel(jreqkey,&tmp);
        tmp->release();
        tmp->close(status,msg);
    }
    catch(vmp::exception &x)
    {
    }
    mutex_.unlock();
}

void Status::register_common(jrp::JrpCommon *common,vmp::str permits)
{
    if(common !=0)
        common->add_reqdata(json::japi("status"),{},{json::japi("statusupdate"),json::japi("statuscancel")},permits);
}

void Status::management_impl(jrp::JrpReq *jreq,vmp::Buf *payload)
{
    try
    {
        request_open(jreq);
    }
    catch(vmp::exception &x)
    {
        jreq->close(jrp::status_err,x.what());
    }
}

void Status::management_kill_impl(jrp::JrpReq *jreq)
{
    request_close(jreq->key(),jrp::status_killed);
}

StatusGet::StatusGet()
{
    reset();
}

StatusGet::~StatusGet()
{
    reset();
}

void StatusGet::reset()
{
    key_="";
    data_.reset();
}

vmp::str StatusGet::key()
{
    return key_;
}

json::JsonObj *StatusGet::data()
{
    return &data_;
}

vmp_bool StatusGet::management_update(vmp::str keyregex,jrp::JrpReq *jreq,json::JsonObj *jdata)
{
   reset();
   if((jreq != 0) && (jdata != 0) && (jreq->jdata_type() == json::japi("status")))
   {
       try
       {
           json::JData jd;
           jd.set(jdata);
           if(jd.jtype() == json::japi("statusupdate"))
           {
               key_=jd.get_text("key");
               if(vmp::unicode::str_regex_matching(key_,keyregex))
               {
                   jd.get_subtype("data",&data_);
                   return true;
               }
           }
           reset();
       }
       catch(vmp::exception &x)
       {
           reset();
       }
   }
   return false;
}

vmp_bool StatusGet::management_cancel(vmp::str keyregex,jrp::JrpReq *jreq,json::JsonObj *jdata)
{
   reset();
   if((jreq != 0) && (jdata != 0) && (jreq->jdata_type() == json::japi("status")))
   {
       try
       {
           json::JData jd;
           jd.set(jdata);
           if(jd.jtype() == json::japi("statuscancel"))
           {
               key_=jd.get_text("key");
               if(vmp::unicode::str_regex_matching(key_,keyregex))
                   return true;
           }
           reset();
       }
       catch(vmp::exception &x)
       {
           reset();
       }
   }
   return false;
}

}}}

