/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 22/07/2025
*/

#include "jrp.h"

namespace vampiria { namespace jrp { namespace misc {

void ping_event(event::Cell *cell)
{
    jrp::misc::Onion *onion=cell->getvar<jrp::misc::Onion>("onion");
    vmp_index *cid=cell->getvar<vmp_index>("cid");
    json::Json json;
    vmp::Buf payload;
    jrp::misc::japi_orping(json.root());
    vmp_size s=(vmp::rand_wrap((vmp::time::time_wrap()%256))%192)+64;
    payload.random_bytes(s);
    onion->update_keys((*cid));
    onion->channel_send((*cid),json.root(),&payload);
}

void cping_event(event::Cell *cell)
{
    cell->remvar<jrp::misc::Onion>("onion");
    delete cell->remvar<vmp_index>("cid");
}

void action_event(event::Cell *cell)
{
    jrp::misc::Onion *onion=cell->getvar<jrp::misc::Onion>("onion");
    jrp::misc::OrChannel *channel=cell->getvar<jrp::misc::OrChannel >("channel");
    onion->mutex_.lock();
    jrp::misc::ONIONCB cb=channel->actioncb_;
    vmp_index cid=channel->cid_;
    onion->mutex_.unlock();
    onion->channel_action_disable(cid);
    cb(onion,cid);
}

void caction_event(event::Cell *cell)
{
    cell->remvar<jrp::misc::Onion>("onion");
    cell->remvar<jrp::misc::OrChannel>("channel");
    
}

Onion::Onion()
{
}

Onion::Onion(jrp::JrpUI *ui)
{
    init_onion(ui);
    actionui_->set_event(jrp::misc::action_event,jrp::misc::caction_event);
}

Onion::~Onion()
{
    destroy_onion();    
}

void Onion::init_onion(jrp::JrpUI *ui)
{
    vmp::except_check_pointer((void *) ui,"jrp::misc::Onion(ui=Null)");
    ui_=ui;
    cindex_=1;
    status_=0;
    pingui_=new event::TimerUI(ui_->manager());
    pingui_->set_event(jrp::misc::ping_event,jrp::misc::cping_event);
    actionui_=new event::TimerUI(ui_->manager());
    
    optimeout_=10;
    ping_=60;
    keyexpired_=7200;
    maxchannel_=100;
    set_source_cb(0,0,0,0);
    set_channel_cb(0,0,0,0,0);
    set_target_cb(0,0,0,0);
    
    addtype(jrp::misc::japi("orchannel"));
    addtype(jrp::misc::japi("orchannelclose"));
}

void Onion::destroy_onion()
{
    ui_=0;
    status_=0;
}

void Onion::active_ping(jrp::misc::OrChannel *channel)
{
    channel->pingevt_=pingui_->new_timer("ping",ping_); 
    channel->pingevt_->alloc();
    channel->pingevt_->setvar<jrp::misc::Onion>("onion",this);
    vmp_index *cid=new vmp_index();
    (*cid)=channel->cid_; 
    channel->pingevt_->setvar<vmp_index>("cid",cid);
    if(channel->orstatus_ == jrp::misc::or_status_source)
        channel->keyexpired_=ui_->manager()->time_now()+keyexpired_;
}

void Onion::connect_handshake(jrp::misc::OrChannel *channel,vmp::Buf *payload)
{
    vmp_size size=channel->ssl_.size();
    crypto::SslBio *bio=channel->ssl_[size-1];
    vmp::Buf out,out2;
    json::Json json;
    vmp_int ret=bio->connect(payload,&out);
    vmp::str peer=channel->chain_[size-1];
    if(ret == -1)
        connect_err(channel,peer,jrp::status_accesspeerdenied);
    else if(ret == 0)
    {
        japi_orchannel(json.root());
        channel->msg_ordata_source_timeout(&json,&out,size-1);
    }
    else
    {
        crypto::X509_Wrap x509;
        bio->get_peer_x509(&x509);
        if(peer == x509.address())
        {
            vmp_int ret=next_channel(channel,&json,payload);
            if(ret == -1)
                connect_err(channel,channel->chain_[channel->ssl_.size()-1],jrp::status_accesspeerdenied);
        }
        else
            connect_err(channel,peer,jrp::status_accesspeerdenied);
    }
}

void Onion::accept_handshake(jrp::misc::OrChannel *channel,vmp::Buf *payload)
{
    vmp::Buf out;
    crypto::SslBio *bio;
    json::Json json;
    vmp_bool init=false;
    if(channel->orstatus_ == jrp::misc::or_status_init)
    {
        bio=storagebio_.get();
        bio->set(ui_->ctx()->get());
        channel->ssl_.push_back(bio);
        channel_status_internal(channel,jrp::misc::or_status_accept);
        init=true;
    }
    else if(channel->orstatus_ == jrp::misc::or_status_accept)
        bio=channel->ssl_[0];
    vmp_int ret=bio->accept(payload,&out);
    if(ret == -1)
    {    
        channel_err(channel,jrp::misc::japi("orchannel"),jrp::status_resource_accessdenied,"",false);
        channel->route_->close(jrp::status_ok);
        free_channel(channel);
    }
    else 
    {
        if(ret == 1)
            channel_ok(channel,&out);
        else if(init)
            channel->msg_orreply_base(jrp::misc::japi("orchannel"),jrp::status_ok,"",&out);
        else 
            channel->msg_orreply(jrp::misc::japi("orchannel"),jrp::status_ok,"",&out);
    }
}

event::Cell *Onion::actionevt_channel(vmp_index cid)
{
    jrp::misc::OrChannel *channel;
    event::Cell *cell=0;
    mutex_.lock();
    if(channel_.search(cid,&channel))
        cell=channel->actionevt_;
    mutex_.unlock();
    return cell;
}

void Onion::actionevt_disable(vmp_index cid)
{
    jrp::misc::OrChannel *channel;
    mutex_.lock();
    if(channel_.search(cid,&channel) && (channel->actionevt_ !=0))
        actionui_->deactive_timer(channel->actionevt_);
    mutex_.unlock();
}

void Onion::update_keys(vmp_index cid)
{
    jrp::misc::OrChannel *channel;
    mutex_.lock();
    if(channel_.search(cid,&channel))
    {
        vmp::time::Time now=ui_->manager()->time_now();
        if((channel->keyexpired_ != 0) && (channel->keyexpired_ <= now))
        {
            for(vmp_index i=0;i<channel->ssl_.size();i++)
                channel->ssl_[i]->key_update();
            channel->keyexpired_=now+keyexpired_;
        }
    }
    mutex_.unlock();
}

void Onion::register_common(jrp::JrpCommon *common,vmp::str permits)
{
    if(common !=0)
    {
        common->add_reqdata(jrp::misc::japi("orchannel"),{jrp::misc::japi("ordata")},{jrp::misc::japi("orreply"),jrp::misc::japi("ordata")},permits);
    }
}

void Onion::management_impl(jrp::JrpReq *jreq,vmp::Buf *payload)
{
    json::JData jdata;
    jrp::misc::OrChannel *channel=0;
    vmp::str type=jreq->jdata_type();
    if(type == jrp::misc::japi("orchannel"))
    {    
        try
        {
            channel=new_channel();
            channel->chain_.push_back(ui_->peer_fingerprint(jreq->cell()));
            set_channel_route(channel,jreq);
            accept_handshake(channel,payload);   
        }
        catch(vmp::exception &x)
        {
            if(channel != 0)
                free_channel(channel);
            jreq->close(jrp::status_internal);
        }
    }
    else
    {
        try
        {
            jdata.set(jreq->jdata_root());
            vmp_index cid=jdata.get_integer_range("cid",1,vmp::INTMAX);
            channel_close_i(cid,jrp::status_resource_killed);
            jreq->close(jrp::status_ok); 
        }
        catch(vmp::exception &x)
        {
            jreq->close(jrp::status_malformed_msg);    
        }
    }
}

void Onion::management_response_impl(jrp::JrpReq *jreq,json::JsonObj *jdata,vmp::Buf *payload)
{
    vmp_index cid;
    json::Json json;
    vmp::Buf out,out2;
    vmp::str peer;
    jrp::misc::OrChannel *channel=0;
    source_.search(jreq->key(),&cid);
    channel_.search(cid,&channel);
    channel->channel_timeout_reset();
    json::JData jd;
    jd.set(jdata);
    vmp::str type=jd.jtype(),cmd,msg;
    vmp_size s;
    vmp_int status;
    try
    {
        if(type == jrp::misc::japi("orreply"))
        {
            if(channel->orstatus_ == jrp::misc::or_status_channel)
            {
                cmd=jd.get_text("cmd");
                status=jd.get_integer("status");
                msg=jd.get_text("msg");
                if(status != jrp::status_ok)
                    connect_err(channel,channel->chain_[0],status,msg);
                else if(cmd == jrp::misc::japi("orchannel"))
                    connect_handshake(channel,payload);
                else
                    channel_close_i(channel->cid_,jrp::status_protocolbad);
            }
            else if(channel->orstatus_ == jrp::misc::or_status_relaychannel)
            {
                 cmd=jd.get_text("cmd");
                 status=jd.get_integer("status");
                 msg=jd.get_text("msg");
                 if(cmd == jrp::misc::japi("orchannel"))
                 {
                     cmd=jrp::misc::japi("orrelay");
                     channel_status_internal(channel,jrp::misc::or_status_relay);
                     channel->msg_orreply_crypt(cmd,status,msg,payload);
                 }
                 else
                     channel_close_i(cid,jrp::status_protocolbad);
            }
            else
                channel_close_i(cid,jrp::status_protocolbad);
        }
        else if(type == jrp::misc::japi("ordata"))
        {
            if(channel->orstatus_ == jrp::misc::or_status_source)
            {
                s=channel->ssl_.size();
                channel->onion_packet_decrypt(payload,&out,s);
                channel->onion_packet_read(&out,&json,&out2);
                jd.set(json.root());
                if(jd.jtype() != jrp::misc::japi("orping"))  
                    source_recv(channel,json.root(),&out2);
                channel->channel_timeout();
            }
            else if(channel->orstatus_ == jrp::misc::or_status_relay)
                channel->relay_from_target(payload);   
            else if((channel->orstatus_ == jrp::misc::or_status_channel) || (channel->orstatus_ == jrp::misc::or_status_channelrelay))
            {
                s=channel->ssl_.size();
                channel->onion_packet_decrypt(payload,&out,s-1);
                channel->onion_packet_read(&out,&json,&out2);
                jd.set(json.root());
                if(jd.jtype() == jrp::misc::japi("orreply"))
                {
                    cmd=jd.get_text("cmd");
                    status=jd.get_integer("status");
                    msg=jd.get_text("msg");
                    if(status != jrp::status_ok)
                        connect_err(channel,channel->chain_[s-1],status,msg);
                    else if((cmd == jrp::misc::japi("orchannel") && channel->orstatus_ == jrp::misc::or_status_channel))
                        connect_handshake(channel,&out2);
                    else if((cmd == jrp::misc::japi("orrelay") && channel->orstatus_ == jrp::misc::or_status_channelrelay))
                    {
                        channel_status_internal(channel,jrp::misc::or_status_channel);
                        connect_handshake(channel,&out2);
                    }
                    else
                        channel_close_i(channel->cid_,jrp::status_protocolbad);
                }
                else
                    channel_close_i(cid,jrp::status_protocolbad);
            }
            else if(channel->orstatus_ == jrp::misc::or_status_channeltarget)
            {
                s=channel->ssl_.size();
                channel->onion_packet_decrypt(payload,&out,s);
                channel->onion_packet_read(&out,&json,&out2);
                jd.set(json.root());
                if(jd.jtype() == jrp::misc::japi("orreply"))
                {
                    cmd=jd.get_text("cmd");
                    status=jd.get_integer("status");
                    msg=jd.get_text("msg");
                    if(status != jrp::status_ok)
                        connect_err(channel,channel->chain_[s-1],status,msg);
                    else if(cmd == jrp::misc::japi("ortarget"))
                    {
                        connect_ok(channel);
                        channel->channel_timeout();
                    }
                    else
                        channel_close_i(channel->cid_,jrp::status_protocolbad);
                }
                else
                    channel_close_i(channel->cid_,jrp::status_protocolbad);
            }
            else
                channel_close_i(cid,jrp::status_protocolbad);
        }
        else
            channel_close_i(cid,jrp::status_protocolbad);    
    }
    catch(vmp::exception &x)
    {
        channel_close_i(cid,jrp::status_protocolbad);
    }     
}
        
void Onion::management_push_impl(jrp::JrpReq *jreq,json::JsonObj *jdata,vmp::Buf *payload)
{
    vmp_index cid;
    json::Json json;
    vmp::Buf out,out2;
    jrp::misc::OrChannel *channel=0;
    json::JData jd;
    route_.search(jreq->key(),&cid);
    channel_.search(cid,&channel);
    channel->channel_timeout_reset();
    jd.set(jdata);
    vmp::str type=jd.jtype();
    try
    {
        if(type == jrp::misc::japi("ordata"))
        {
            if(channel->orstatus_ == jrp::misc::or_status_relay)
                channel->relay_from_source(payload);
            else if(channel->orstatus_ == jrp::misc::or_status_target)
            {
                channel->decrypt_packet(payload,&out);
                channel->onion_packet_read(&out,&json,&out2);
                jd.set(json.root());
                if(jd.jtype() != jrp::misc::japi("orping"))  
                    target_recv(channel,json.root(),&out2);
                channel->channel_timeout();
            }
            else if(channel->orstatus_ == jrp::misc::or_status_accept)
            {
                channel->onion_packet_read(payload,&json,&out);
                jd.set(json.root());
                if(jd.jtype() == jrp::misc::japi("orchannel"))
                    accept_handshake(channel,&out);
                else
                    channel_close_i(cid,jrp::status_protocolbad);
            }
            else if(channel->orstatus_ == jrp::misc::or_status_wait)
            {
                channel->decrypt_packet(payload,&out);
                channel->onion_packet_read(&out,&json,&out2);
                jd.set(json.root());
                type=jd.jtype();
                if(type == jrp::misc::japi("orrelay"))
                {
                    vmp::str target=jd.get_text("target");
                    event::Cell *cell=ui_->connect_search(target);
                    if(cell == 0)
                        channel_err(channel,type,jrp::status_resource_notfound);
                    else if(relay_verify(channel,target))
                    {
                        jrp::misc::japi_orchannel(json.json_new());
                        try
                        {
                            jrp::JrpReq *req=ui_->request(cell,json.root(),&out2);
                            set_channel_source(channel,req);
                            channel_status_internal(channel,jrp::misc::or_status_relaychannel);
                            channel->channel_timeout();
                        }
                        catch(vmp::exception &x)
                        {
                            channel_err(channel,type,jrp::status_internal);
                        }
                    }
                    else
                        channel_err(channel,type,jrp::status_resource_accessdenied);
                }
                else if(type == jrp::misc::japi("ortarget"))
                {    
                    if(!target(channel))
                        channel_err(channel,type,jrp::status_resource_accessdenied);
                }
                else
                    channel_close_i(cid,jrp::status_protocolbad);
            }
            else
                channel_close_i(cid,jrp::status_protocolbad);
        }
        else
            channel_close_i(cid,jrp::status_protocolbad);
    }
    catch(vmp::exception &x)
    {
        channel_close_i(cid,jrp::status_protocolbad);
    }
}

void Onion::management_kill_impl(jrp::JrpReq *jreq)
{
    vmp_index cid;
    jrp::misc::OrChannel *channel;
    if(route_.search(jreq->key(),&cid))
    {
        channel_.search(cid,&channel);
        channel_close_i(cid,jrp::status_ok);
    }
}
        
void Onion::management_close_impl(jrp::JrpReq *jreq)
{
    vmp_index cid;
    jrp::misc::OrChannel *channel;
    if(source_.search(jreq->key(),&cid))
    {
       if(channel_.search(cid,&channel))
       {
           if(channel->orstatus_==jrp::misc::or_status_close)
               free_channel(channel);
           else
           {
               vmp_int status=jreq->status();
               vmp::str msg=jreq->msg();
               channel_close_i(cid,status,msg);
           }
       }
    }
}

void Onion::register_management(jrp::JrpCommon *common,vmp::str permits)
{
    common->add_reqdata(jrp::misc::japi("orchannelclose"),{},{},permits);
}

vmp::time::Time Onion::optimeout()
{
    return optimeout_;
}

vmp::time::Time Onion::ping()
{
    return ping_;
}

vmp::time::Time Onion::keyexpired()
{
    return keyexpired_;
}

vmp_size Onion::maxchannel()
{
    return maxchannel_;
}

void Onion::set_params(vmp::time::Time optimeout,vmp::time::Time ping,vmp::time::Time keyexpired,vmp_size maxchannel)
{
    if(optimeout != 0)
        optimeout_=optimeout;
    if(ping != 0)
        ping_=ping;
    if(keyexpired != 0)
        keyexpired_=keyexpired;
    if(maxchannel != 0)
        maxchannel_=maxchannel;
}

void Onion::set_status(jrp::misc::Status *status,vmp::str permits)
{
    status_=status;
    if(permits != "*")
        status_->set_limit(jrp::misc::japi("orchannelinfo"),permits);
}

void Onion::set_source_cb(jrp::misc::ONIONCB connect,jrp::misc::ONIONRECVCB recvsource,jrp::misc::ONIONERRCB connecterr,jrp::misc::ONIONCLOSECB closesource)
{
    if(connect == 0)
        connect_=jrp::misc::empty_onioncb;
    else
        connect_=connect;
    if(recvsource == 0)
        recvsource_=jrp::misc::empty_onionrecvcb;
    else
        recvsource_=recvsource;
    if(connecterr == 0)
        connecterr_=jrp::misc::empty_onionerrcb;
    else
        connecterr_=connecterr;
    if(closesource == 0)
        closesource_=jrp::misc::empty_onionclosecb;
    else
        closesource_=closesource;
}

void Onion::set_channel_cb(jrp::misc::ONIONACCEPTCB accept,jrp::misc::ONIONCLOSECB channelerr,jrp::misc::ONIONVRFYCB relayvrfy,jrp::misc::ONIONCB relay,jrp::misc::ONIONCLOSECB closerelay)
{
    if(accept == 0)
        accept_=jrp::misc::empty_onionacceptcb;
    else
        accept_=accept;
    if(channelerr == 0)
        channelerr_=jrp::misc::empty_onionclosecb;
    else
        channelerr_=channelerr;
    if(relayvrfy == 0)
        relayvrfy_=jrp::misc::empty_onionvrfycb;
    else
        relayvrfy_=relayvrfy;
    if(relay == 0)
        relay_=jrp::misc::empty_onioncb;
    else
        relay_=relay;
    if(closerelay == 0)
        closerelay_=jrp::misc::empty_onionclosecb;
    else
        closerelay_=closerelay;
}

void Onion::set_target_cb(jrp::misc::ONIONVRFYCB targetvrfy,jrp::misc::ONIONCB target,jrp::misc::ONIONRECVCB recvtarget,jrp::misc::ONIONCLOSECB closetarget)
{
    if(targetvrfy == 0)
        targetvrfy_=jrp::misc::empty_onionvrfycb;
    else
        targetvrfy_=targetvrfy;
    if(target == 0)
        target_=jrp::misc::empty_onioncb;
    else
        target_=target;
    if(recvtarget == 0)
        recvtarget_=jrp::misc::empty_onionrecvcb;
    else
        recvtarget_=recvtarget;
    if(closetarget == 0)
        closetarget_=jrp::misc::empty_onionclosecb;
    else
        closetarget_=closetarget;
}

vmp_int Onion::channel_status(vmp_index cid)
{
    vmp_int status=-1;
    jrp::misc::OrChannel *ret;
    mutex_.lock();
    if(channel_.search(cid,&ret))
        status=ret->orstatus_;
    mutex_.unlock();
    return status;
}

vmp_uint Onion::channel_permits(vmp_index cid)
{
    vmp_uint permits=0;
    jrp::misc::OrChannel *ret;
    mutex_.lock();
    if(channel_.search(cid,&ret))
        permits=ret->cid_;
    mutex_.unlock();
    return permits;
}

vmp::vector<vmp::str> Onion::channel_chain(vmp_index cid)
{
    vmp::vector<vmp::str> chain;
    jrp::misc::OrChannel *ret;
    mutex_.lock();
    if(channel_.search(cid,&ret))
        chain=ret->chain_;
    mutex_.unlock();
    return chain;
}

vmp::str Onion::channel_next(vmp_index cid)
{
    vmp::str fingerprint="";
    jrp::misc::OrChannel *ret;
    mutex_.lock();
    if(channel_.search(cid,&ret))
        if(ret->source_ != 0)
            fingerprint=ui_->peer_fingerprint(ret->source_->cell());
    mutex_.unlock();
    return fingerprint;
}

vmp::str Onion::channel_prev(vmp_index cid)
{
    vmp::str fingerprint="";
    jrp::misc::OrChannel *ret;
    mutex_.lock();
    if(channel_.search(cid,&ret))
        if(ret->route_ != 0)
            fingerprint=ui_->peer_fingerprint(ret->route_->cell());
    mutex_.unlock();
    return fingerprint;
}

vmp_bool Onion::channel_peer_x509(vmp_index cid,vmp::str fingerprint,crypto::X509_Wrap *cout)
{
    vmp_bool ret=false;
    jrp::misc::OrChannel *oret;
    mutex_.lock();
    if(channel_.search(cid,&oret))
    {    
        for(vmp_index i=0;i<oret->ssl_.size();i++)
        {
            try
            {
                oret->ssl_[i]->get_peer_x509(cout);
            }
            catch(vmp::exception &x)
            {
                break;
            }
            if(cout->address() == fingerprint)
              ret=true;
            else
              cout->reset();
        }
    }
    mutex_.unlock();
    return ret;
}

vmp::str Onion::channel_target(vmp_index cid)
{
    vmp::str ret="";
    jrp::misc::OrChannel *oret;
    mutex_.lock();
    if(channel_.search(cid,&oret))
    {
        if(((oret->orstatus_ == jrp::misc::or_status_channel) || (oret->orstatus_ == jrp::misc::or_status_channelrelay) || (oret->orstatus_ == jrp::misc::or_status_channeltarget) || (oret->orstatus_ == jrp::misc::or_status_source)) && (oret->chain_.size() > 0))
            ret=oret->chain_[oret->chain_.size()-1];
    }
    mutex_.unlock();
    return ret;   
}

vmp::str Onion::channel_source(vmp_index cid)
{
    vmp::str ret="";
    jrp::misc::OrChannel *oret;
    mutex_.lock();
    if(channel_.search(cid,&oret))
    {
        if(((oret->orstatus_ == jrp::misc::or_status_accept) || (oret->orstatus_ == jrp::misc::or_status_wait) || (oret->orstatus_ == jrp::misc::or_status_relaychannel) || (oret->orstatus_ == jrp::misc::or_status_relay) || (oret->orstatus_ == jrp::misc::or_status_target)) && (oret->ssl_.size() > 0))
        {
            crypto::X509_Wrap cout;
            oret->ssl_[0]->get_peer_x509(&cout);
            ret=cout.address();
        }
    }
    mutex_.unlock();
    return ret;  
}

vmp::str Onion::channel_relay(vmp_index cid)
{
    vmp::str ret="";
    jrp::misc::OrChannel *oret;
    mutex_.lock();
    if(channel_.search(cid,&oret))
    {
        if(((oret->orstatus_ == or_status_relaychannel) || (oret->orstatus_ == or_status_relay)) && (oret->chain_.size() > 1))
            ret=oret->chain_[1];
    }
    mutex_.unlock();
    return ret; 
}

vmp_index Onion::channel(vmp::vector<vmp::str> chain)
{
    json::Json json;
    vmp::Buf crypt;
    jrp::misc::OrChannel *channel;
    if(chain.size() == 0)
        vmp::except("jrp::misc::Onion::channel(chains.size()=0)");
    mutex_.lock();
    try
    {
        channel=new_channel();
        channel->chain_=chain;
        if(next_channel(channel,&json,&crypt) == -1)
        {
            free_channel(channel);
            vmp::except_s(channel->ssl_[0]->get_error());
        }    
    }
    catch(vmp::exception &x)
    {
        mutex_.unlock();
        vmp::except("jrp::misc::Onion::channel() '%s'",x.what());
    }
    try
    {
        set_channel_source(channel,ui_->request_f(channel->chain_[0],json.root(),&crypt));
        channel->channel_timeout();
    }
    catch(vmp::exception &x)
    {    
        free_channel(channel);
        mutex_.unlock();
        vmp::except_s(x.what());
    }
    mutex_.unlock();
    return channel->cid_;
}

void Onion::channel_send(vmp_index cid,json::JsonObj *jdata,vmp::Buf *payload)
{
    jrp::misc::OrChannel *oret;
    mutex_.lock();
    if(jdata == 0)
        vmp::except_s("jrp::misc::Onion::channel_send(jdata=Null)");
    if(channel_.search(cid,&oret))
    {
        json::Json json;
        json.json_new_obj(jdata);
        if(oret->orstatus_ == or_status_source)
            oret->msg_ordata_source(&json,payload,oret->ssl_.size());
        else if(oret->orstatus_ == or_status_target)
            oret->msg_ordata_target(&json,payload);
    }
    mutex_.unlock();
}

void Onion::channel_action(vmp_index cid,jrp::misc::ONIONCB cb,vmp::time::Time time)
{
    jrp::misc::OrChannel *oret;
    mutex_.lock();
    if(channel_.search(cid,&oret))
    {
        if(cb == 0)
        {
            if(oret->actionevt_ != 0)
                actionui_->deactive_timer(oret->actionevt_);   
            
        }
        else if(time == 0)
        {
            if(oret->actionevt_ != 0)
                actionui_->deactive_timer(oret->actionevt_);
            mutex_.unlock();
            cb(this,cid);    
            return;
        }
        else if(time > 0)
        {
            if(oret->actionevt_ == 0)
            {
                try
                {
                    oret->actionevt_=actionui_->new_timer("actionevt",time);
                    oret->actionevt_->setvar<jrp::misc::Onion>("onion",this);
                    oret->actionevt_->setvar<jrp::misc::OrChannel>("channel",oret);
                    oret->actionevt_->alloc();
                    oret->actioncb_=cb;
                }
                catch(vmp::exception &x)
                {
                    mutex_.unlock();
                    return;
                }
            }
            else
            {    
                actionui_->active_timer(oret->actionevt_,time);
                oret->actioncb_=cb;
            }
        }
    }
    mutex_.unlock();
}

void Onion::channel_action_disable(vmp_index cid)
{
    jrp::misc::OrChannel *oret;
    mutex_.lock();
    if(channel_.search(cid,&oret))
    {
        if(oret->actionevt_ != 0)
            actionui_->deactive_timer(oret->actionevt_);
    }
    mutex_.unlock();
}

void Onion::channel_close(vmp_index cid,vmp_int status,vmp::str msg)
{
    mutex_.lock();
    channel_close_i(cid,status,msg);
    mutex_.unlock();
}

}}}

