/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 22/07/2025
*/

#ifndef VAMPIRIA_JRP_MISC_ONION_CHANNEL_H

#define VAMPIRIA_JRP_MISC_ONION_CHANNEL_H 1

namespace vampiria { namespace jrp { namespace misc {

const vmp_int or_status_init=0;/*!<Channel or in initial state */
const vmp_int or_status_channel=1;/*!<Channel or in creates channel*/
const vmp_int or_status_channelrelay=2;/*!<Waiting for channel confirmation from relay*/
const vmp_int or_status_channeltarget=3;/*!<Waiting for channel confirmation from target*/
const vmp_int or_status_source=4;/*!<Channel or in source*/
const vmp_int or_status_accept=5;/*!<Channel or in accept channel*/
const vmp_int or_status_wait=6;/*!<Channel or in wait command*/
const vmp_int or_status_relaychannel=7;/*!<Channel or in state creates a relay between peer */
const vmp_int or_status_relay=8;/*!<Channel or in realy mode*/
const vmp_int or_status_target=9;/*!<Channel or in target mode*/
const vmp_int or_status_close=10;/*!<Channel or in close*/

//! Takes in input a or int status is returns the associated status string 
/*!
    @param status message states
    @return status string
*/
vmp::str or_msg_status(vmp_int status);

//!Class used for onion routing channel
class OrChannel
{
    public:
        //!A constructor
        OrChannel();
        
        //!A destructor
        ~OrChannel();
        
        jrp::JrpUI *ui_;/*!<jrp interface used for onion routing*/
        vmp::time::Time optimeout_;/*!<Timeout for connection operations */
        vmp::time::Time ping_;/*!<Number of seconds for each ping command(60 s)*/
        vmp::time::Time keyexpired_;/*!<Exipred keys,used from source node*/
        
        //!Internal usage
        void channel_timeout();
        
        //!Internal usage
        void channel_timeout_reset();
        
        vmp_index cid_;/*!<Channel id*/
        vmp_int orstatus_;/*!<Channel status*/
        jrp::JrpReq *source_;/*!<Request sent by local peer*/
        jrp::JrpReq *route_;/*!<Request received from a peer*/
        vmp_uint permits_;/*!<Permits channel*/
        vmp::vector<vmp::str> chain_;/*!<Peer chain*/
        vmp::vector<crypto::SslBio *> ssl_;/*!<Bio channels*/
        
        event::Cell *pingevt_;/*!<ping event*/
        event::Cell *actionevt_;/*!<aux event*/
        jrp::misc::ONIONCB actioncb_;/*!< action callback*/
        
        //! Create a json data jrp::misc::japi("ornodeinfo") used for channel info
        /*!
            <pre>
            { 
                jtype: json::japi("ornodeinfo"),
                jbody: 
                {
                    address:'node address',
                    subject:'node subject',
                    type:'node type'
                }
            }
            </pre>
            @param obj object where the json data is inserted
            @param address node address(read with jdata.get_text("address"))
            @param subject node subject(read with jdata.get_text("subject"))
            @param type node type (read with jdata.get_text("type"))
            @return void or except in case of failure
        */
        void japi_ornodeinfo(json::JsonObj *obj,vmp::str address,vmp::str subject,vmp::str type);
        
        //! Create a json data jrp::misc::japi("orchannelinfo") used for channel info
        /*!
            <pre>
            { 
                jtype: json::japi("orchannelinfo"),
                jbody: 
                {
                    cid:'channel id'(read with jdata.get_text("cid")),
                    status:channel status(read with jdata.get_integer("status")),
                    status_str:'Channel status str'(read with jdata.get_text("status_str")),
                    permits:Channel permits(read with jdata.get_integer_range("permits",0,vmp::INTMAX)),
                    nodes:'[{jrp::misc::japi("ornodeinfo")},...](read with jdata.get_list("nodes",list,jrp::misc::japi("ornodeinfo")))
                }
            }
            </pre>
            @param obj object where the json data is inserted
            @return void or except in case of failure
        */
        void japi_orchannelinfo(json::JsonObj *obj);
        
        //!Internal usage
        void onion_packet_write(vmp::Buf *buf,json::Json *json,vmp::Buf *payload=0);
        
        //!Internal usage
        void onion_packet_read(vmp::Buf *buf,json::Json *json,vmp::Buf *payload);
        
        //!Internal usage
        void onion_packet_crypt(vmp::Buf *ibuf,vmp::Buf *obuf,vmp_size size);
        
        //!Internal usage
        void onion_packet_decrypt(vmp::Buf *ibuf,vmp::Buf *obuf,vmp_size size);
        
        //!Internal usage
        void crypt_packet(vmp::Buf *ibuf,vmp::Buf *obuf);
        
        //!Internal usage
        void decrypt_packet(vmp::Buf *ibuf,vmp::Buf *obuf);
        
        //!Send message ordata from source node
        /*!
            @param json json json parser jdata
            @param payload payload message
            @param size Number of channels used
        */
        void msg_ordata_source(json::Json *obj,vmp::Buf *payload,vmp_size size);
        
        //!Send message ordata from source and set timeout
        /*!
            @param json json json parser jdata
            @param payload payload message
            @param size Number of channels used
        */
        void msg_ordata_source_timeout(json::Json *json,vmp::Buf *payload,vmp_size size);
        
        //!Send message ordata from target node
        /*!
             @param json json json parser jdata
             @param payload payload message
             @param size Number of channels used
        */
        void msg_ordata_target(json::Json *json,vmp::Buf *payload);
        
        //!Send message ordata from target node and set timeout
        /*!
             @param json json json parser jdata
             @param payload payload message
             @param size Number of channels used
        */
        void msg_ordata_target_timeout(json::Json *json,vmp::Buf *payload);
        
        //!Send orreply message without a ordata label
        /*!
            @param cmd replication command
            @param status reply jrp status
            @param msg msg jrp status
            @param payload payload message
        */
        void msg_orreply_base(vmp::str cmd,vmp_int status,vmp::str msg="",vmp::Buf *payload=0);
        
        //!Send orreply message
        /*!
            @param cmd replication command
            @param status reply jrp status
            @param msg msg jrp status
            @param payload payload message
        */
        void msg_orreply(vmp::str cmd,vmp_int status,vmp::str msg="",vmp::Buf *payload=0);
        
        //!Send orreply message with crypt data
        /*!
            @param cmd replication command
            @param status reply jrp status
            @param msg msg jrp status
            @param payload payload message
        */
        void msg_orreply_crypt(vmp::str cmd,vmp_int status,vmp::str msg="",vmp::Buf *payload=0);
        
        //!Send orrealy message
        /*!
            @param target relay target
            @param payload payload message
        */
        void msg_orrelay(vmp::str target,vmp::Buf *payload);
        
        //!Relay a message from the source
        /*!
            @param payload payload buffer
        */
        void msg_orrelay_source(vmp::Buf *payload);
        
        //!Relay a message from the target
        /*!
            @param payload payload buffer
        */
        void msg_orrelay_target(vmp::Buf *payload);
        
        //!Send ortarget message
        void msg_ortarget();
        
        //!Internal usage
        void relay_from_source(vmp::Buf *payload);
        
        //!Internal usage
        void relay_from_target(vmp::Buf *payload);
};

}}}

#endif

