/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 22/07/2025
*/

#include "jrp.h"

namespace vampiria { namespace jrp { namespace misc {

vmp::str or_msg_status(vmp_int status)
{
    switch(status)
    { 
        case jrp::misc::or_status_init:
            return "Init";
        case jrp::misc::or_status_channel:
            return "Channel handshake";
        case jrp::misc::or_status_channelrelay:
             return "Channel relay confirm";
        case jrp::misc::or_status_channeltarget:
            return "Channel target confirm";
        case jrp::misc::or_status_source:
            return "Channel source";
        case jrp::misc::or_status_accept:
            return "Accept handshake";
        case jrp::misc::or_status_wait:
            return "Wait command";
        case jrp::misc::or_status_relaychannel:
            return "Wait Relay channel";
        case jrp::misc::or_status_relay:
            return "Relay channel";
        case jrp::misc::or_status_target:
            return "Channel target";
        case jrp::misc::or_status_close:
            return "Channel close";
    }
    return "";
}

OrChannel::OrChannel()
{
    optimeout_=0;
    ping_=0;
    keyexpired_=0;
    cid_=0;
    orstatus_=jrp::misc::or_status_init;
    source_=0;
    route_=0;
    permits_=0;
    
    pingevt_=0;
    actionevt_=0;
    actioncb_=jrp::misc::empty_onioncb;
}

OrChannel::~OrChannel()
{
    ui_=0;
}

void OrChannel::channel_timeout()
{
    switch(orstatus_)
    {
        case jrp::misc::or_status_channel:
        case jrp::misc::or_status_channelrelay:
        case jrp::misc::or_status_channeltarget:
        case jrp::misc::or_status_relaychannel:
            source_->set_timeout(optimeout_);
            break;
        case jrp::misc::or_status_source:
            source_->set_timeout(ping_+optimeout_);
            break;
        case jrp::misc::or_status_accept:
        case jrp::misc::or_status_wait:
            route_->set_timeout(optimeout_);
            break;    
        case jrp::misc::or_status_relay:
        case jrp::misc::or_status_target:
            route_->set_timeout(ping_+optimeout_);
            break;
        default:
            break;
    }
}

void OrChannel::channel_timeout_reset()
{
    if(source_ != 0)
        source_->set_timeout(0);
    if(route_ != 0)
        route_->set_timeout(0);
}

void OrChannel::japi_ornodeinfo(json::JsonObj *obj,vmp::str address,vmp::str subject,vmp::str type)
{
    json::JData jdata;
    vmp::except_check_pointer((void *) obj,"jrp::misc::OrChannel::japi_ornodeinfo(obj=Null)");
    jdata.new_data(obj,jrp::misc::japi("ornodeinfo"));
    jdata.set_text("address",address);
    jdata.set_text("subject",subject);
    jdata.set_text("type",type);
}

void OrChannel::japi_orchannelinfo(json::JsonObj *obj)
{
    json::Json tmp;
    json::JData jdata;
    json::JList nodes;
    vmp_size s;
    vmp::except_check_pointer((void *) obj,"jrp::misc::OrChannel::japi_orchannelinfo(obj=Null)");
    jdata.new_data(obj,jrp::misc::japi("orchannelinfo"));
    jdata.set_integer_range("cid",1,vmp::INTMAX,(vmp_int)cid_);
    jdata.set_integer("status",orstatus_);
    jdata.set_text("status_str",jrp::misc::or_msg_status(orstatus_));
    jdata.set_integer_range("permits",0,vmp::INTMAX,(vmp_int)permits_);
    jdata.new_list("nodes",&nodes,jrp::misc::japi("ornodeinfo"));
    crypto::X509_Wrap cout;
    vmp::str address,address2,subject,type;
    if((orstatus_ == jrp::misc::or_status_channel) || (orstatus_ == jrp::misc::or_status_channelrelay) || (orstatus_ == jrp::misc::or_status_channeltarget) || (orstatus_ == jrp::misc::or_status_source))
    {
        japi_ornodeinfo(tmp.json_new(),ui_->ctx()->fingerprint(),ui_->ctx()->subject(),"L");
        nodes.push(tmp.root());
        s=chain_.size();
        for(vmp_index i=0;i<s;i++)
        {
            address=chain_[i];
            try
            {
                if(i < ssl_.size())
                {
                    ssl_[i]->get_peer_x509(&cout);
                    subject=cout.subject();    
                }
                else
                    subject="";
                if(i == s-1)
                    type="T";
                else
                    type="R";
                japi_ornodeinfo(tmp.json_new(),address,subject,type);
                nodes.push(tmp.root());
            }
            catch(vmp::exception &x)
            {
            }
        }
    }
    else if((orstatus_ == jrp::misc::or_status_accept) || (orstatus_ == jrp::misc::or_status_wait) || (orstatus_ == jrp::misc::or_status_target))
    {
        try
        {
            ssl_[0]->get_peer_x509(&cout);
            address=cout.address();
            japi_ornodeinfo(tmp.json_new(),address,cout.subject(),"S");
            nodes.push(tmp.root());
            address2=ui_->peer_fingerprint(route_->cell());
            if(address != address2)
            {
                japi_ornodeinfo(tmp.json_new(),address2,ui_->peer_subject(route_->cell()),"R");
                nodes.push(tmp.root());
            }
            japi_ornodeinfo(tmp.json_new(),ui_->ctx()->fingerprint(),ui_->ctx()->subject(),"L");
            nodes.push(tmp.root());
        }
        catch(vmp::exception &x)
        {
        }
    }
    else if((orstatus_ == jrp::misc::or_status_relaychannel) || (orstatus_ == jrp::misc::or_status_relay))
    {
        try
        {
            ssl_[0]->get_peer_x509(&cout);
            address=cout.address();
            japi_ornodeinfo(tmp.json_new(),address,cout.subject(),"S");
            nodes.push(tmp.root());
            address2=ui_->peer_fingerprint(route_->cell());
            if(address != address2)
            {
                japi_ornodeinfo(tmp.json_new(),ui_->peer_fingerprint(route_->cell()),ui_->peer_subject(route_->cell()),"R");
                nodes.push(tmp.root());
            }
            japi_ornodeinfo(tmp.json_new(),ui_->ctx()->fingerprint(),ui_->ctx()->subject(),"L");
            nodes.push(tmp.root());
            japi_ornodeinfo(tmp.json_new(),ui_->peer_fingerprint(source_->cell()),ui_->peer_subject(source_->cell()),"R");
            nodes.push(tmp.root());
        }
        catch(vmp::exception &x)
        {
        }
    }
}

void OrChannel::onion_packet_write(vmp::Buf *buf,json::Json *json,vmp::Buf *payload)
{
    buf->reset();
    vmp::str str=json->json_str();
    buf->write_size(str.size(),4);
    buf->write_str(str);
    if((payload != 0) && (payload->size() != 0))
        buf->write_buf(payload);
    buf->index();
}
        
void OrChannel::onion_packet_read(vmp::Buf *buf,json::Json *json,vmp::Buf *payload)
{
    buf->index();
    payload->reset();
    vmp_size psize=buf->read_size(4);
    json->parse_from_str(buf->read_str(psize));
    buf->read_buf(payload,buf->size_reading());
    buf->index();
}

void OrChannel::onion_packet_crypt(vmp::Buf *ibuf,vmp::Buf *obuf,vmp_size size)
{
    vmp::Buf src;
    obuf->write_buf(ibuf);
    vmp_int ret;
    for(vmp_index i=size;i>0;i--)
    {
        src.reset();
        src.write_buf(obuf);
        ret=ssl_[i-1]->send(&src,obuf);
        if(ret == -1)
            vmp::except("Chain=%s '%s'",chain_[i-1].c_str(),ssl_[i-1]->get_error());
    }
}

void OrChannel::onion_packet_decrypt(vmp::Buf *ibuf,vmp::Buf *obuf,vmp_size size)
{
    vmp::Buf src;
    obuf->write_buf(ibuf);
    vmp_int ret;
    for(vmp_index i=0;i<size;i++)
    {
        src.reset();
        src.write_buf(obuf);
        ret=ssl_[i]->recv(&src,obuf);
        if(ret == -1)
            vmp::except("Chain=%s '%s'",chain_[i].c_str(),ssl_[i]->get_error().c_str());
        else if(ret == -2)
            vmp::except("Chain=%s 'Bio closed'",chain_[i].c_str());
    }
}

void OrChannel::crypt_packet(vmp::Buf *ibuf,vmp::Buf *obuf)
{
     vmp_int ret=ssl_[0]->send(ibuf,obuf);
     if(ret == -1)
        vmp::except_s(ssl_[0]->get_error());
}

void OrChannel::decrypt_packet(vmp::Buf *ibuf,vmp::Buf *obuf)
{
    vmp_int ret=ssl_[0]->recv(ibuf,obuf);
    if(ret == -1)
        vmp::except_s(ssl_[0]->get_error());
    else if(ret == -2)
        vmp::except_s("Bio closed");
}

void OrChannel::msg_ordata_source(json::Json *json,vmp::Buf *payload,vmp_size size)
{
    json::Json ordata;
    vmp::Buf in,out;
    jrp::misc::japi_ordata(ordata.root());
    onion_packet_write(&in,json,payload);
    onion_packet_crypt(&in,&out,size);
    source_->push(ordata.root(),&out);
}

void OrChannel::msg_ordata_source_timeout(json::Json *json,vmp::Buf *payload,vmp_size size)
{
    msg_ordata_source(json,payload,size);
    channel_timeout();
}

void OrChannel::msg_ordata_target(json::Json *json,vmp::Buf *payload)
{
    json::Json ordata;
    vmp::Buf in,out;
    onion_packet_write(&in,json,payload);
    crypt_packet(&in,&out);
    jrp::misc::japi_ordata(ordata.root());
    route_->response(ordata.root(),&out);
}

void OrChannel::msg_ordata_target_timeout(json::Json *json,vmp::Buf *payload)
{
    msg_ordata_target(json,payload);
    channel_timeout();
}

void OrChannel::msg_orreply_base(vmp::str cmd,vmp_int status,vmp::str msg,vmp::Buf *payload)
{
    json::Json json;
    jrp::misc::japi_orreply(json.root(),cmd,status,msg);
    route_->response(json.root(),payload);
    channel_timeout();
}

void OrChannel::msg_orreply(vmp::str cmd,vmp_int status,vmp::str msg,vmp::Buf *payload)
{
    vmp::Buf out;
    json::Json json;
    jrp::misc::japi_orreply(json.root(),cmd,status,msg);
    onion_packet_write(&out,&json,payload);
    jrp::misc::japi_ordata(json.json_new());
    route_->response(json.root(),&out);
    channel_timeout();
}

void OrChannel::msg_orreply_crypt(vmp::str cmd,vmp_int status,vmp::str msg,vmp::Buf *payload)
{
    json::Json json;
    jrp::misc::japi_orreply(json.root(),cmd,status,msg);
    msg_ordata_target_timeout(&json,payload);
}

void OrChannel::msg_orrelay(vmp::str target,vmp::Buf *payload)
{
    json::Json json;
    jrp::misc::japi_orrelay(json.root(),target);
    msg_ordata_source_timeout(&json,payload,ssl_.size()-1);
}

void OrChannel::msg_orrelay_source(vmp::Buf *payload)
{
    json::Json ordata;
    vmp::Buf out;
    decrypt_packet(payload,&out);
    jrp::misc::japi_ordata(ordata.root());
    source_->push(ordata.root(),&out);
    channel_timeout();
}

void OrChannel::msg_orrelay_target(vmp::Buf *payload)
{
    vmp::Buf out;
    json::Json ordata;
    crypt_packet(payload,&out);
    jrp::misc::japi_ordata(ordata.root());
    route_->response(ordata.root(),&out);
    channel_timeout();
}
    
void OrChannel::msg_ortarget()
{
    json::Json json;
    vmp::Buf payload;
    jrp::misc::japi_ortarget(json.root());
    msg_ordata_source_timeout(&json,0,ssl_.size());
}

void OrChannel::relay_from_source(vmp::Buf *payload)
{
    json::Json json;
    vmp::Buf out;
    jrp::misc::japi_ordata(json.root());
    decrypt_packet(payload,&out);
    source_->push(json.root(),&out);
    channel_timeout();
}
        
void OrChannel::relay_from_target(vmp::Buf *payload)
{
    json::Json json;
    vmp::Buf out;
    jrp::misc::japi_ordata(json.root());
    crypt_packet(payload,&out);
    route_->response(json.root(),&out);
    channel_timeout();
}

void Onion::channel_status_internal(jrp::misc::OrChannel *channel,vmp_int orstatus)
{
    if(status_ != 0)
    {
        vmp::str key;
        json::Json json;
        vmp::unicode::str_write(&key,"or_%u",channel->cid_);
        channel->orstatus_=orstatus;
        channel->japi_orchannelinfo(json.root());
        status_->update(key,json.root());
    }
}

jrp::misc::OrChannel *Onion::new_channel()
{
    if(channel_.size() == maxchannel_)
        vmp::except("jrp::misc::Onion::new_channel() overflow channel");
    jrp::misc::OrChannel *ret;
    ret=storageor_.get();
    if(ret->cid_ == 0)
    {
        ret->optimeout_=optimeout_;
        ret->ping_=ping_;
        ret->cid_=cindex_++;
        ret->ui_=ui_;
    }
    channel_.insert(ret->cid_,ret);
    channel_status_internal(ret,jrp::misc::or_status_init);
    return ret;
}

void Onion::set_channel_source(jrp::misc::OrChannel *channel,jrp::JrpReq *req)
{
    channel->source_=req;
    channel->source_->alloc();
    try
    {
        source_.insert(channel->source_->key(),channel->cid_);
    }
    catch(vmp::exception &x)
    {
        vmp::error("net::misc::onion::set_channel_source() route='%s' %s",channel->route_->key().c_str(),x.what());
    }
}

vmp_bool Onion::close_channel_source(jrp::misc::OrChannel *channel)
{
    if((channel->source_ != 0) && (channel->source_->status() == 1))
    {    
        channel->source_->kill();
        return true;
    }
    return false;
}

void Onion::set_channel_route(jrp::misc::OrChannel *channel,jrp::JrpReq *req)
{
    channel->route_=req;
    channel->route_->alloc();
    try
    {
        route_.insert(channel->route_->key(),channel->cid_);
    }
    catch(vmp::exception &x)
    {
        vmp::error("net::misc::onion::set_channel_route() route='%s' %s",channel->route_->key().c_str(),x.what());
    }
}

void Onion::close_channel_route(jrp::misc::OrChannel *channel,vmp_int status,vmp::str msg)
{
    vmp_index tmp;
    if(msg == "")
        msg=jrp::msg_status(status);
    if(channel->route_ != 0)
    {
       channel->route_->close(status,msg);
       try
       {
           route_.cancel(channel->route_->key(),&tmp);
       }
       catch(vmp::exception &x)
       {
           vmp::error("net::misc::Onion::freechannel() route='%s' %s",channel->route_->key().c_str(),x.what());
       }
       channel->route_->release();
       channel->route_=0;
    }
}

void Onion::free_channel(jrp::misc::OrChannel *channel)
{
    channel->orstatus_=jrp::misc::or_status_init;
    vmp_index tmp;
    jrp::misc::OrChannel *ctmp;
    if(channel->source_ != 0)
    {
       try
       {
           source_.cancel(channel->source_->key(),&tmp);
       }
       catch(vmp::exception &x)
       {
           vmp::error("net::misc::Onion::freechannel() source='%s' %s",channel->source_->key().c_str(),x.what());
       }
       channel->source_->release();
       channel->source_=0;
    }
    channel->permits_=0;
    channel->optimeout_=0;
    channel->ping_=0;
    channel->keyexpired_=0;
    channel->chain_.clear();
    for(vmp_index i=0;i<channel->ssl_.size();i++)
    {
        channel->ssl_[i]->reset();
        storagebio_.free(channel->ssl_[i]);
    }
    channel->ssl_.clear();
    if(status_ != 0)
    {
        vmp::str key;
        vmp::unicode::str_write(&key,"or_%u",channel->cid_);
        status_->cancel(key);
    }
    channel_.cancel(channel->cid_,&ctmp);
    storageor_.free(channel);
}

vmp_int Onion::next_channel(jrp::misc::OrChannel *channel,json::Json *json,vmp::Buf *out)
{
    if(channel->ssl_.size() == channel->chain_.size())
    {    
        channel_status_internal(channel,jrp::misc::or_status_channeltarget);
        channel->msg_ortarget();
        return 1;
    }
    vmp::Buf in;
    vmp_int ret;
    crypto::SslBio *bio=storagebio_.get();
    bio->set(ui_->ctx()->get());
    channel->ssl_.push_back(bio);
    ret=bio->connect(&in,out);
    if(ret == -1)
        return -1;
    if(channel->ssl_.size() == 1)
    {
        channel_status_internal(channel,jrp::misc::or_status_channel);
        jrp::misc::japi_orchannel(json->json_new());
    }
    else
    {
        channel_status_internal(channel,jrp::misc::or_status_channelrelay);
        channel->msg_orrelay(channel->chain_[channel->ssl_.size()-1],out);
    }
    return 0;
}

void Onion::channel_close_i(vmp_index cid,vmp_int status,vmp::str msg)
{
    jrp::misc::OrChannel *channel=0;
    if(msg == "")
        msg=jrp::msg_status(status);
    if(channel_.search(cid,&channel) && (channel->orstatus_ != jrp::misc::or_status_close))
    {
        if(channel->pingevt_ != 0)
        {
            channel->pingevt_->close();
            channel->pingevt_->release();
            channel->pingevt_=0;
        }
        if(channel->actionevt_ != 0)
        {
            channel->actionevt_->close();
            channel->actionevt_->release();
            channel->actionevt_=0;
        }
        close_channel_cb(channel,status,msg);
        channel_status_internal(channel,jrp::misc::or_status_close);
        close_channel_route(channel,status,msg);
        if(!close_channel_source(channel))
            free_channel(channel);
    }
}

}}}

