/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 22/07/2025
*/

#include "onion/japi.h"
#include "onion/cb.h"
#include "onion/channel.h"

#ifndef VAMPIRIA_JRP_MISC_ONION_H

#define VAMPIRIA_JRP_MISC_ONION_H 1

namespace vampiria { namespace jrp { namespace misc {

//!Internal usage
void action_event(event::Cell *cell);

//!Class used for onion connections
class Onion:public jrp::misc::JrpApi_I
{
    private:
        jrp::JrpUI *ui_;/*!<jrp interface used for onion routing*/
        vmp::time::Time optimeout_;/*!<Timeout for connection operations */
        vmp::time::Time ping_;/*!<Number of seconds for each ping command(60 s)*/
        vmp::time::Time keyexpired_;/*!<Exipred keys,used from source node(default 7200 secondi)*/
        vmp_size maxchannel_;/*!<Maximum number of channels that can be created*/
        jrp::misc::Status *status_;/*!<Status object*/
        
        vmp_index cindex_;/*!<Used to assign channel ids*/
        vmp::utils::Storage<crypto::SslBio> storagebio_;/*!<Storage ssl bio*/
        vmp::utils::Storage<jrp::misc::OrChannel> storageor_;/*!<Storage or channel*/
        vmp::Table<vmp::str,vmp_index> source_;/*!<Source table*/
        vmp::Table<vmp::str,vmp_index> route_;/*!<Route table*/
        vmp::Table<vmp_index,jrp::misc::OrChannel *> channel_;/*!<Channel table*/
        
        jrp::misc::ONIONCB connect_;/*!<Callack called from source when a connection is complete*/
        jrp::misc::ONIONRECVCB recvsource_;/*!<Callack calls from source when it receives a message*/
        jrp::misc::ONIONERRCB connecterr_;/*!<Callack called from source when a connection error occurs*/
        jrp::misc::ONIONCLOSECB closesource_;/*!<Callack called from source when a stabilized connection is closed*/
        
        jrp::misc::ONIONACCEPTCB accept_;/*!<Callback call when there is a request for a channel from a remote peer */
        jrp::misc::ONIONCLOSECB channelerr_;/*!<Callback call when there is an error in the construction of a channel initiated by a peer*/
        jrp::misc::ONIONVRFYCB relayvrfy_;/*!<Call callback when a Relay node is verified*/
        jrp::misc::ONIONCB relay_;/*!<Call callback when a Relay node is created*/
        jrp::misc::ONIONCLOSECB closerelay_;/*!<Call callback when a relay node is shutdown*/
        
        jrp::misc::ONIONVRFYCB targetvrfy_;/*!<Call callback when a Target node is verified */
        jrp::misc::ONIONCB target_;/*!<Call callback when a Target node is created*/
        jrp::misc::ONIONRECVCB recvtarget_;/*!<Callack calls from target when it receives a message*/
        jrp::misc::ONIONCLOSECB closetarget_;/*!<Callack called from target when a stabilized connection is closed*/
        
        event::TimerUI *pingui_;/*!<Ping timerui*/
        
        //!Internal usage
        void active_ping(jrp::misc::OrChannel *channel);
        
        //!Internal usage
        void channel_status_internal(jrp::misc::OrChannel *channel,vmp_int orstatus);

        //!Internal usage
        jrp::misc::OrChannel *new_channel();
        
        //!Internal usage
        void set_channel_source(jrp::misc::OrChannel *channel,jrp::JrpReq *req);
        
        //!Internal usage
        vmp_bool close_channel_source(jrp::misc::OrChannel *channel);
        
        //!Internal usage
        void set_channel_route(jrp::misc::OrChannel *channel,jrp::JrpReq *req);
        
        //!Internal usage
        void close_channel_route(jrp::misc::OrChannel *channel,vmp_int status,vmp::str msg="");
        
        //!Internal usage
        void free_channel(jrp::misc::OrChannel *channel);
        
        //!Intenal usage
        /*!
            @return 1 if the connection is complete,0 if a new channel is created,-1 in case of error
        */
        vmp_int next_channel(jrp::misc::OrChannel *channel,json::Json *json,vmp::Buf *out);
        
        //!Internal usage
        void channel_close_i(vmp_index cid,vmp_int status,vmp::str msg="");
        
        //!Internal usage
        void connect_ok(jrp::misc::OrChannel *channel);
        
        //!Interanl usage
        void connect_err(jrp::misc::OrChannel *channel,vmp::str peer,vmp_int status,vmp::str msg="");
        
        //!Internal usage
        void source_recv(jrp::misc::OrChannel *channel,json::JsonObj *jdata,vmp::Buf *payload);
        
        //!Internal usage
        void channel_ok(jrp::misc::OrChannel *channel,vmp::Buf *payload);
        
        //!Internal usage
        void channel_err(jrp::misc::OrChannel *channel,vmp::str cmd,vmp_int status,vmp::str msg="",vmp_bool crypt=true);
        
        //!Internal usage
        vmp_bool relay_verify(jrp::misc::OrChannel *channel,vmp::str target);
        
        //!Internal usage
        vmp_bool target(jrp::misc::OrChannel *channel);
        
        //!Internal usage
        void target_recv(jrp::misc::OrChannel *channel,json::JsonObj *jdata,vmp::Buf *payload);
        
        //!Internal usage
        void close_channel_cb(jrp::misc::OrChannel *channel,vmp_int status,vmp::str msg);
        //!Internal usage
        void connect_handshake(jrp::misc::OrChannel *channel,vmp::Buf *payload);
        
        //!Internal usage
        void accept_handshake(jrp::misc::OrChannel *channel,vmp::Buf *payload);
    protected:
        event::TimerUI *actionui_;/*!<Action event timerui*/
        
        //!Internal usage
        void init_onion(jrp::JrpUI *ui);
        
        //Internal uadge
        void destroy_onion();
    public:
        //!Internal usage
        Onion();
        
        //! A Constructor
        /*!
            @param ui interface for jrp connections used for onion channels
        */
        Onion(jrp::JrpUI *ui);
        
        //! A destructor
        ~Onion();
        
        //!Internal usage
        event::Cell *actionevt_channel(vmp_index cid);
        
        //!Internal usage
        void actionevt_disable(vmp_index cid); 
        
        //!Internal usage
        void update_keys(vmp_index cid);
        
        //!Virtual function
        void register_common(jrp::JrpCommon *common,vmp::str permits="*");
        
        //!Virtual function
        void management_impl(jrp::JrpReq *jreq,vmp::Buf *payload);

        //!Virtual function
        void management_response_impl(jrp::JrpReq *jreq,json::JsonObj *jdata,vmp::Buf *payload);
        
        //!Virtual function
        void management_push_impl(jrp::JrpReq *jreq,json::JsonObj *jdata,vmp::Buf *payload);
        
        //!Virtual function
        void management_kill_impl(jrp::JrpReq *jreq);
        
        //!Virtual function
        void management_close_impl(jrp::JrpReq *jreq);
        
        //!Record jrp request information in jrpcommon for management message 
        /*!
            @param jrp common object
            @param permits Permissions to be assigned
        */
        void register_management(jrp::JrpCommon *common,vmp::str permits="*");
        
        //!Returns operation timeout
        /*!
            @sa optimeout_
            @return operation timeout
        */
        vmp::time::Time optimeout();
        
        //! Returns ping sending interval
        /*!
            @sa ping_
            @return ping sending interval
        */
        vmp::time::Time ping();
        
        //!Returns interval update keys ssl(only source)
        /*
            @sa keyexpired_
            @return interval update keys ssl
        */
        vmp::time::Time keyexpired();
        
        //!Returns Maximum number of channels that can be opened(only target and realy)
        /*!
            @sa maxchannel_
            @return maximum number of channels that can be opened
        */
        vmp_size maxchannel();
        
        //!Set the connection parameters(if value== 0 not setting)
        /*!
            @param optimeout_ operation timeout
            @param ping_ ping sending interval
            @param keyexpired interval update keys ssl
            @param maxchannel maximum number of channels that can be opened
        */
        void set_params(vmp::time::Time optimeout,vmp::time::Time ping,vmp::time::Time keyexpired,vmp_size maxchannel);
        
        //!Set the state variable if you want to insert updates in the status 
        /*!
            @param status the state variable
            @param permits permissions users to send the update
        */
        void set_status(jrp::misc::Status *status,vmp::str permits="*");        
        
        //! Set callbacks for source(if value == 0 default callback)
        /*!
            @param connect callback called from source when a connection is complete
            @param recvsource callback calls from source when it receives a message
            @param connecterr callback called from source when a connection error occurs
            @param closesource callback called from source when a stabilized connection is closed
        */
        void set_source_cb(jrp::misc::ONIONCB connect,jrp::misc::ONIONRECVCB recvsource,jrp::misc::ONIONERRCB connecterr,jrp::misc::ONIONCLOSECB closesource);
        
        //! Set callbacks for create channel(if value == 0 default callback)
        /*!
            @param accept callback call when there is a request for a channel from a remote peer 
            @param channelerr callback call when there is an error in the construction of a channel initiated by a peer
            @param relayvrfy call callback when a Relay node is verified
            @param relay call callback when a Relay node is created
            @param closerelay call callback when a relay node is shutdown
        */
        void set_channel_cb(jrp::misc::ONIONACCEPTCB accept,jrp::misc::ONIONCLOSECB channelerr,jrp::misc::ONIONVRFYCB relayvrfy,jrp::misc::ONIONCB relay,jrp::misc::ONIONCLOSECB closerelay);
        
        //! Set callbacks for target(if value == 0 default callback)
        /*!
            @param targetvrfy call callback when a Target node is verified
            @param target_ call callback when a Target node is created
            @param recvtarget callback calls from target when it receives a message
            @param closetarget callback called from target when a stabilized connection is closed
        */
        void set_target_cb(jrp::misc::ONIONVRFYCB targetvrfy,jrp::misc::ONIONCB target,jrp::misc::ONIONRECVCB recvtarget,jrp::misc::ONIONCLOSECB closetarget);
        
        //! Returns channel status
        /*!
            @param cid channel cid
            @return channel status or -1 in case the channel was not found
        */
        vmp_int channel_status(vmp_index cid);
        
        //! Returns channel permits(only target)
        /*!
            @param cid channel id
            @return channel permits
        */
        vmp_uint channel_permits(vmp_index cid);
        
        //! Returns channel chain
        /*!
            @param cid channel id
            @return channel chain
        */
        vmp::vector<vmp::str> channel_chain(vmp_index cid);
        
        //! Return the address of the next node in the onion chain
        /*!
            @param cid channel id
            @return address previous node or "" in case of failure
        */
        vmp::str channel_next(vmp_index cid);
        
        //! Return the address of the previous node in the onion chain
        /*!
            @param cid channel id
            @return address previous node or "" in case of failure
        */
        vmp::str channel_prev(vmp_index cid);
        
        //! Returns certificate of a peer in the connection chain
        /*!
            @param cid channel id
            @param finegrprint peer in the connection chain
            @param cout certificate output
            @return true if the transaction was successful,otherwise false
        */
        vmp_bool channel_peer_x509(vmp_index cid,vmp::str fingerprint,crypto::X509_Wrap *cout);
        
        //! Returns address the target node
        /*!
            @param cid channel id
            @return address the target node or "" in case of failure
        */
        vmp::str channel_target(vmp_index cid);
        
        //! Returns address the source node
        /*!
            @param cid channel id
            @return address the source node or "" in case of failure
        */
        vmp::str channel_source(vmp_index cid);
        
        //! Returns address the relay node
        /*!
            @param cid channel id
            @return address the relay node or "" in case of failure
        */
        vmp::str channel_relay(vmp_index cid);
        
        //! Create channel
        /*!
            @param chain address chain for channel creation
            @return channel id or except in case of failure
        */
        vmp_index channel(vmp::vector<vmp::str> chain);
        
        //! Send a package through the channel
        /*!
            @param cid channel id
            @param jdata json data
            @param payload payload data
            @return void or except in case of failure
        */
        void channel_send(vmp_index cid,json::JsonObj *jdata,vmp::Buf *payload=0);
        
        //! Active channel action
        /*!
            @param cid channel id
            @param cb action callback
            @param time number of seconds to be taken before the action
        */
        void channel_action(vmp_index cid,jrp::misc::ONIONCB cb,vmp::time::Time time=0);
        
        //!Disable channel action
        /*!
            @param cid channel id
        */
        void channel_action_disable(vmp_index cid);
        
        //! Close channel
        /*!
            @param cid channel id
            @param status exit status
            @param msg exit message
        */
        void channel_close(vmp_index cid,vmp_int status,vmp::str msg="");
};

}}}

#endif

