/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 07/05/2025
 */

#include "crypto.h"

namespace vampiria { namespace crypto {

SslBio::SslBio()
{
    ssl_=0;
    reset();
}

SslBio::SslBio(crypto::Ctx *ctx)
{
    ssl_=0;
    set(ctx);
}

SslBio::~SslBio()
{
    reset();
    ctx_.reset();
}

void SslBio::except_ctx(vmp::str func)
{
    vmp::str err;
    vmp::unicode::str_write(&err,"crypto::SslBio::%s() context not set",func.c_str());
    vmp::except_check_pointer((void *)ctx_.ctx_,err);
}

void SslBio::set_errno(vmp_int ret)
{
    errno_=SSL_get_error(ssl_,ret);
    error_=crypto::err_string();
}

vmp_int SslBio::bio_read(vmp::Buf *out)
{
    out->reset();
    vmp_int n;
    vmp_int retval=0;
    vmp_size s;
    while(1)
    {
        s=vmp::BUFSIZEMAX-out->size();
        if(s == 0)
            break;
        else if(s > vmp::MAXSIZEREAD)
            s=vmp::MAXSIZEREAD;
        out->newsize(out->size()+s);
        n=BIO_read(wbio_,out->pointer(retval),s);
        if(n < 0)
        {
            out->reset();
            return n;
        }
        retval+=n;
        if((vmp_size) n < s)
            break;
    }
    out->newsize((vmp_size)retval);
    return retval;
}

void SslBio::reset()
{
    errno_=0;
    error_="";
    ctype_=0;
    status_=INIT;
    if(ssl_ != 0)
    {
        SSL_clear(ssl_);
        SSL_free(ssl_);
        ssl_=0;
        rbio_=0;
        wbio_=0;
    }
}

void SslBio::set(crypto::Ctx *ctx)
{
    reset();
    ctx_.copy(ctx);
    vmp::except_check_pointer((void *)ctx_.ctx_,"crypto::SslBio::set(ctx=NULL)");
}

vmp_int SslBio::get_errno()
{
    return errno_;
}

vmp::str SslBio::get_error()
{
    return error_;
}
 
crypto::Ctx *SslBio::ctx()
{
    except_ctx("ctx");
    return &ctx_;
}

void SslBio::get_local_x509(crypto::X509_Wrap *cout)
{
    ctx_.get_x509(cout);
}

void SslBio::get_peer_x509(crypto::X509_Wrap *cout)
{
    vmp::except_check_pointer((void *)cout,"crypto::SslBio::get_peer_cerificate(cout=NULL)");
    except_ctx("get_peer_cerificate");
    STACK_OF(X509) *chain;
    chain=SSL_get_peer_cert_chain(ssl_);
    vmp::except_check_pointer((void *)chain,"No certificate was presented by the peer or no connection was established");
    vmp_size num = sk_X509_num(chain);
    if(num == 0)
    {
        X509 *x509=SSL_get_peer_certificate(ssl_);
        vmp::except_check_pointer((void *)x509,"No certificate was presented by the peer or no connection was established");
        cout->set(x509);
    }
    else
    {
        crypto::X509_Wrap *tmp;
        for(vmp_index i=0;i<num;i++)
        {
            if(i == 0)
            {
                cout->set(sk_X509_value(chain,i));
                tmp=cout;
            }
            else
            {
                tmp->parent_=new X509_Wrap();
                tmp=tmp->parent_;
                tmp->set(sk_X509_value(chain,i));
            }
        }
    }
}
vmp_int SslBio::connect(vmp::Buf *in,vmp::Buf *out)
{
    except_ctx("connect");
    vmp::except_check_pointer((void *)in,"crypto::SslBio::connect(in=Null)");
    vmp::except_check_pointer((void *)out,"crypto::SslBio::connect(out=Null)");
    if(status_ == INIT)
    {
        ssl_=SSL_new(ctx_.ctx_);
        rbio_=BIO_new(BIO_s_mem());
        wbio_=BIO_new(BIO_s_mem());
        SSL_set_bio(ssl_,rbio_,wbio_);
        SSL_set_connect_state(ssl_);
        status_=CONNECT;
        ctype_=1;
    }
    else if((status_ == CONNECT) || (status_ == CONNECTACK))
        BIO_write(rbio_,in->pointer(),in->size());
    else
        vmp::except("crypto::SslBio::connect() wrong function call");
    vmp_int ret=SSL_do_handshake(ssl_);
    if(ret < 0)
    {
        vmp_int err=SSL_get_error(ssl_,ret);
        if((err != SSL_ERROR_WANT_READ) && (err != SSL_ERROR_WANT_WRITE))
        {
            errno_=err;
            error_=crypto::err_string();
            return -1;
        }
    }
    if(status_ == CONNECTACK)
    {
        status_=ESTABLISHED;
        out->reset();
        return 1;
    }
    vmp_int retval=bio_read(out);
    if(retval < 0)
    {
        set_errno(ret);
        return -1;
    }
    out->newsize((vmp_size)retval);
    out->index();
    if(SSL_is_init_finished(ssl_))
        status_=CONNECTACK;
    return 0;
}

vmp_int SslBio::accept(vmp::Buf *in,vmp::Buf *out)
{
    except_ctx("accept");
    vmp::except_check_pointer((void *)in,"crypto::SslBio::accept(in=Null)");
    vmp::except_check_pointer((void *)out,"crypto::SslBio::accept(out=Null)");
    if(status_ == INIT)
    {
        ssl_=SSL_new(ctx_.ctx_);
        rbio_=BIO_new(BIO_s_mem());
        wbio_=BIO_new(BIO_s_mem());
        SSL_set_accept_state(ssl_);
        SSL_set_bio(ssl_,rbio_,wbio_);
        status_=ACCEPT;
        ctype_=2;
    }
    else if(status_ != ACCEPT)
        vmp::except("crypto::SslBio::accept() wrong function call");
    BIO_write(rbio_,in->pointer(),in->size());
    vmp_int ret=SSL_do_handshake(ssl_);
    if(ret < 0)
    {
        vmp_int err=SSL_get_error(ssl_,ret);
        if((err != SSL_ERROR_WANT_READ) && (err != SSL_ERROR_WANT_WRITE))
        {
            errno_=err;
            error_=crypto::err_string();
            return -1;
        }
    }
    vmp_int retval=bio_read(out);
    if(retval < 0)
    {
        set_errno(ret);
        return -1;
    }
    out->newsize((vmp_size)retval);
    out->index();
    if(SSL_is_init_finished(ssl_))
    {
        status_=ESTABLISHED;
        return 1;
    }
    return 0;
}

vmp_bool SslBio::is_client()
{
    return (ctype_ == 1);
}
        
vmp_bool SslBio::is_server()
{
    return (ctype_ == 2);
}

vmp_int SslBio::recv(vmp::Buf *in,vmp::Buf *out)
{
    except_ctx("recv");
    vmp::except_check_pointer((void *)in,"crypto::SslBio::recv(in=Null)");
    vmp::except_check_pointer((void *)out,"crypto::SslBio::recv(out=Null)");
    if(status_ == SHUTDOWN)
    {    
        close();
        return -2;
    }
    if(status_ != ESTABLISHED)
        vmp::except("crypto::SslBio::recv() wrong function call");
    if(in->size() == 0)
        return 0;
    BIO_write(rbio_,in->pointer(),in->size());
    out->reset();
    vmp_int n;
    vmp_size s;
    vmp_int retval=0;
    while(1)
    {
        s=vmp::BUFSIZEMAX-out->size();
        if(s == 0)
            break;
        else if(s > vmp::MAXSIZEREAD)
            s=vmp::MAXSIZEREAD;
        out->newsize(out->size()+s);
        n=SSL_read(ssl_,out->pointer(retval),s);
        if(n < 0)
            break;
        retval+=n;
        if((vmp_size) n < s)
           break;
    }
    if(n < 0)
    {
        out->reset();
        set_errno(n);
        return -1;
    }
    if(retval == 0)
    {
        shutdown(out);
        out->index();
        return -2; 
    }
    out->newsize((vmp_size)retval);
    out->index();
    return out->size();
}

vmp_int SslBio::send(vmp::Buf *in,vmp::Buf *out)
{
    except_ctx("send");
    vmp::except_check_pointer((void *)in,"crypto::SslBio::send(in=Null)");
    vmp::except_check_pointer((void *)out,"crypto::SslBio::send(out=Null)");
    if(status_ != ESTABLISHED)
         vmp::except("crypto::SslBio::send() wrong function call");
    vmp_int n=SSL_write(ssl_,in->pointer(),in->size());
    vmp_int ret=SSL_get_error(ssl_,n);
    if(ret != SSL_ERROR_NONE)
    {
        out->reset();
        set_errno(n);
        return -1;
    }
    vmp_int retval=bio_read(out);
    out->index();
    return retval;
}

vmp_bool SslBio::key_update()
{
    except_ctx("key_update");
    if(status_ != ESTABLISHED)
        vmp::except("crypto::SslBio::recv() wrong function call");
    vmp_int ret;
    if(SSL_version(ssl_) <= TLS1_2_VERSION)
        ret=SSL_renegotiate_abbreviated(ssl_);
    else
        ret=SSL_key_update(ssl_,SSL_KEY_UPDATE_REQUESTED);
    return (vmp_bool) ret;
}

vmp_int SslBio::shutdown(vmp::Buf *out)
{
    except_ctx("shutdown");
    vmp::except_check_pointer((void *)out,"crypto::SslBio::shutdown(out=Null)");
    if(status_ != ESTABLISHED)
         vmp::except("crypto::SslBio::shutdown() wrong function call");
    status_=SHUTDOWN;
    vmp_int ret=SSL_shutdown(ssl_);
    if(ret < 0)
    {
        out->reset();
        set_errno(ret);
        return -1;
    }
    bio_read(out);
    out->index();
    return out->size();
}

void SslBio::close()
{
    except_ctx("close");
    error_="Success";
    status_=CLOSE;
}

}}

