/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 05/07/2025
 */

#include "crypto.h"

namespace vampiria { namespace crypto {

Ssl::Ssl()
{
    ssl_=0;
    setting_=false;
}

Ssl::~Ssl()
{
    reset();
}

void Ssl::except_null_pointer(vmp::str func)
{
    vmp::str err;
    vmp::unicode::str_write(&err,"crypto::Ssl::%s() null ssl connection",func.c_str());
    vmp::except_check_pointer((void *)ssl_,err);
}

void Ssl::close()
{
    except_null_pointer("close");
    closed_=true;
}

void Ssl::reset()
{
    errno_=0;
    closed_=false;
    if(setting_)
    {
        setting_=false;
        SSL_clear(ssl_);
        SSL_free(ssl_);
        ssl_=0;
    }
}

void Ssl::set(crypto::Ctx *ctx,net::Socket sock)
{
    reset();
    ctx_.copy(ctx);
    vmp::except_check_pointer((void *)ctx->ctx_,"crypto::Ssl::set(ctx=NULL) ");
    ssl_=SSL_new(ctx->ctx_);
    if(SSL_set_fd(ssl_,sock) == 1)
        setting_=true;
    else
    {
        SSL_free(ssl_);
        ssl_=0;
        vmp::except("crypto::Ssl::set() '%s'",crypto::err_string().c_str()); 
    }
    SSL_set_options(ssl_,SSL_OP_IGNORE_UNEXPECTED_EOF);
}

net::Socket Ssl::get_socket()
{
    except_null_pointer("get_socket");
    net::Socket sock=SSL_get_fd(ssl_);
    if(sock == -1)
        return -1;
    return sock;
}

vmp_int Ssl::get_errno()
{
    return errno_;
}

crypto::Ctx *Ssl::ctx()
{
    except_null_pointer("ctx");
    return &ctx_;
}

void Ssl::get_local_x509(crypto::X509_Wrap *cout)
{
    ctx_.get_x509(cout);
}

void Ssl::get_peer_x509(crypto::X509_Wrap *cout)
{
    vmp::except_check_pointer((void *)cout,"crypto::Ssl::get_peer_cerificate(cout=NULL)");
    except_null_pointer("get_peer_cerificate");
    STACK_OF(X509) *chain;
    chain=SSL_get_peer_cert_chain(ssl_);
    vmp::except_check_pointer((void *)chain,"No certificate was presented by the peer or no connection was established");
    vmp_size num = sk_X509_num(chain);
    if(num == 0)
    {
        X509 *x509=SSL_get_peer_certificate(ssl_);
        vmp::except_check_pointer((void *)x509,"No certificate was presented by the peer or no connection was established");
        cout->set(x509);
    }
    else
    {
        crypto::X509_Wrap *tmp;
        for(vmp_index i=0;i<num;i++)
        {
            if(i == 0)
            {
                cout->set(sk_X509_value(chain,i));
                tmp=cout;
            }
            else
            {
                tmp->parent_=new X509_Wrap();
                tmp=tmp->parent_;
                tmp->set(sk_X509_value(chain,i));
            }
        }
    }
}

vmp_int Ssl::send(vmp::Buf *buf)
{
    vmp::except_check_pointer((void *)buf,"crypto::Ssl::send(buf=NULL)");
    except_null_pointer("send");
    if(closed_)
        return 0;
    vmp_index index=0,ret;
    while(index < buf->size())
    {
        ret=SSL_write(ssl_,(void *)buf->pointer(index),buf->size()-index);
        index+=ret;
        if(ret <= 0)
        {
            errno_=SSL_get_error(ssl_,ret); 
            close();
            return -1;
        }
    }
    return index+1;
}

vmp_int Ssl::recv(vmp::Buf *buf)
{
    vmp::except_check_pointer((void *)buf,"crypto::Ssl::recv(buf=NULL)");
    except_null_pointer("recv");
    if(closed_)
        return 0;
    vmp_size s;
    vmp_int tmp,retval=0;
    buf->reset();
    while(1)
    {
        s=vmp::BUFSIZEMAX-buf->size();
        if(s == 0)
            break;
        else if(s > vmp::MAXSIZEREAD)
            s=vmp::MAXSIZEREAD;
        buf->newsize(buf->size()+s);
        tmp=SSL_read(ssl_,buf->pointer(retval),s);
        if(tmp == -1)
        {
            vmp_uint err= SSL_get_error(ssl_,tmp);
            if((err == SSL_ERROR_WANT_READ) || (err == SSL_ERROR_WANT_WRITE))
            {    
                if(retval == 0)
                    return -2;
                else
                    break;
            }
            else if(err == SSL_ERROR_ZERO_RETURN)
            {
                SSL_shutdown(ssl_);
                close();
                return 0;
            }
            buf->reset();
            errno_=err;
            close();
            return -1;
        }
        retval+=tmp;
        if((vmp_size) tmp < s)
            break;
    }
    if(retval != 0)
        buf->newsize((vmp_size)retval);
    else
        buf->reset();
    buf->index();
    return retval;
}

vmp_int Ssl::key_update()
{
    vmp_int ret;
    except_null_pointer("key_update");
    if(closed_)
        return -1;
    net::Socket sock=get_socket();
    if(sock == -1)
        return -1;
    vmp::fd_block(sock);
    if(SSL_version(ssl_) <= TLS1_2_VERSION)
        ret=SSL_renegotiate_abbreviated(ssl_);
    else
        ret=SSL_key_update(ssl_,SSL_KEY_UPDATE_REQUESTED);
    vmp::fd_noblock(sock);
    if(ret == 0)
        return -1;
    return 0;
}

vmp_int Ssl::connect()
{
    except_null_pointer("connect");
    vmp_int ret=SSL_connect(ssl_);
    if(ret == 1)
        return 0;
    vmp_int err= SSL_get_error(ssl_,ret);
    if((err == SSL_ERROR_WANT_READ) || (err == SSL_ERROR_WANT_WRITE))
        return -2;
    errno_=err;
    close();
    return -1;
}

vmp_int Ssl::accept()
{
    except_null_pointer("accept");
    vmp_int ret=SSL_accept(ssl_);
    if(ret == 1)
        return 0;
    vmp_uint err= SSL_get_error(ssl_,ret);
    if((err == SSL_ERROR_WANT_READ) || (err == SSL_ERROR_WANT_WRITE))
        return -2;
    errno_=err;
    close();
    return -1;
}

vmp_int Ssl::shutdown()
{
    except_null_pointer("shutdown");
    if(closed_)
        return 0;
    vmp_int ret=SSL_shutdown(ssl_);
    if(ret >= 0)
    {
        close();
        return 0;
    }
    errno_=SSL_get_error(ssl_,ret);
    close();
    return -1;
}

}}

