/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 30/06/2025
 */
 
#ifndef VAMPIRIA_CRYPTO_EVENTSSL_H

#define VAMPIRIA_CRYPTO_EVENTSSL_H 1

namespace vampiria { namespace crypto {

typedef void (*TIMERCB)(void *ref);/*!< Ssl timer references*/

//! emepty callback for timer event
void timercb_empty(void *ref);

//! Timer event for additional operations in the connections
class SslTimerRef:public event::EventTimer
{
    public:
        //! A constructor
        SslTimerRef();
        
        //! A destructor
        ~SslTimerRef();
        
        crypto::TIMERCB event_;/*!< Event Ssl timer references*/
        crypto::TIMERCB close_;/*!< Close Ssl timer references*/
        void *ref_;/*!<reference associated with the event */
        vmp::str identity_;/*!<Identity event*/
        
        //! Reset object
        void reset();
};

//!Class shared by all ssl connections
class SslCommon:public event::UI
{
    private:
        vmp::utils::Storage<crypto::SslTimerRef> tref_;/*!<Storage ssl timer event*/
    public:
        //! A constructor
        /*!
            @param manager event timer manager
        */
        SslCommon(event::Manager *manager);
        
        //! A destructor
        ~SslCommon();
        
        crypto::Ctx ctx_;/*!<Ssl context associated*/
        
        vmp::time::Time ctimeout_;/*!<Connection timeout(default 3s)*/
        vmp_uint ckeyupdate_;/*!<Number of minutes I need to spend updating the session key(client session).if value == 0 not active update*/
        vmp_uint skeyupdate_;/*!<Number of minutes I need to spend updating the session key(server session).if value == 0 not active update */
        event::EVTCB tcpconnect_;/*!<Called when a connection tcp connect is complete*/
        net::EVTCBACCEPT tcpaccept_;/*!<Tcp server accept callback.Called when a connection tcp is accepted*/
        event::EVTCB sslconnect_;/*!<Called when client ssl handshake is complete*/
        event::EVTCB sslaccept_;/*!<Called when server ssl handshake is complete*/
        event::EVTCBBUF crecv_;/*!<Recv data callback for client*/
        event::EVTCBBUF srecv_;/*!<Recv data callback for server*/
        event::EVTCB    lclose_;/*!<Close listen connection callback*/
        event::EVTCB    cclose_;/*!<Close client connection callback*/
        event::EVTCB    sclose_;/*!<Close server connection callback*/
        
        //! init object
        /*!
            @param ctx ssl context associated with common connection 
            @param ctimeout connection timeout(setting if value >=0)
        */
        void set_ssl_init(crypto::Ctx *ctx,vmp::time::Time ctimeout);
        
        //! Set Number of minutes I need to spend updating the session key
        /*!
            @param ckeyupdate client time routine(if == 0 disable routine)
            @param skeyupdate server time routine(if == 0 disable routine)
        */
        void set_ssl_keyupdate(vmp_uint ckeyupdate,vmp_uint skeyupdate);
        
        //! Set ssl event client(if input value = 0 doesn't know anything)
        /*!
            @param tcpconnect called when a connection tcp connect is complete
            @param sslconnect called when client ssl handshake is complete
            @param crecv called when client recv data
            @param cclose called when the client is close
        */
        void set_ssl_client_event(event::EVTCB tcpconnect,event::EVTCB sslconnect,event::EVTCBBUF crecv,event::EVTCB cclose);
        
        //! Set ssl event server(if input value = 0 doesn't know anything)
        /*!
            @param tcpaccept called when tcp server accept connection
            @param sslaccept called when server ssl handshake is complete
            @param srecv called when server recv data
            @param lclose called when the server listen close connection
            @param sclose called when the server server close connection
        */
        void set_ssl_server_event(net::EVTCBACCEPT tcpaccept,event::EVTCB sslaccept,event::EVTCBBUF srecv,event::EVTCB lclose,event::EVTCB sclose);
        
        packet::websocket::FramingExtManager framing_ext_;/*!<framing extensions,add the frame extensions here*/
        vmp_size framing_maxsize_;/*!<Max frame size accepted(framing protocol)*/
        vmp_size framing_maxn_;/*!<Max frames number frame accepted for message(frame protocol)*/
        vmp::time::Time framing_rtimeout_;/*!<Routine timeout for event that controls via ping pong if the remote socket is still active(Default 60.0)(framing protocol)*/
        vmp::time::Time framing_ptimeout_;/*!<Ping timeout(Default 5.0)(framing protocol)*/
        vmp_size framing_pinglenbody_;/*!<Number of bytes of ping body websocket framing(framing protocol)(default 120 byte)*/
        
        event::EVTCBBUF framing_recv_text_;/*!<Function performed when a complete package text was received by framing protocol*/
        event::EVTCBBUF framing_recv_bin_;/*!<Function performed when a complete
        package binary was received by framing protocol*/
        
        //! Set parameters for communication frames
        /*!
            @param maxframesize max frame size accepted(if value == 0 not setting nothing)   
            @param maxnframes max frames number frame accepted for message(if value == 0 not setting nothing)
            @param rtimeout routine timeout for event that controls via ping pong if the remote socket is still active(if value <= 0.0 not setting nothing)   
            @param ptimeout Ping timeout(if value <= 0.0 not setting nothing)
            @param pinglenbody Number of bytes of ping body websocket framing(if value == 0 not setting nothing)
            @return void or except in case of failure
        */
        void set_ssl_framing_config(vmp_size maxframesize,vmp_size maxnframes,vmp::time::Time rtimeout,vmp::time::Time ptimeout,vmp_size pinglenbody);
        
        //! Set recv callback function for framin protocol. if value == 0 not setting callback.
        /*!
            @param recv_text function performed when text data is received
            @param recv_bin function performed when text binary data is recei
        */
        void set_ssl_framing_cb(event::EVTCBBUF recv_text,event::EVTCBBUF recv_bin);
        
        //!Virtual class implemented
        vmp::str identity(event::Cell *cell);
 
        //!Virtual class implemented
        void close_event(event::Cell *cell);
        
        //!Virtual class implemented
        void free_ref(event::Cell *cell);
        
        //!Create a new event routine
        /*!
            @param  event callback performed on active routine event
            @param  close function performed at the end of the routine event
            @param  ref object references associated with the event
            @param  identity event identity
            @param  timeval time interval to active event(> 0.0 active timer, = 0.0 create timer but not active, < 0.0 except error)
            @return the new cell associated with the event(except error)
        */
        event::Cell *routine(crypto::TIMERCB event,crypto::TIMERCB close,void *ref,vmp::str identity,vmp::time::Time timeval=0.0);
        
        //!Active timer event.If already active set the new time(> 0.0 active timer, = 0.0 deactive timer ,< 0.0 except error)
        /*!
            @param cell cell event
            @param timeval time interval to active event
            @return void or except in case of failure
        */
        void routine_active(event::Cell *cell,vmp::time::Time timeval);
        
        //!Deactive timer event.To reactivate it, use evt_timer_active
        /*!
            @param cell cell event
        */
        void routine_deactive(event::Cell *cell);
       
        //! Check to see if the routine is active
        /*!
            @param cell cell event
            @return true timer is active otherwise false
        */
        vmp_bool routine_isactive(event::Cell *cell);
};

//!Interface for subprotocols managed by an ssl connection
class EventSslSub
{
    public:
        //! A constructor
        EventSslSub();
        
        //! A destructor
        virtual ~EventSslSub();
        
        event::Cell *cell_;/*!<eventSsl event cell*/
        
        vmp::str sessiontype_;/*!<type of session used*/
        vmp::str sessionid_;/*!<session id*/
        vmp_uint sessionpermits_;/*!<session permits*/
        
        event::EVTCBBUF recv_;/*!<subevent recv callback*/
        event::EVTCB close_;/*!<subevent close callback*/
        
        //! Reset ssl structure
        void evtsub_ssl_reset();
        
        //! Recv data from peer
        /*!
            @param recv buffer data
        */
        void evtsub_ssl_recv(vmp::Buf *buf);
        
        //! Setting new session for client
        /*!
            @param permits peer permits
            @return string pair (subprotocols string,session string) data to send peer or except in case of failure.if session == "" empty session setting
        */
        virtual vmp::pair<vmp::str,vmp::str> evtsub_session_client(vmp_uint permits)=0;
        
        //! Buffers the data generated by evtsub_session_client to be sent to the server(session data).
        /*!
            @param data value returned from evtsub_session_client
            @param out  pointer to output buffer
            @return void or except in case of failure
        */
        virtual void evtsub_session_client_buf(vmp::pair<vmp::str,vmp::str> data,vmp::Buf *out)=0;
        
        //! Setting new session for server
        /*!
            @param pemits peer permits
            @param subprotocols peer client subprotocols
            @param session peer client session.if session == "" empty session setting
            @return string pair (subprotocols string,session string) data to send peer or except in case of failure
        */
        virtual vmp::pair<vmp::str,vmp::str> evtsub_session_server(vmp_uint permits,vmp::str subprotocols,vmp::str session)=0;
        
        //! //! Buffers the data generated by evtsub_session_server to be sent to the client(session data).
        /*!
            @param data value returned from evtsub_session_server
            @param out  pointer to output buffer
            @return void or except in case of failure
        */
        virtual void evtsub_session_server_buf(vmp::pair<vmp::str,vmp::str> data,vmp::Buf *out)=0;
        
        //! Confirm the data sent from the server to the client
        /*!
            @param subprotocols peer server subprotocols
            @param session peer server session
            @return string pair (subprotocols string,session string) data to send peer or except in case of failure
        */
        virtual void evtsub_session_client_confirm(vmp::str subprotocols,vmp::str session)=0;
        
        //!Close event
        void evtsub_ssl_close();
        
        //! Call from evtsub_ssl_close()
        virtual void evtsub_ssl_close_impl()=0;
        
        //!Free event
        void evtsub_ssl_free();
        
        //! Call from evtsub_ssl_free()
        virtual void evtsub_ssl_free_impl()=0;
};

const vmp_int CONN_NONE=0;/*!<Connection Type not connected*/
const vmp_int CONN_SSLLISTEN=1;/*!<Connection Type ssl listen*/
const vmp_int CONN_SSLSERVER=2;/*!<Connection Type ssl server*/
const vmp_int CONN_SSLCLIENT=3;/*!<Connection Type ssl client*/
const vmp_int CONN_SSLPROXY=4;/*!<Connection Type ssl client via proxy mode*/

const vmp_int LINK_NORMAL=0;/*!<Link Type normal*/
const vmp_int LINK_WSFRAMING=1;/*!<Link Type use websocket framing protocol*/

//!Ssl class event(see framework event)
class EventSsl:public net::EventConnection
{
    private:
        //! Reset Ssl event
        void evt_ssl_reset(vmp_bool init=false);
    public:
        //! A Constructor
        EventSsl();

        //! A Destructor
        ~EventSsl();
       
        crypto::SslCommon *common_;/*!<Shared object for all events*/
        crypto::Ssl ssl_;/*!<Ssl connection associated*/
        vmp::time::Time ctimeout_;/*!<Connection timeout*/
        
        packet::websocket::FramingHelper framing_;/*!<Framing protocol helper*/
        vmp_int linktype_;/*!<The type of communication used for the link*/
        vmp::time::Time timeping_;/*!<Time when the ping was sent*/
        
        crypto::EventSslSub *sub_;/*!<subprotocols managed*/
        event::EVTCBBUF sslsend_;/*!<Call from evt_ssl_send,used for subprotocols*/
         
        //! Internal usage,active and disactive connection timeout
        /*!
            @param active if true active connection timeout,if false restore normal conditions
        */
        void evt_ssl_ctimeout(vmp_bool active);
        
        //!Create a new event ssl client
        /*!
            @param ui user interface (see Class UI)
            @param peer remote server address
            @param common shared object for all events
            @param proxy contains the information of proxy connections(0 no proxy connection)
            @return the new cell associated with the event or except in case of failure
        */
        event::Cell *evt_ssl_client(event::UI *ui,net::Address *peer,crypto::SslCommon *common,net::ProxyChain *proxy=0);
       
        //!Create a new event ssl server
        /*!
            @param ui user interface (see Class UI)
            @param local address to listen server connection
            @param common shared object for all events
            @param backlog the maximum length to which the  queue  of pending  connections
            @return the new cell associated with the event or except in case of failure
        */
        event::Cell *evt_ssl_listen(event::UI *ui,net::Address *local,crypto::SslCommon *common,vmp_uint backlog);
        
        //! Renegotiate the session key with the peer
	/*!
	    @param cell event cell
	    @return void or except in case of failure
	*/
	void evt_ssl_key_update();
	
	//! Internal usage active routine event key update
	void evt_ssl_key_update_routine();
	
        //! Returns ssl connection type
        /*!
            @return connection type
        */
        vmp_int evt_ssl_type();
        
        //! Returns connection type in string format
        /*!
            @return connection type in string format
        */
        vmp::str evt_ssl_strtype();
        
        //!Gets the local's certificate
        /*!
            @param cell cell associated
            @param cout certificate output
            @return void or except in case of failure
        */
        void evt_ssl_local_x509(crypto::X509_Wrap *cout);
        
        //!Gets the local's certificate
        /*!
            @param cell cell associated
            @param cout certificate output
            @return void or except in case of failure
        */
        void evt_ssl_peer_x509(crypto::X509_Wrap *cout);
        
        //! Enable ssl connection in framing mode,sslsend_ setting in bin mode,for send text data use evt_ssl_framing_text(protocol used by websockets)
        void evt_ssl_framing_active();
        
        //! Internal usage(Used by callbacks to send frames in sequence)
        void evt_ssl_framing_next();
        
        //! Send a text frame(framing protocol)
        /*!
            @param buf data to send
        */
        void evt_ssl_framing_text(vmp::Buf *buf);
        
        //! Send a binary frame(framing protocol)
        /*!
            @param buf data to send
        */
        void evt_ssl_framing_bin(vmp::Buf *buf);
        
        
        //! Send a connection closure for framing protocol
        /*!
            @param code websocket framing code
            @param message closing string
        */
        void evt_ssl_framing_close(vmp_uint code=packet::websocket::WebSocketCode_Normal,vmp::str msg="");
        
        //! Activate a subprotocol associated with the event
        /*!
            @param sub subprotols object
        */
        void evt_ssl_sub_active(crypto::EventSslSub *sub);
        
        //!Send a package to the remote peer, use the callback in sslsend_. Used to send packages in a generic way.
        /*!
            @ref sslsend_
            @param buf send buffer
            @return void or except in case of failure
        */
        void evt_ssl_send(vmp::Buf *buf);
        
        //! Remove a subprotocol from the connection
        /*!
            @return the subprotocols removed
        */
        crypto::EventSslSub *evt_ssl_sub_reset();
        
        //! Close event
        void evt_ssl_close();
       
        //! Free event
        void evt_ssl_free();
};

//!Shutdown ssl connection(Monitor mode)
/*!
    @param cell Cell with associated ssl event
    @param closelevel close level value
*/
void ssl_shutdown(event::Cell *cell,vmp::str closelevel="");

}}

#endif

