/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 14/09/2024
 */

#include "crypto.h"

namespace vampiria { namespace crypto {

static vmp_uint init_=0;

void broken_pipe(vmp_int)
{
    vmp::utils::syslog_s(vmp::utils::LOG_ERR,"crypto::broken_pipe()");
}

void init()
{
    if(init_==0)
    {
        ERR_load_crypto_strings();
        SSL_load_error_strings();
        OpenSSL_add_all_algorithms();
        if(OPENSSL_init_ssl(0, NULL) < 0)
            vmp::except_s("crypto::init() Could not initialize the library");
        vmp::vector<vmp::str> sig;
        sig.push_back("sigpipe");
        vmp::signal_wrap(sig,crypto::broken_pipe);
    }
    init_++;
}

void end()
{
    if(init_ == 1)
    {
        ERR_free_strings();
        EVP_cleanup();
        CRYPTO_cleanup_all_ex_data();
        ERR_free_strings();
        init_=0;
    }
    else if(init_ > 0)
        init_--;
}

SslUI::SslUI(event::Manager *manager,crypto::Ctx *ctx,vmp::time::Time ctimeout):event::UI(manager)
{
    crypto::init();
    common_=new crypto::SslCommon(manager);
    common_->set_ssl_init(ctx,ctimeout);
}

SslUI::~SslUI()
{
    crypto::end();
    delete common_;
}

vmp::str SslUI::identity(event::Cell *cell)
{
    vmp::str ret="";
    crypto::EventSsl *ssl=cell->event<crypto::EventSsl>();
    vmp_int type=ssl->evt_ssl_type();
    vmp::str strtype=ssl->evt_ssl_strtype();
    net::Address *local=ssl->evt_connection_localaddr(),*peer;
    if(type == crypto::CONN_SSLLISTEN)
        vmp::unicode::str_write(&ret,"%s:[%s:%s]",strtype.c_str(),local->host().c_str(),local->service().c_str());
    else if((type == crypto::CONN_SSLSERVER) || (type == crypto::CONN_SSLCLIENT))
    {
        peer=ssl->evt_connection_peeraddr();
        vmp::unicode::str_write(&ret,"%s:[%s:%s]<->[%s:%s]",strtype.c_str(),local->host().c_str(),local->service().c_str(),
                                                            peer->host().c_str(),peer->service().c_str());
    }
    else if(type == crypto::CONN_SSLPROXY)
    {
        peer=ssl->evt_connection_peeraddr();
        vmp::unicode::str_write(&ret,"%s:[%s:%s]<->[proxy]<->[%s:%s]",strtype.c_str(),local->host().c_str(),local->service().c_str(),
                                                                      peer->host().c_str(),peer->service().c_str());
    }
    return ret;
}

void SslUI::close_event(event::Cell *cell)
{
    crypto::EventSsl *ssl=cell->event<crypto::EventSsl>();
    ssl->evt_ssl_close();
}

void SslUI::free_ref(event::Cell *cell)
{
    crypto::EventSsl *ssl=cell->event<crypto::EventSsl>();
    ssl->evt_ssl_free();
    sref_.free(ssl);
}

event::Event *SslUI::child_event_new(event::Cell *cell)
{
    return (event::Event *) sref_.get();
}

void SslUI::set_event_client(event::EVTCB tcpconnect,event::EVTCB sslconnect,event::EVTCBBUF crecv,event::EVTCB cclose)
{
    common_->set_ssl_client_event(tcpconnect,sslconnect,crecv,cclose);
}

void SslUI::set_event_server(net::EVTCBACCEPT tcpaccept,event::EVTCB sslaccept,event::EVTCBBUF srecv,event::EVTCB lclose,event::EVTCB sclose)
{
    common_->set_ssl_server_event(tcpaccept,sslaccept,srecv,lclose,sclose);
}

void SslUI::set_time_keyupdate(vmp_uint ckeyupdate,vmp_uint skeyupdate)
{
    common_->set_ssl_keyupdate(ckeyupdate,skeyupdate);
}

event::Cell *SslUI::new_client(net::Address *peer,net::ProxyChain *proxy)
{
    event::Cell *cell=0;
    manager_->lock();
    crypto::EventSsl *ssl=sref_.get();
    try
    {
        cell=ssl->evt_ssl_client(this,peer,common_,proxy);
    }
    catch(vmp::exception &x)
    {
        sref_.free(ssl);
        manager_->unlock();
        vmp::except("crypto::SslUI::new_client() '%s'",x.what());
    }
    manager_->unlock();
    return cell;
}

event::Cell *SslUI::new_listen(net::Address *local,vmp_size backlog)
{
    event::Cell *cell=0;
    manager_->lock();
    crypto::EventSsl *ssl=sref_.get();
    try
    {
        cell=ssl->evt_ssl_listen(this,local,common_,backlog);
    }
    catch(vmp::exception &x)
    {
        sref_.free(ssl);
        manager_->unlock();
        vmp::except("crypto::SslUI::new_listen() '%s'",x.what());
    }
    manager_->unlock();
    return cell;
}

void SslUI::send(event::Cell *cell,vmp::Buf *buf)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    crypto::EventSsl *ssl=cell->event<crypto::EventSsl>();
    ssl->evt_connection_send(buf);
    manager->unlock();
}

void SslUI::key_update(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    crypto::EventSsl *ssl=cell->event<crypto::EventSsl>();
    ssl->evt_ssl_key_update();
    manager->unlock();
}

net::Address *SslUI::localaddr(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    crypto::EventSsl *ssl=cell->event<crypto::EventSsl>();
    net::Address *ret=ssl->evt_connection_localaddr();
    manager->unlock();
    return ret;
}

net::Address *SslUI::peeraddr(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    crypto::EventSsl *ssl=cell->event<crypto::EventSsl>();
    net::Address *ret=ssl->evt_connection_peeraddr();
    manager->unlock();
    return ret;
}

net::ProxyChain *SslUI::proxychain(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    crypto::EventSsl *ssl=cell->event<crypto::EventSsl>();
    net::ProxyChain *ret=ssl->evt_connection_proxychain();
    manager->unlock();
    return ret;
}

vmp_int SslUI::connection_type(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    crypto::EventSsl *ssl=cell->event<crypto::EventSsl>();
    vmp_int ret=ssl->evt_ssl_type();
    manager->unlock();
    return ret;
}

vmp::str SslUI::connection_strtype(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    crypto::EventSsl *ssl=cell->event<crypto::EventSsl>();
    vmp::str ret=ssl->evt_ssl_strtype();
    manager->unlock();
    return ret;
}

void SslUI::get_local_x509(event::Cell *cell,crypto::X509_Wrap *cout)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    crypto::EventSsl *ssl=cell->event<crypto::EventSsl>();
    ssl->evt_ssl_local_x509(cout);
    manager->unlock();
}

void SslUI::get_peer_x509(event::Cell *cell,crypto::X509_Wrap *cout)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    crypto::EventSsl *ssl=cell->event<crypto::EventSsl>();
    ssl->evt_ssl_peer_x509(cout);
    manager->unlock();
}

}}

