/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 09/10/2024
 */

#include "crypto.h"

namespace vampiria { namespace crypto {

void wss_tcpaccept(event::Cell *cell,event::Cell *child)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    crypto::EventWss *parent=cell->event<crypto::EventWss>();
    crypto::EventWss *wss=child->event<crypto::EventWss>();
    wss->hosts_=parent->hosts_;
    wss->hostsp_=parent->local_.service();
    wss->localhost_=parent->localhost_;
    manager->unlock();
    crypto::WssCommon *common=(crypto::WssCommon *)parent->common_;
    common->wss_tcpaccept_(cell,child);
}

void wss_auth_default(event::Cell *cell,json::JData *auth)
{
    vmp::str user="";
    vmp_uint permits=0;
    event::Manager *manager=cell->get_manager();
    manager->lock();
    crypto::EventWss *wss=cell->event<crypto::EventWss>();
    crypto::WssCommon *common=(crypto::WssCommon *)wss->common_;
    if(auth->jtype() == json::japi("login"))
    {
        try
        {
            user=auth->get_text("user");
            permits=common->ctx_->verify_acl_user(user,auth->get_text("password"),wss->localhost_,auth->get_bool("md5_password"));
            
        }
        catch(vmp::exception &x)
        {
        }
    }
    manager->unlock();
    wss->wss_utils_continue(user,permits);
}

void wssrecv_http(event::Cell *cell,vmp::Buf *buf)
{
    event::Manager *manager=cell->get_manager();
    vmp_size s;
    vmp_index i,j;
    vmp::str stmp,uritype,urischeme,uri,headers,name,value,ext="",token;
    vmp_uint uitmp;
    vmp::vector<vmp::str> split;
    json::Json jtoken;
    json::JData jdata;
    packet::http::UriHelper huri;
    packet::Packet *request=0,*response=0;
    packet::http::HeadersHelper header;
    vmp::PairList<vmp::str,vmp::str> *pair;
    manager->lock();
    crypto::EventWss *wss=cell->event<crypto::EventWss>();
    crypto::WssCommon *common=(crypto::WssCommon *)wss->common_;
    wss->sub_->sessiontype_="wss";
    wss->httprequest_.write_buf(buf);
    wss->httprequest_.index();
    wss->httpversion_=common->http_versions[0];
    try
    {
        request=common->helper_.read(packet::http::P_HTTP_REQUEST,buf);
        wss->httpversion_=request->get("version");
        if(!vmp::invector<vmp::str>(wss->httpversion_,common->http_versions))
            wss->wss_utils_response_except(&response,packet::http::HttpCode_HTTPVersionNotSupported);
        wss->httpmethod_=request->get("method");
        if(wss->httpmethod_ != "GET")
            wss->wss_utils_response_except(&response,packet::http::HttpCode_MethodNotAllowed);
        uri=request->get("uri");
        if(uri.size() > common->maxsizeuri_)
            wss->wss_utils_response_except(&response,packet::http::HttpCode_URITooLong);
        huri.read(uri);
        uritype=huri.type();
        urischeme=huri.scheme();
        if(!((uritype == "path") || ((uritype == "abs") && ((urischeme == "https") || (urischeme == "wss"))))) 
            vmp::except_s("");
        wss->path_=huri.path();
        if(huri.fragment()->size() != 0)
            vmp::except_s("");
        headers=request->get("headers");
        wss->httpbody_=request->get("body");
        if((headers.size() + wss->httpbody_.size() + 2) > common->maxsizepayload_)
            wss->wss_utils_response_except(&response,packet::http::HttpCode_PayloadTooLarge);    
        header.read(headers);
        pair=header.headers();
        for(i=0;i<pair->size();i++)
        {
            name=pair->get(i).first;
            value=pair->get(i).second;
            if((name.size()+value.size()+4) > common->maxsizeheader_)
                wss->wss_utils_response_except(&response,packet::http::HttpCode_RequestHeaderFieldsTooLarge);   
            if(vmp::unicode::str_casecmp("Host",name))
            {
                split=vmp::unicode::str_split(value,":");
                if(split.size() == 1)
                {
                    wss->host_=split[0];
                    wss->hostp_="80";
                }
                else if (split.size() == 2)
                {
                    wss->host_=split[0];
                    wss->hostp_=split[1];
                }
            }
            else if(vmp::unicode::str_casecmp("Sec-WebSocket-Protocol",name))
                wss->subprotocols_=value;
            else if(vmp::unicode::str_casecmp("Sec-WebSocket-Extensions",name))
            {
                if(ext == "")
                    vmp::unicode::str_write(&ext,"%s",value.c_str());
                else
                    vmp::unicode::str_cwrite(&ext,",%s",value.c_str());
            }
            else if(vmp::unicode::str_casecmp("Origin",name) || vmp::unicode::str_casecmp("Sec-WebSocket-Origin",name))
                wss->wsorigin_=value;
            else if(vmp::unicode::str_casecmp("Sec-WebSocket-Key",name))
            {
                if(wss->wskey_ != "")
                    vmp::except_s("");
                wss->wskey_=value;
            }
            else if(vmp::unicode::str_casecmp("Upgrade",name))
                wss->wsupgrade_=value;
            else if(vmp::unicode::str_casecmp("Connection",name))
                wss->wsconnection_=value;
            else if(vmp::unicode::str_casecmp("Sec-WebSocket-Version",name))
            {
                if(wss->wsversion_ != 0)
                    vmp::except_s("");
                split=vmp::unicode::str_split(value,",");
                for(j=0;j<split.size();j++)
                {
                    uitmp=vmp::unicode::str_todigit_range(vmp::unicode::str_trim(split[j]),1,vmp::INT16MAX);
                    if(vmp::invector<vmp_uint>(uitmp,common->wss_versions) && uitmp > wss->wsversion_)
                        wss->wsversion_=uitmp;
                }
                if(wss->wsversion_ == 0)
                    wss->wss_utils_badversion(&response);
            }
            else if(vmp::unicode::str_casecmp("Cookie",name))
                wss->cookies_=value;
        }
        if(wss->host_ == "" || !vmp::invector<vmp::str>(wss->host_,wss->hosts_) || wss->hostp_ != wss->hostsp_)
            vmp::except_s("");
        if(wss->path_ == "/login")
        {
            if(huri.query()->size() != 1 || !huri.query()->search("auth-token",&token))
                vmp::except_s("");
            json::japi_login_b64_d(&jtoken,token);
            jdata.set(jtoken.root());
            if(!((uritype == "path") || ((uritype == "abs") && (urischeme == "wss")))) 
                wss->wss_utils_upgrade_except(&response);
            if(!vmp::unicode::str_casecmp(wss->wsupgrade_,"websocket"))
                wss->wss_utils_upgrade_except(&response);
            split=vmp::unicode::str_split(wss->wsconnection_,",");
            s=split.size();
            for(i=0;i<s;i++)
            {    
                if(vmp::unicode::str_casecmp(vmp::unicode::str_trim(split[i]),"Upgrade"))
                    break;
            }
            if(i == s)
                wss->wss_utils_upgrade_except(&response);
            if((wss->wsorigin_ == "") || ((wss->origins_.size() != 0)  && !vmp::invector<vmp::str>(wss->wsorigin_,wss->origins_)))
                wss->wss_utils_response_except(&response,packet::http::HttpCode_Forbidden);    
            
            if(wss->subprotocols_.size() == 0) 
                vmp::except_s("");
            if(wss->wskey_ == "")
                wss->wss_utils_upgrade_except(&response);
            wss->framing_.add_exts(ext,true);
        }
        else
        {
            if(huri.query()->size() != 0)
                vmp::except_s("");
            if(wss->wskey_ != "")
                vmp::except_s("");
            wss->wss_utils_web(&response);
        }
    }
    catch(vmp::exception &x)
    {
        if(response == 0)
            response=wss->wss_utils_response_text(packet::http::HttpCode_BadRequest);
    }
    if(request != 0)
        packet::packet_free(request);
    if(response != 0)
    {
         common->helper_.write(response,&wss->httpresponse_);
         wss->evt_connection_send(&wss->httpresponse_);
         wss->httpresponse_.index();
         if(packet::http::HttpCode_isSuccessful(wss->httpcode_))
             manager->cell_close_ok_spec(cell,"crypto::EventWss");
         else
             manager->cell_close_err_spec(cell,"crypto::EventWss",wss->httpcode_,response->get("reason"));
         packet::packet_free(response);
         manager->unlock();   
    }
    else
    {
        manager->unlock();
        common->wss_auth_(cell,&jdata);
    }
}

}}

