/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 02/03/2025
 */

#include <openssl/ssl.h>
#include <openssl/err.h>
#include <openssl/md5.h>
#include <openssl/sha.h>
#include <openssl/x509v3.h>
#include <openssl/bn.h>
#include <openssl/asn1.h>
#include <openssl/evp.h>
#include <openssl/x509.h>
#include <openssl/x509_vfy.h>
#include <openssl/pem.h>
#include <openssl/bio.h>

#include "vmp.h"
#include "net.h"
#include "packet.h"
#include "event.h"
#include "json.h"
#include "db.h"

#include "crypto/lib.h"
#include "crypto/hash.h"
#include "crypto/pkey.h"
#include "crypto/x509.h"
#include "crypto/ctx.h"
#include "crypto/ssl.h"
#include "crypto/sslbio.h"
#include "crypto/cb.h"
#include "crypto/eventssl.h"
#include "crypto/eventp2p.h"
#include "crypto/eventwss.h"

#ifndef VAMPIRIA_CRYPTO_H

#define VAMPIRIA_CRYPTO_H 1

namespace vampiria { namespace crypto {

//Initialized ssl lib
void init();

//Free structure ssl lib
void end();

//! Ssl user interface
class SslUI:public event::UI
{
    private:
        vmp::utils::Storage<crypto::EventSsl> sref_;/*!<Storage event*/

        crypto::SslCommon *common_;/*!<Object containing the common connection configurations */
        
    public:
        //!A Constructor
        /*!
            @param manager input manager
            @param ctx ssl context
            @param ctimeout Connection timeout(setting if value >=0)
        */
        SslUI(event::Manager *manager,crypto::Ctx *ctx,vmp::time::Time ctimeout=0);
        
        //!A Destructorne
        ~SslUI();

        //!Virtual class implemented
        vmp::str identity(event::Cell *cell);

        //!Virtual class implemented
        void close_event(event::Cell *cell);

        //!Virtual class implemented
        void free_ref(event::Cell *cell);

        //!Virtual class implemented
        event::Event *child_event_new(event::Cell *cell);

        //!Set client event(if input value = 0 doesn't know anything)
        /*!
            @param tcpconnect called by the client after the tcp connect is done
            @param sslconnect callback call after client ssl handshake is done
            @param crecv receive client callback
            @param cclose close client callback
        */
        void set_event_client(event::EVTCB tcpconnect,event::EVTCB sslconnect,event::EVTCBBUF crecv,event::EVTCB cclose);
        
        //! Set server event(if input value = 0 doesn't know anything)
        /*!
            @param tcpaccept called by the client after the tcp connect is done
            @param sslsconnect callback call after server ssl handshake is done
            @param srecv receive server callback
            @param lclone close listen event
            @param sclose close connection server
        */
        void set_event_server(net::EVTCBACCEPT tcpaccept,event::EVTCB sslaccept,event::EVTCBBUF srecv,event::EVTCB lclose,event::EVTCB sclose);
        
        //! Set Number of minutes I need to spend updating the session key
        /*!
            @param ckeyupdate client time routine(if == 0 disable routine)
            @param skeyupdate server time routine(if == 0 disable routine)
        */
        void set_time_keyupdate(vmp_uint ckeyupdate,vmp_uint skeyupdate);
        
        //! Add ssl client event. Callback events assigned with set_event_client
        /*!
            @param peer remote server address
            @param proxy proxy chain information,if use proxy
            @return event cell or except in case of failure
        */
        event::Cell *new_client(net::Address *peer,net::ProxyChain *proxy=0);

        //! Add ssl server listen event. Callback events assigned with set_event_server
        /*!
            @param local address listener
            @param backlog max connection server
            @return event cell or except in case of failure
        */
        event::Cell *new_listen(net::Address *local,vmp_size backlog);

        //! Send packet
        /*!
            @param cell event cell
            @param buf packet buffer
	    @return void or except in case of failure
        */
        void send(event::Cell *cell,vmp::Buf *buf);
	
	//! Renegotiate the session key with the peer
	/*!
	    @param cell event cell
	    @return void or except in case of failure
	*/
	void key_update(event::Cell *cell);
	
	//!Gets socket address local
        /*!
            @param cell cell associate
            @return local addres
        */
        net::Address *localaddr(event::Cell *cell);

        //!Gets socket address peer
        /*!
            @param cell cell associated
            @return peer address 
        */
        net::Address *peeraddr(event::Cell *cell);

        //! Gets peer proxy info(for client connection)
        /*!
            @param cell cell associated
            @return proxy info or except in case of failure
        */
        net::ProxyChain *proxychain(event::Cell *cell);

        //! Gets connection type(see EventConnection)
        /*!
            @param cell cell associated
            @return connection type
        */
        vmp_int connection_type(event::Cell *cell);
        
        //! Gets connection type in string format(see EventCoonection)
        /*!
            @param cell cell associated
            @return connection type
        */
        vmp::str connection_strtype(event::Cell *cell);
        
        //!Gets the local's certificate
        /*!
            @param cell cell associated
            @param cout certificate output
            @return void or except in case of failure
        */
        void get_local_x509(event::Cell *cell,crypto::X509_Wrap *cout);
        
        //!Gets the peer's certificate
        /*!
            @param cell cell associated
            @param cout certificate output
            @return void or except in case of failure
        */
        void get_peer_x509(event::Cell *cell,crypto::X509_Wrap *cout);
};

}}

#endif

