/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 24/10/2024
 */

#include "event_.h"

const vmp_int CELLRET_NOCLOSE=-1;
const vmp_int CELLRET_SUCCESS=0;
const vmp_int CELLRET_ERROR=1;
const vmp_int CELLRET_TIMEOUT=2;

struct stub_Manager_cell_close:public pylib::Member_def<event::Manager>
{
    pylib::Objref *def(event::Manager *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        event::Cell *cell;
        vmp_int nretcode;
        event::CELLRET retcode;
        vmp::str errinfo;
        if(args.size() >= 2 && args.size() <=3)
        {
            cell=args.get<event::Cell *>(0);
            nretcode=args.get<vmp_int>(1);
            switch(nretcode)
            {
                case CELLRET_NOCLOSE:
                    retcode=event::NOCLOSE;
                    break;
                case CELLRET_SUCCESS:
                    retcode=event::SUCCESS;
                    break;
                case CELLRET_ERROR:
                    retcode=event::ERROR;
                    break;
                case CELLRET_TIMEOUT:
                    retcode=event::TIMEOUT;
                    break;
                default:
                    args.arg_error();
                    break;
            }
            if(args.size() == 2)
                instance->cell_close(cell,retcode);
            else
            {
                errinfo=args.get<vmp::str>(2);
                instance->cell_close(cell,retcode,errinfo);
            }
            return pylib::retnone();
        }
        return 0;
    }
};

struct stub_Manager_cell_close_wait:public pylib::Member_def<event::Manager>
{
    pylib::Objref *def(event::Manager *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        event::Cell *cell;
        vmp_uint n;
        vmp::str closelevel;
        if(args.size() == 2)
        {
            cell=args.get<event::Cell *>(0);
            n=args.get<vmp_uint>(1);
            instance->cell_closewait(cell,n);
            return pylib::retnone();
        }
        else if(args.size() == 3)
        {
            cell=args.get<event::Cell *>(0);
            n=args.get<vmp_uint>(1);
            closelevel=args.get<vmp::str>(2);
            instance->cell_closewait(cell,n);
            instance->cell_closewait(cell,n,closelevel);
            return pylib::retnone();
        }
        return 0;
    }
};

struct stub_Manager_cell_master:public pylib::Member_def<event::Manager>
{
    pylib::Objref *def(event::Manager *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 1)
        {
            event::Cell *cell=args.get<event::Cell *>(0);
            event::Cell *ret=instance->cell_master(cell);
            if(ret == 0)
                return pylib::retnone();
            return pylib::Converter<event::Cell *>(ret,false); 
        }
        return 0;
    }
};

struct stub_Manager_cell_searchsub:public pylib::Member_def<event::Manager>
{
    pylib::Objref *def(event::Manager *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 2)
        {
            event::Cell *cell=args.get<event::Cell *>(0);
            vmp::str sid=args.get<vmp::str>(1);
            event::Cell *ret=instance->cell_searchsub(cell,sid);
            if(ret == 0)
                return pylib::retnone();
            return pylib::Converter<event::Cell *>(ret,false);
        }
        return 0;
    }
};

void event_init_manager(pylib::Module module)
{
    //event.h 
    pylib::Class<event::Manager> manager=module.class_<event::Manager>("Manager");
    manager.def_("init",&event::Manager::init);
    manager.def_("breakloop",&event::Manager::breakloop);
    manager.def_("lock",&event::Manager::lock);
    manager.def_("unlock",&event::Manager::unlock);
    manager.def_("forced_unlock",&event::Manager::forced_unlock);
    manager.def_("recovery_lock",&event::Manager::recovery_lock);
    manager.def_("cell_update",&event::Manager::cell_update);
    manager.def_("cell_timewait",&event::Manager::cell_timewait);
    manager.def_("cell_get_timewait",&event::Manager::cell_get_timewait);
    manager.def_("cell_get_timeout",&event::Manager::cell_timewait);
    manager.def_with_stub("cell_close",stub_Manager_cell_close());
    manager.def_("cell_close_ok_spec",&event::Manager::cell_close_ok_spec);
    manager.def_("cell_close_err_spec",&event::Manager::cell_close_err_spec);
    manager.def_with_stub("cell_closewait",stub_Manager_cell_close_wait());
    manager.def_("time_now",&event::Manager::time_now);
    manager.def_("cell_alloc",&event::Manager::cell_alloc);
    manager.def_("cell_release",&event::Manager::cell_release);
    manager.def_("cell_set_priority",&event::Manager::cell_set_priority);
    manager.def_("cell_get_priority",&event::Manager::cell_get_priority);
    manager.def_("stop",&event::Manager::stop);
    manager.def_("loop",&event::Manager::loop);
    manager.def_with_stub("cell_master",stub_Manager_cell_master());
    manager.def_("cell_subid",&event::Manager::cell_subid);
    manager.def_("cell_addsub",&event::Manager::cell_addsub);
    manager.def_with_stub("cell_searchsub",stub_Manager_cell_searchsub());
    manager.def_("cell_delsub",&event::Manager::cell_delsub);
}

struct stub_cell_close:public pylib::Member_def<event::Cell>
{
    pylib::Objref *def(event::Cell *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
         if(args.size() == 0)
         {
             instance->close();
             return pylib::retnone();
         }
         else if(args.size() == 1)
         {
             vmp::str closelevel=args.get<vmp::str>(0);
             instance->close(closelevel);
             return pylib::retnone();
         }
         return 0;
    }
};

struct stub_cell_close_wait:public pylib::Member_def<event::Cell>
{
    pylib::Objref *def(event::Cell *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        vmp_uint n;
        if(args.size() == 1)
        {
            n=args.get<vmp_uint>(0);
            instance->close_wait(n);
            return pylib::retnone();
        }
        else if(args.size() == 2)
        {
            n=args.get<vmp_uint>(0);
            vmp::str closelevel=args.get<vmp::str>(1);
            instance->close_wait(n,closelevel);
            return pylib::retnone();
        }
        return 0;
    }
};

struct stub_cell_ret:public pylib::Member_def<event::Cell>
{
    pylib::Objref *def(event::Cell *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 0)
        {
            vmp_int ret=-1;
            switch(instance->ret())
            {
                case event::SUCCESS:
                    ret=CELLRET_SUCCESS;
                    break;
                case event::ERROR:
                    ret=CELLRET_ERROR;
                    break;
                case event::TIMEOUT:
                    ret=CELLRET_TIMEOUT;
                    break;
                default: 
                    ret=CELLRET_NOCLOSE;
                    break;
            }
            return pylib::Converter<vmp_int>(ret); 
        }
        return 0;
    }
};

struct stub_cell_setvar:public pylib::Member_def<event::Cell>
{
    pylib::Objref *def(event::Cell *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 2)
        {
            vmp::str key=args.get<vmp::str>(0);
            pylib::PyObj *obj=new pylib::PyObj();
            args.get_obj(1,(*obj));
            try
            {
                instance->setvar<pylib::PyObj>(key,obj);
            }
            catch(vmp::exception &x)
            {
                delete obj;
                vmp::except_s(x.what());
            }
            return pylib::retnone(); 
        }
        return 0;
    }
};

struct stub_cell_getvar:public pylib::Member_def<event::Cell>
{
    pylib::Objref *def(event::Cell *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 1)
        {
            vmp::str key=args.get<vmp::str>(0);
            pylib::PyObj *ret=instance->getvar<pylib::PyObj>(key);
            if(ret == 0)
                return pylib::retnone();
            return pylib::Converter<pylib::Objref *>(ret->ref());
        }
        return 0;
    }
};

struct stub_cell_remvar:public pylib::Member_def<event::Cell>
{
    pylib::Objref *def(event::Cell *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 1)
        {
            vmp::str key=args.get<vmp::str>(0);
            pylib::PyObj *ret=instance->remvar<pylib::PyObj>(key);
            if(ret == 0)
                return pylib::retnone();
            pylib::Objref *rvalue=pylib::Converter<pylib::Objref *>(ret->ref());
            delete ret;
            return rvalue;
        }
        return 0;
    }
};

struct stub_cell_master:public pylib::Member_def<event::Cell>
{
    pylib::Objref *def(event::Cell *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 0)
        {
            event::Cell *ret=instance->master();
            if(ret == 0)
                return pylib::retnone();
            return pylib::Converter<event::Cell *>(ret,false); 
        }
        return 0;
    }
};

struct stub_cell_searchsub:public pylib::Member_def<event::Cell>
{
    pylib::Objref *def(event::Cell *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 1)
        {
            vmp::str sid=args.get<vmp::str>(0);
            event::Cell *ret=instance->searchsub(sid);
            if(ret == 0)
                return pylib::retnone();
            return pylib::Converter<event::Cell *>(ret,false);
        }
        return 0;
    }
};

void event_init_cell(pylib::Module module)
{
    //CELLRET enum in event.h
    module.add_const<vmp_int>("CELLRET_NOCLOSE",CELLRET_NOCLOSE);
    module.add_const<vmp_int>("CELLRET_SUCCESS",CELLRET_SUCCESS);
    module.add_const<vmp_int>("CELLRET_ERROR",CELLRET_ERROR);
    module.add_const<vmp_int>("CELLRET_TIMEOUT",CELLRET_TIMEOUT);
    
    //max event priority
    module.add_const<vmp_uint>("MAX_CELL_PRIORITY",event::MAX_CELL_PRIORITY);

    //event.h
    pylib::Class<event::Cell> cell=module.class_no_constructor<event::Cell>("Cell");
    cell.def_("identity",&event::Cell::identity);
    cell.def_("id",&event::Cell::id);
    cell.def_with_stub("close",stub_cell_close());
    cell.def_with_stub("close_wait",stub_cell_close_wait());
    cell.def_("close_err",&event::Cell::close_err);
    cell.def_("close_err_spec",&event::Cell::close_err_spec);
    cell.def_("alloc",&event::Cell::alloc);
    cell.def_("release",&event::Cell::release);
    cell.def_("set_priority",&event::Cell::set_priority);
    cell.def_("get_priority",&event::Cell::get_priority);
    cell.def_with_stub("ret",stub_cell_ret());
    cell.def_("closelevel",&event::Cell::closelevel);
    cell.def_("errcode",&event::Cell::errcode);
    cell.def_("str_error",&event::Cell::str_error);
    cell.def_("set_timewait",&event::Cell::set_timewait);
    cell.def_("get_timewait",&event::Cell::get_timewait);
    cell.def_("get_timeout",&event::Cell::get_timeout);
    cell.def_("get_manager",&event::Cell::get_manager,false);
    cell.def_with_stub("setvar",stub_cell_setvar());
    cell.def_with_stub("getvar",stub_cell_getvar());
    cell.def_with_stub("remvar",stub_cell_remvar());
    cell.def_("all_keys",&event::Cell::all_keys);
    cell.def_with_stub("master",stub_cell_master());
    cell.def_("subid",&event::Cell::subid);
    cell.def_("addsub",&event::Cell::addsub);
    cell.def_with_stub("searchsub",stub_cell_searchsub());
    cell.def_("delsub",&event::Cell::delsub);
}

VAMPIRIA_PYLIB_INIT(vmp_event_);

void vampiria_pylib_init_module(pylib::Module init_module)
{
    event_init_manager(init_module);
    event_init_cell(init_module);
    event_init_inputui(init_module);
    event_init_timerui(init_module);
    event_init_croniterui(init_module);
    event_init_simplexui(init_module);
    event_init_processui(init_module);
}

