/* -*- Mode:Javascript; javscript-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 24/02/2025
 */
/**
  *@return {String} author project
*/
function vmp_env_author(){return VAUTHOR;}
/**
  *@return {String} email author project
*/
function vmp_env_email(){return VEMAIL;}
/**
  *@return {String} web page project
*/
function vmp_env_webpage(){return VWEBPAGE;}
/**
  *@return {String} license project
*/
function vmp_env_license(){return VLICENSE;}
/**
  *@return {String} version project
*/
function vmp_env_version(){return VVERSION;}
/**
  *Convert the input string to lowercase.
  *@param str {String} input string
  *@return {String} lowercase string 
*/
function vmp_unicode_str_tolower(str){return str.toLowerCase();}
/**
  *Convert the input string to uppercase.
  *@param str {String} input string
  *@return {String} uppercase string 
*/
function vmp_unicode_str_toupper(str){return str.toUpperCase();}
/**
  *Split string
  *@param str {String} input string
  *@param delimiter {String} token value
  *@return {Array} split strings
*/
function vmp_unicode_str_split(str,delimiter){return str.split(delimiter);}
/**
  *Creates a string concatenate strings in list, 
  *inserting the token between the strings,if delimiter="" return charatters list
  *@param list {ArrayString} List of strings to be concatenated
  *@param token {String} String to be inserted between the strings to be concatenated
  *@return {String} the resulting string                        
*/
function vmp_unicode_str_join(list,token){return list.join(token);}
/**
  *Converts the string input to int
  *@param istr {String}string input
  *@throws in case of failure
  *@return {Integer} The converted value
*/ 
function vmp_unicode_str_todigit(istr)
{
  let ret=parseInt(istr);
  if(ret === NaN)
    vmp_except(`Invalid Conversion from string to int because '${istr}' not integer`);
  if(Number(istr) != ret)
    vmp_except(`Invalid Conversion from string to int because '${istr}' not integer`);
  return ret;
}
/** 
  *Converts the string input to int in range min-max
  *@param istr {String}string input
  *@param min {Integer} value accepted
  *@param max {Integer} value accepted
  *@throws in case of failure
  *@return {Integer} The converted value
*/ 
function vmp_unicode_str_todigit_range(istr,min,max)
{
  if(min > max)
    vmp_except(`Min value ${min} not less max value ${max} in input`);
  let ret=vmp_unicode_str_todigit(istr);
  if((ret < min) || (ret > max))
    vmp_except(`${istr} not integer in range [${min},${max}]`);
  return ret; 
}
/**
  *Converts the string input to real
  *@param dstr {String} string input
  *@throws in case of failure
  *@return {Number} The converted value
*/ 
function vmp_unicode_str_toreal(dstr)
{
  let ret=Number(istr)
  if (ret == NaN)
    vmp_except(`Invalid Conversion from string to int because '${dstr}' not real`); 
  return ret;
}
/**
  *Converts the string input to real in range min-max
  *@param dstr {String} string input
  *@param min {Number} value accepted
  *@param max {Number} value accepted
  *@throws in case of failure
  *@return {Number} The converted value
*/ 
function vmp_unicode_str_toreal_range(dstr,min,max)
{
  if(min > max)
    vmp_except(`Min value ${min} not less max value ${max} in input`);
  let ret=vmp_unicode_str_toreal(dstr);
  if((ret < min) || (ret > max))
    vmp_except(`'${dstr}' not real in range [${min},${max}]`);
  return ret; 
}
/** 
  *Converts a string of exadecimal values in the byte vector (2 characters 0-F 1 byte)
  *@param xstr {String} string input
  *@throws in case of failure
  *@return {ArrayBytes} byte vector.
*/
function vmp_unicode_xstr_tobytes(xstr)
{
    let ret=new Array();
    let tmp;
    let b=0x00;
    if((xstr.length % 2) == 1)
      tmp=`0${xstr}`;
    else
      tmp=xstr;
    for(let i=0;i<tmp.length;i++)
    {
      switch(tmp[i])
      {
        case '0': b|=0x00;break;
        case '1': b|=0x01;break;
        case '2': b|=0x02;break;
        case '3': b|=0x03;break;
        case '4': b|=0x04;break;
        case '5': b|=0x05;break;
        case '6': b|=0x06;break;
        case '7': b|=0x07;break;
        case '8': b|=0x08;break;
        case '9': b|=0x09;break;
        case 'a': b|=0x0A;break;
        case 'A': b|=0x0A;break;
        case 'b': b|=0x0B;break;
        case 'B': b|=0x0B;break;
        case 'c': b|=0x0C;break;
        case 'C': b|=0x0C;break;
        case 'd': b|=0x0D;break;
        case 'D': b|=0x0D;break;
        case 'e': b|=0x0E;break;
        case 'E': b|=0x0E;break;
        case 'f': b|=0x0F;break;
        case 'F': b|=0x0F;break;
        default:vmp_except(`String ${xstr} is not hexadecimal`);
       }
       if((i%2) == 0)
         b=b<<4;
       else
       {
         ret.push(b);
         b=0x00;
       }    
    }
    return ret;
}
/** 
  *Converts a bytes vector to a string of hexadecimal values
  *@param bytes {ArrayBytes} byte vector input
  *@return {String} The converted string value.
*/
function vmp_unicode_bytes_toxstr(bytes)
{
  let ret="",tmp;
  for(let i=0;i<bytes.length;i++)
  {
    tmp=bytes[i].toString(16);
    if(tmp.length == 1)ret+=`0${tmp}`
    else ret+=tmp;
  }
  return ret;
}
/**
  *Converts a bytes vector to a string of hexadecimal values in human mode(the reverse function does not work)
  *@param bytes byte {ArrayBytes} vector input
  *@param delimiter {String} byte delimiter
  *@return {String} The converted string value in human mode.
*/
function vmp_unicode_bytes_toxstr_hm(bytes,delimiter)
{
  let ret="",tmp;
  for(let i=0;i<bytes.length;i++)
  {
    tmp=bytes[i].toString(16);
    if(tmp.length == 1)tmp=`0${tmp}`;
    if(i == 0)
      ret+=tmp;
    else
      ret+=`${delimiter}${tmp}`;
  }
  return ret;
}
/**
  *Generate substring
  *@param str {String} input string
  *@param pos {Integer} position of the first character to be copied as a substring
  *@param len {Integer} Number of characters to include in the substring
  *@throws in case of failure
  *@return {String} substring 
*/
function vmp_unicode_str_sub(str,pos,len){return str.substring(pos,pos+len);}
/**
  *Search substring sub in str
  *@param str {String} input string
  *@param sub {String} substring to search
  *@return {Boolean} true if found sub in str otherwise false
*/
function vmp_unicode_str_findsub(str,sub){return str.includes(sub);}
/**
  *Replace in text the find occurrences with sub
  *@param text {String} input text
  *@param occurrence {String} to be replaced
  *@param string {String} to be replaced at occurrences
  *@return {String} text with occurrences replaced
*/
function vmp_unicode_str_replace(text,find,sub)
{
  let vtext=vmp_unicode_str_split(text,find);
  return vmp_unicode_str_join(vtext,sub);
}
/**
  *Performs a byte-by-byte comparison of the strings s1 and s2, ignoring the case of the characters
  *@param s1 {String} input string
  *@param s2 {String} input string
  *@return {Boolean} true if the comparison is verified,otherwise false
*/
function vmp_unicode_str_casecmp(str1,str2)
{
  if((str1.length === str2.length) && (vmp_unicode_str_tolower(str1) ===vmp_unicode_str_tolower(str2)))
    return true;
  return false;
}
/**
  *Format the input string by replacing all characters with ascii code <= 0x20 or >= 0x7E with a space
  *@param content {String} string input
  *@return {String} formatted string
*/
function vmp_unicode_str_format(content)
{
  let ret="";
  for(let i=0;i<content.length;i++)
  {
    let c=content.charCodeAt(i);
    if(!(c <= 0x20) || (c >= 0x7E))
      ret +=content[i];        
  }  
  return ret;
}
/**
  *Trims a string for leading and trailing all characters with ascii code <= 0x20 or >= 0x7E.
  *@param str {String} The string to trim
  *@return {String} The input string without leading and trailing spaces
*/
function vmp_unicode_str_trim(str)
{
  let ret=vmp_unicode_str_format(str);
  let s=ret.length;    
  while((s > 0) && (ret[0] == ' '))
  {
    s=s-1;
    ret=vmp_unicode_str_sub(ret,1,s);
  }
  while((s > 0) && (ret[s-1] == ' '))
  {
    s=s-1;
    ret=vmp_unicode_str_sub(ret,0,s);
  }
  return ret;
}
/**
  *Check if a string has ascii code characters <= 0x20 or >= 0x7E(\ n, \ t DEL SPACE etc ..)
  *@param str {String} string input
  *@return {Boolean} true If there are no characters <= 0x20 or >= 7E and word size > 0,otherwise false
*/
function vmp_unicode_str_isword(str)
{
  if(str.length == 0)return false; 
  for(let s of str)
  {
    let c=s.charCodeAt(0)
    if((c <= 0x20)||(c >= 0x7E))
      return false;
    return true;
  }
}
/**
  *Format the input string by replacing all characters with ascii code <= 0x20 with a space or \n if the line size exceeds maxline 
  *@param content {String} string input
  *@param maxline {Integer} max line size
  *@return {String} formatted string
*/
function vmp_unicode_str_format_maxline(content,maxline)
{
  let ret="";
  content=vmp_unicode_str_format(content);
  let split=vmp_unicode_str_split(content," ");
  let line=0;        
  for(let i=0;i<split.length;i++)
  {
    if(line >= maxline)
    {
      ret += "\n";
      line=0;
    }
    else if(i != 0)
    {
      ret += " ";
      line++;                
    }                
    ret += split[i];
    line += split[i].length;        
  }
  return ret;
}
function vmp_unicode_str_format_lines(content,linesize)
{
  let line=0;
  let ret="";       
  for(let i=0;i<content.length;i++)
  {
    if(line > linesize)
    {
      ret += "\n";
      line=0;
    }
    ret += content[i];
    line++;
  }
  return ret;
}
/**
  *Remove all characters with ascii code <= 0x20 or >= 0x7E at the end of the string
  *@param content {String} input string
  *@return {String} formatted string
*/
function vmp_unicode_str_format_end(content)
{
  let i;
  for(i=content.length()-1;i>=0;i--)
  {
    let c=charCodeAt(i);
    if((c > 0x20) && (c <= 0x7E))
      break;
  }
  if(i < 0)
    return "";
  return vmp_unicode_str_sub(content,0,i+1);
}
/** 
  *Extracts all the characters in the string clist from the string str and returns them
  *@param str {String} input string
  *@param clist {String}char extract string
  *@return {String} characters found
*/
function vmp_unicode_str_extract_char(str,clist)
{
  let ret="";
  for(let s of str)
  {
    for(let c of clist)
    {
      if(s === c)
      {
        ret+=s;
        break;
      }
    }
  }
  return ret;
}
/**
  *Removes duplicate strings from the input vector
  *@param svect {ArrayString} input vector
  *@return {ArrayString } a vector with all different strings    
*/
function vmp_unicode_str_remove_duplex(svect)
{
  let tmp={}
  let ret=[]
  for(let s of svect)
  {
    if(!(s in tmp))      
    {
      tmp[s]='';        
      ret.push(s);
    }
  }
  return ret;
}
/**
  *Check if there are regular expressions of types Basic POSIX grammar and case insensitive in a string and return them
  *@param str {String} string to analize
  *@param str_regex {String} regular espression to search
  *@param true {Boolean} use case-insensitive mode, false use case-sensitive mode
  *@return {ArrayString}the vector of all matched strings
*/
function vmp_unicode_str_regex(str,str_regex,icase=false)
{
  if(icase)
    return str.match(str_regex);
  return str.match(str_regex,"i");
}
/**
  *Checks whether string str is verified by regular expression of types Basic POSIX(case-sensitive mode)
  *@param str {String} string matching
  *@param regex {String} regular espression to string matching
  *@return true {Boolean} if matching,otherwise false
*/ 
function vmp_unicode_str_regex_matching(str,str_regex)
{
  let m=vmp_unicode_str_regex(str,str_regex);
  if((m.length == 1) && (m[0] == str))
    return true;
  return false;
}
/**
  *Takes a shell command in string format and splits the arguments
  *@param args {String} arguments in string format
  *@throws in case of failure
  *@return {String} split strings
*/
function vmp_unicode_shlex_split(args)
{
  let ret,token="",state=' ',escape=' ';
  let whitespace="[\\s\\t\\r\\n]";
  let wordchar="[\\w@%+=:,\\./\\-]";
  for(let nextchar of args)
  {
    switch(state)
    {
      case ' ':
        if(vmp_unicode_str_regex_matching(nextchar,whitespace))
        {
          if(token != "")
            ret.push(token);
          token="";
        }
        else if(vmp_unicode_str_regex_matching(nextchar,wordchar))
          token += nextchar;
        else if(nextchar === "\\")
        {
          escape=state;
          state='e';
        }
        else if(nextchar === "'")
          state='s';
        else if(nextchar === "\"")
          state='d';
        else
          vmp_except(`Args '${args}' syntax error`);
        break;
      case 'e'://escape
        token += nextchar;
        state=escape;
        break;
      case 's'://single quoted
        if(nextchar === "'")
          state=' ';
        else if(nextchar === "\\")
        {
          escape=state;
          state='e';
        }
        else
          token += nextchar;
        break;
      case 'd'://double quoted
        if(nextchar === "\"")
          state=' ';
        else if(nextchar === "\\")
        {
          escape=state;
          state='e';
        }
        else
          token += nextchar;
        break;
     }
  }
  if(state != ' ')
    vmp_except( `Args '${args}' syntax error`);
  if(token != "")
    ret.push(token);
  return ret;
}
/**
  *Takes a shell command in vector format and returns it as a string
  *@param {ArrayString} vargs arguments in vector format
  *@throws in case of failure
  *@return {String} resulting string
*/
function vmp_unicode_shlex_join(vargs)
{
  let ret="",value;
  for(let v of vargs)
  {
    if(v.length != 0)
    {
      if(vmp_unicode_str_regex_matching(v,"[\\w@%+=:,\\./\\-]+"))
        value=v;
      else
      {
        value=vmp_unicode_str_replace(v,"\"","\\\"");
        value=`"${value}"`;
      }
    }
    if(ret == "")
      ret=value;
    else
      ret=` ${value}`;
  }
  return ret;
}
/**
  *Input a string in format V1,V2,.. where V? == * (all values)
  *V? == Single digit value , V? == DIGIT1-DIGIT2 interval value,
  *V? == -DIGITMAX interval from minvalue to DIGITMAX, 
  *DIGITMIN- interval from DIGITMIN to maxvalue example(2,3-5,12-,-11)
  *@param istr {String} string input
  *@param minvalue {Integer} min value
  *@param maxvale {Integer} max value
  *@throws in case of failure
  *@return {Arraydigit}an ordered list of digit values without duplication
*/
function vmp_unicode_str_toindex_list(istr,minvalue,maxvalue)
{
  let ret=[],value,split,split2,min,max;
  if(minvalue > maxvalue)
    vmp_except(`minvalue ${minvalue} not less maxvalue ${maxvalue} in input`);
  split=vmp_unicode_str_split(istr,",");
  for(let s of split)
  {
    value=vmp_unicode_str_trim(s);
    try
    {
      if(value == "*")
      {
        min=minvalue;
        max=maxvalue;
      }
      else if(vmp_unicode_str_regex_matching(value,"[0-9]+"))
      {
        min=vmp_unicode_str_todigit_range(value,minvalue,maxvalue);
        max=min;
      }
      else if(vmp_unicode_str_regex_matching(value,"[0-9]+\\-[0-9]+"))
      {
        split2=vmp_unicode_str_split(value,"-");
        min=vmp_unicode_str_todigit_range(split2[0],minvalue,maxvalue);
        max=vmp_unicode_str_todigit_range(split2[1],minvalue,maxvalue);
      }
      else if(vmp_unicode_str_regex_matching(value,"[0-9]+\\-"))
      {
        value=vmp_unicode_str_sub(value,0,value.size()-1);
        min=vmp_unicode_str_todigit_range(value,minvalue,maxvalue);
        max=maxvalue;
      }
      else if(vmp_unicode_str_regex_matching(value,"\\-[0-9]+"))
      {
        value=vmp_unicode_str_sub(value,1,value.size()-1);
        min=minvalue;
        max=vmp_unicode_str_todigit_range(value,minvalue,maxvalue);
      }
      else
        vmp_except("");
      if((min < minvalue) || (max > maxvalue))
        vmp_except(`interval range (${minvalue}-${maxvalue})`);
      for(let j=min;j<=max;j++)
        ret.push(j);
    }
    catch(error)
    {
      vmp_except(`Syntax error '${value}' (${error})`);
    }
  }
  return vmp_sort(ret,false,false);  
}    
/**Internal usage*/
const VMP_BUFSIZEMAX=Number.MAX_VALUE-1;
/**Internal usage*/
const VMP_BUF_SIZE_INIT=32;
/**Internal usage*/
const VMP_BUF_EOB_ERROR="End Of Buf Error";

/**Buffer type */
class Vmp_Buf
{
  /**
    *Create an buffer.
    *@constructor
  */
  constructor()
  {
    this.maxsize_=VMP_BUF_SIZE_INIT;
    this.mem_=new ArrayBuffer(this.maxsize_);
    this.reset();
  }
  /**Internal usage*/
  expand(size)
  {
    if(size > this.maxsize_)
    {
      do
      {
        if(this.maxsize_ > VMP_BUFSIZEMAX/2)this.maxsize_=VMP_BUFSIZEMAX;
        else this.maxsize_ *= 2;  
      }
      while(size > this.maxsize_);
      let old=this.pointer();
      this.mem_=new ArrayBuffer(this.maxsize_);
      let mem=this.pointer();
      for(let i=0;i<this.size_;i++)mem[i]=old[i];  
    }
    if(size > this.size_)
      this.size_=size;
  }
  /** 
    *New size buffer
    *@param size {Size} buffer size                        
  */
  newsize(size)
  {forw
    if(size > this.size_)        
      this.expand(size);
    else if(size < this.size_)
    {        
      let view=this.pointer(size);
      for(let i=0;i<view.length;i++)view[i]=0x00;
      this.size_=size;
    }
  }
  /** 
    *Size buffer
    @return {Size} buffer size                        
  */        
  size(){return this.size_;}
  /** 
    *@return {Size} the number of bytes available for reading.(size()-index())  
  */
  size_reading(){return (size_-index_);}
  /**
    *Set index position
    *@param point {Index} index pointer to setting. If point > size_ setting index_=size_,if point < 0 setting index_=0
  */
  index(point=0){this.index_=(point<this.size_)?point:this.size_;}
  /**
    *@return {Index} current index(End of Buf index == size)
  */
  get_index(){return this.index_}
  /**
    *@return {Boolean} True if end of buf occurred
  */
  eob(){return (this.index_===this.size_)?true:false;}
  /**reset buffer*/
  reset()
  {
    let mem=this.pointer();
    for(let i=0;i<this.size_;i++)
      mem[i]=0x00;
    this.size_=0;
    this.index(); 
  }
  /**
    *Get array buffer pointer memory
    *@param point {Index} first byte pointed
    *@return {Uint8Array} an view Uint8Array
  */
  pointer(point=0){return (point<this.size_)?new Uint8Array(this.mem_,point,this.size_-point):null;}
  /**
    *Check if the buffer contents are equal
    *@param ibuf {Vmp_Buf} input buffer
    *@return {Boolean} true if the buffer contents are equal,otherwise false
  */
  equal(ibuf)
  {
    if(this.size_ === ibuf.size())
    {  
      let b=this.pointer(),bi=ibuf.pointer();
      for(let i=0;i<this.size_;i++)
        if(b[i] != bi[i])
          return false;
    }    
    return true;
  }
  /**
    *Writes the input byte array to the buffer at the this.index_ position and increments the this.index_
    *@param uint8array {Uint8Array} input array
    *@return {Index} bytes written in the buffer
  */
  write(uint8array)
  {
    let size=uint8array.length;
    let bmax=VMP_BUFSIZEMAX-this.index_;
    if(bmax < size)
      size=bmax;
    let nindex=this.index_+size
    this.expand(nindex);
    let mem=this.pointer(this.index_);
    for(let i=0;i<size;i++)mem[i]=uint8array[i];
    this.index(nindex);
    return size;
  }
  /**
    *Reads size bytes from the buffer in the this.index_ position and increments the this.index_ value.
    *@param size {Size} number of bytes to read
    *@throws in case of failure  
    *@return {Uint8Array} byte reading
  */ 
  read(size)
  {
    let nindex=this.index_+size;
    if(nindex > this.size_)
      vmp_except(`${VMP_BUF_EOB_ERROR} read buf size ${size}`);
    let ret=new Uint8Array(this.mem_,this.index_,size);
    this.index(nindex);
    return ret;
  }
  /** 
    *Write input byte to buffer to the this.index_ position and increments the this.index_ value
    *@param value {bytes} input bytes
    *@return {Size} bytes written in the buffer
  */                
  write_byte(byte)
  {
    let arr=new Array(1);
    arr[0]=byte;
    return this.write(new Uint8Array(arr));
  }
  /**
    *Reads size byte from the buffer in the this.index_ position and increments the this.index_ value.
    *@param size {Size} number of bytes to read    
    *@return {Byte} byte reading
  */ 
  read_byte()
  {
    let arr=this.read(1);
    return arr[0];
  }
  /** 
    *Write input array byte to buffer to the this.index_ position and increments the this.index_ value
    *@param value {ArrayBytes} input bytes
    *@return {Size} bytes written in the buffer
  */                
  write_array(bytes){return this.write(new Uint8Array(bytes));}
  /**
    *Reads size byte from the buffer in the this.index_ position and increments the this.index_ value.
    *@param size {Size} number of bytes to read    
    *@return {ArrayBytes} bytes reading
  */ 
  read_array(size)
  {
    let arr=this.read(size);
    let ret=new Array();
    for(let byte of arr)
      ret.push(byte);
    return ret;
  }
  /**
    *Write arraybuffer content to buffer input to the this.index_ position and increments the this.index_ value
    *@param abuf {ArrayBuffer} input arrayBuffer
    *@return {Size} bytes reading
  */
  write_arraybuffer(abuf){return this.write(new Uint8Array(abuf));}
  /**
    *@return Return {ArrayBuffer} a data buffer array from this.index_ size position and increments the this.index_ value
  */
  read_arraybuffer(size)
  {
    let arr=this.read(size);
    let ret=new ArrayBuffer(size);
    let v=new Uint8Array(ret);
    for(let i=0;i<arr.length;i++)
      v[i]=arr[i]
    return ret;
  }
  /** 
    *Write n random bytes in the buffer to the this.index_ position and increments the this.index_ value
    *@param n {Size} bytes number
    *@return {Size} the number of bytes written in the buffer 
  */
  random_bytes(n)
  {
    let arr=new Array();
    for(let i=0;i<n;i++)
      arr.push(vmp_wrap_random(256));
    return this.write(new Uint8Array(arr));
  }
  /**
    *Write size value in the buffer to the this.index_ position and increments the this.index_ value
    *@param value {Size} input size
    *@param dim {Size} number of bytes to write input(1-4)[b8,b7,b6,b5,b4,b3,b2,b1]
    *@return {Size} the number of bytes written in the buffer
  */   
  write_size(value,dim)
  {
    let arr=new Array();
    if(dim < 0)
      dim =0;
    else if(dim > 8)
      dim=8;
    for(let i=0;i<dim;i++)
      arr.push((value >> ((dim-i-1)*8)) & 0xFF);
    return this.write(new Uint8Array(arr)); 
  }
  /**
    *Read size value from the buffer in the this.index_ position and increments the this.index_ value.
    *@param dim {Size} number of bytes to read(1-4)b8,b7,b6,b5,b4,b3,b2,b1]
    *@return {Size} size reading
  */ 
  read_size(dim)
  {
    let ret=0;
    if(dim == 0)
      return ret;
    else if(dim > 8)
      dim=8;
    let arr=this.read(dim);
    for(let i=0;i<dim;i++)
      ret+=(arr[i]<<((dim-i-1)*8));
    return ret;
  }
  /**
    *Write string values in the buffer to the this.index_ position and increments the this.index_ value
    *@param value {String} input string
    *@return {Size} the number of bytes written in the buffer
  */                  
  write_str(value){return this.write(new Uint8Array(new TextEncoder().encode(value)));}
  /**
    *Read string values from the buffer in the this.index_ position and increments the this.index_ value
    *@param {Size} size to read
    *@return {String} string value
  */  
  read_str(size){return new TextDecoder().decode(this.read(size));}
  /**
    *Write string of exadecimal values in the buffer to the this.index_ position and increments the this.index_ value
    *@param value {StringExadecimal} input string
    *@return {Size} bytes written in the buffer
  */                  
  write_xstr(value){return this.write(new Uint8Array(vmp_unicode_xstr_tobytes(value)));}
  /**
    *Read string of exadecimal values from the buffer in the this.index_ position and increments the this.index_ value.
    *@param size {Size} size to read
    *@return {StringExadecimal} string value
  */
  read_xstr(size){return vmp_unicode_bytes_toxstr(this.read(size));}
  /**
    *Read string of exadecimal in human mode values from the buffer in the index_ position and increments the index value.(The reverse function does not work)
    *@param size {Size} size to read
    *@param delimiter {String} byte delimiter
    *@return {StringExadecimal} string value
  */
  read_xstr_hm(size,delimiter){return vmp_unicode_bytes_toxstr_hm(this.read(size),delimiter);}
  /**
    *Concatenating the contents of the input buffer to the end of the buffer.               
    *@param value {Vmp_Buf} input buffer
    *@return {Size} the number of bytes written in the buffer
  */
  cat(value)
  {
    this.index(this.size_);
    let ret=this.write(value.pointer());
    this.index();
    return ret;
  }
  /**
    *Concatenating the contents of the input buffer atRegister a module in the project
 the beginning of the buffer.             
    *@param value {Vmp_Buf} input buffer
    *@return {Size} the number of bytes written in the buffer
  */
  cathead(value)
  {
    let arr=new Array();
    for(let o of this.pointer())
      arr.push(o);
    this.reset();
    this.write(value.pointer());
    this.write(arr);
    this.index();
  }
  /**
    *Write buffer value in the buffer to the this.index_ position and increments the this.index_ value
    *@param value {Vmp_Buf} input buffer
    *@return {Size} the number of bytes written in the buffer
  */
  write_buf(value){return this.write(value.pointer());}
  /**
    *Read buffer value from the buffer in the this.index_ position and increments the this.index_ value
    *@param size {Size} number of bytes to read
    *return {Vmp_Buf} output buffer    
  */ 
  read_buf(size)
  {
    let ret=new Vmp_Buf();
    let tmp=this.read(size);
    ret.write(tmp);
    return ret;
  }
  /**
    *Resizes the buffer set this.index_ position 0
    *@param init {Index} first byte
    *@param len {Size} new buffer size
  */                
  cut(init,len)
  {
    let mtmp=new Uint8Array(this.mem_,init,len);
    let arr=new Array();
    for(let o of mtmp)
      arr.push(o);
    this.reset();
    return this.write(new Uint8Array(arr));
  }
};
/**
  *Encoding a string in base 64
  *@param src {String} input string
  *@return {String} string encoded in base 64
*/
function vmp_unicode_base64_encode(text){return btoa(text);}
/**
  *Decoding a base 64 string
  *@param enc {String} oncoding 64 string
  *@return {String} decoded string
*/
function vmp_unicode_base64_decode(enc){return atob(enc);}
/**
  *Check if a string is in utf8 format
  *@param src [String} input string
  *@return {Boolean} true if src string is in utf8 format,otherwise false
*/
function vmp_unicode_utf8_check(src)
{
  let len=src.lenght,n;
  for(let i=0;i<len;i++)
  {
    c=src.charAt(i);
    if(0x00 <= c && c <= 0x7f) 
      n=0;
    else if ((c & 0xE0) == 0xC0)
      n=1;
    else if ( c==0xed && i<(len-1) && (src.charAt(i+1) & 0xa0)==0xa0)
      return false;
    else if ((c & 0xF0) == 0xE0)
      n=2;
    else if ((c & 0xF8) == 0xF0)
      n=3;
    else 
      return false;
    for (let j=0; j < n && i < len;j++) 
      if ((++i == len) || ((src.charAt(i+1) & 0xC0) != 0x80))
        return false;
    }
    return true;
}
/**
  *Check if the input string is an ipv4 address
  *@param ip {String} input string
  *@return {Boolean} true if input string is ipv4 address,otherwise false
*/
function vmp_is_ipv4_raw(ip)
{
  let split=vmp_unicode_str_split(ip,".");
  let s=split.length;
  if(s == 4)
  {
    let i;
    for(i=0;i<s;i++)
    {
       try
       {
         vmp_unicode_str_todigit_range(s,0,255);
       }
       catch(error)
       {
         break;
       }
    }
    if(i === 4)
      return true;
  }
  return false;     
}
/**Internal usage*/
function vmp_is_ipv6_raw_frag(ip)
{
  let split=vmp_unicode_str_split(ip,":");	
  let size=0;
  let n=split.length,l;
  if(n > 8)
    return -1;
  else
  {
    for(let i=0;i<n;i++)
    {
      l=split[i].length;
      if(l>0 && l<=4 && vmp_isxstring(split[i]))
        size+=1;	
      else
        return -1;
    }
  }
  return size;
}
/**
  *Check if the input string is an ipv6 address
  *@param ip {String} input string
  *@return {Boolean} true if input string is ipv6 address,otherwise false
*/
function vmp_is_ipv6_raw(ip)
{
  let split=vmp_unicode_str_split(ip,"::");
  let n=split.length,size1,size2;       
  if(n == 0 && ip == "::")
    return true;
  else if(n==1 && ip.length > 2)
  {
    if((vmp_unicode_str_sub(ip,0,2) == "::") && (vmp_unicode_str_sub(ip,ip.length-2,2) != "::") && ((size1=vmp_is_ipv6_raw_frag(split[0])) != -1))  
      return true;
    else if((vmp_unicode_str_sub(ip,0,2) != "::") && (vmp_unicode_str_sub(ip,ip.length-2,2) != "::") && ((size1=vmp_is_ipv6_raw_frag(split[0])) != -1))
      return true;
    if((vmp_unicode_str_sub(ip,0,2) != "::") && (vmp_unicode_str_sub(ip,ip.length-2,2) != "::") && ((size1=vmp_is_ipv6_raw_frag(split[0])) != -1))  
      return true;        
  }        
  else if(n==2 && ((size1=vmp_is_ipv6_raw_frag(split[0])) != -1) && ((size2=vmp_is_ipv6_raw_frag(split[1]) != -1)) && ((size1+size2) < 8))
    return true;
  return false;    
}
/**
  *Check if the input string is an ip address
  *@param ip {String} input string
  *@return {Boolean} true if input string is ip address,otherwise false
*/
function vmp_is_ip_raw(ip)
{
  if(vmp_is_ipv4_raw(ip) || vmp_is_ipv6_raw(ip))
    return true;
  return false; 
}
/**
  *Check if the input string is an mac ethernet address
  *@param mac {String} input string
  *@return {Boolean} true is mac ethernet address,otherwise false
*/
function vmp_is_macaddress_raw(mac)
{
  let split=vmp_unicode_str_split(addr,":");
  if(split.length() == 6)
  {
    for(let s of split)
      if(!((s.length == 2) && vmp_isxstring(s)))
        return false;
  }
  else
    return false;
  return true; 
}

