/* -*- Mode:Javascript; javscript-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 27/01/2025
 */

/**Interface component for main application*/
class Vmp_Main extends vmp_component(HTMLElement)
{
  /**
    *constructor
    *@param section {object} passage of the basic components:<BR> 
    *"login"  (extends class Vmp_Cmp_Login_I),<BR>
    *"waiting"(extends class Vmp_Cmp_Waiting_I),<BR>
    *"session"(extends class Vmp_Cmp_Session_I),<BR>
    *"manager"(extends class Vmp_Cmp_Manager_I)<BR>
    *if not setting use default component
  */
  constructor(section={})
  {
    super();
    vmp_component_init(this);
    this.cmpid_="main";
    window.main_=this;
    /**
     *login timeout 
     *@type {Number}
     *@public
    */
    this.login_timeout_=5;
    /**
     *loading component timeout
     *@type {Number}
     *@public
    */
    this.loading_timeout_=10;
    /**
     *kill operation timeout 
     *@type {Number}
     *@public
    */
    this.kill_timeout_=30;
    /**
     *component loading
     *@type {Set}
     *@public
    */
    this.cmploads_=vmp_wrap_set();
    /**
     *subprotocol web socket
     *@type {String}
     *@public
    */
    this.wsprotocols_="jrp.ragnu.it";
    this.register_cmp("login",vmp_wrap_createElement(("login" in section)?section["login"]:"vmp-login",'div'));
    this.register_cmp("waiting",vmp_wrap_createElement(("waiting" in section)?section["waiting"]:"vmp-waiting",'div'));
    let session=vmp_wrap_createElement(("session" in section)?section["session"]:"vmp-session",'div');
    this.register_cmp("session",session);
    this.register_cmp("manager",vmp_wrap_createElement(("manager" in section)?section["manager"]:"vmp-manager",'div'));
    session.add_action("reload","reload",this.action.bind(this,"reload"),Vmp_Img.load);
    session.add_action("main","main",this.action.bind(this,"main"),Vmp_Img.main);
    session.add_action("info","manager info",this.action.bind(this,"info"),Vmp_Img.info);
    session.add_action("logout","logout",this.action.bind(this,"logout"),Vmp_Img.logout);
    this.reset_impl();
  }
  /**Call from index.html when the program finishes loading*/
  init(){this.main();}
  /**Internal usage*/
  reset_impl()
  {
    this.cmploads_.clear();
    this.code_=0;//exit code
    this.reason_="";//exit reason 
    this.socket_=null;
    this.status_="init";
    /**
     *user logged
     *@type {String}
     *@public
    */
    this.user_='';
    /**
     *remote nodetype
     *@type {String}
     *@public
    */
    this.nodetype_=undefined
    /**
     *User permits
     *@type {Number}
     *@public
    */
    this.permits_=0
    /**
     *Session broadcast data
     *@type {Object}
     *@public
    */
    this.bdata_=undefined
    /**
     *Session jrp request data
     *@type {Object}
     *@public
    */
    this.reqdata_=undefined
    this.waitcode_=0;
    this.waitmsg_="";
    this.recv_=this.recv_session.bind(this);
    this.unregister_cmp("module");
  }
  action(name)
  {
    let session=this.search_cmp("session");
    switch(name)
    {
      case "info":
        session.run_action("info",false);
        session.run_action("main",true);
        session.set_view(this.search_cmp("manager"));
        break;
      case "main":
        session.run_action("info",true);
        session.run_action("main",false);
        session.set_view(this.search_cmp("module"));
        break;
      case "reload":
        this.waiting("Reloading Components",this.loading_timeout_,Vmp_Jrp_StatusCode.timeout_,"loading timeout")
        this.loading();
        this.main();
        break;
      case "logout":this.logout();break;
    }
  }
  /**
    *receives session from the server
    *@param jrp {Object} json request
    *@param payload {Vmp_Buf} buffer payload table lines associated with keys
  */
  recv_session(jrp,payload)
  {
    switch(jrp.msgtype)
    {
      case "session":
        this.nodetype_=jrp.nodetype;
        this.permits_=jrp.permits;
        this.bdata_=jrp.broadcast;
        this.reqdata_=jrp.reqdata;
        let manager=this.search_cmp("manager");
        manager.session(this.user_,this.nodetype_,this.permits_,this.bdata_,this.reqdata_);
        let module=vmp_module_load(this.nodetype_);
        if(module === undefined)
        {  
          let empty=vmp_cmp_div();
          empty.style.setProperty("width","100%");
          empty.innerHTML=`<center><H2>Module not found for nodetype ${this.nodetype_}</H2></center>`;
          this.register_cmp("module",empty);
        }
        else
        {
          this.register_cmp("module",module);
          this.loadCmp(module);
        }
        this.action("main");
        this.recv_=this.recv_jrp.bind(this);
        this.status_="ready";
        this.main();
        break;
      default:
        let msg=vmp_jrp_abort(Vmp_Jrp_StatusCode.protocolbad);
        this.close(Vmp_WebSocketCode.Protocol,msg.read_str(msg.size()));
        return;
    }
  }
  /**
    *receives a routine message from the server
    *@param jdata {Object} json data
    *@param payload {Vmp_Buf} buffer payload 
  */
  recv_jrp(jrp,payload)
  {
     let manager=this.search_cmp("manager");
     switch(jrp.msgtype)
     {
       case "response":
         for(let j of jrp.outputs)
         {
           let jdata;
           try
           {
             jdata=new Vmp_JData();
             jdata.set_data(j);
           }
           catch(err)
           {
             let msg=vmp_jrp_abort(Vmp_Jrp_StatusCode.protocolbad);
             this.close(Vmp_WebSocketCode.Protocol,msg.read_str(msg.size()));
           }
           manager.recv_response(jrp.rid,jdata,payload);
         }
         break;
       case "close":
         manager.recv_close(jrp.rid,jrp.status,jrp.msg);
         break;
       case "broadcast":
         let jdata;
         for(let j of jrp.outputs)
         {
           try
           {
             jdata=new Vmp_JData();
             jdata.set_data(j);
           }
           catch(err)
           {
             let msg=vmp_jrp_abort(Vmp_Jrp_StatusCode.protocolbad);
             this.close(Vmp_WebSocketCode.Protocol,msg.read_str(msg.size()));
           }
           this.recv_broadcast(jdata,payload);
         }
         break;
       default:
         let msg=vmp_jrp_abort(Vmp_Jrp_StatusCode.protocolbad);
         this.close(Vmp_WebSocketCode.Protocol,msg.read_str(msg.size()));
         return;
     }
  }
  /**
   *Send an request
   *@param cmp {Component} component associated
   *@param jdata {Vmp_JData} json data request
   *@param payload {Vmp_Buf} json payload data
   *@return requets object
  */
  request(cmp,jdata,payload=undefined)
  {
    let manager=this.search_cmp("manager");
    let ret=manager.request(cmp,jdata,payload);
    if(ret[0] === -1)
      vmp_wrap_alert(ret[1]);
    return manager.table_.get(ret[0]);
  }
  /**
   *Send an broadcast
   *@param outputs {Array(Vmp_JData)} or {Vmp_JData} json data outputs
   *@param payload {Vmp_Buf} json payload data
  */
  broadcast(outputs,payload)
  {
    let out=[]
    outputs.map(item=>out.push(item.obj_));
    let psize=(payload != undefined)?0:payload.size();
    if(psize != 0 && out.length != 1)
      vmp_wrap_except("in broadcast msg:multi json data with payload present");
    let msg={
      msgtype:"broadcast",
      outputs:out,
      payload:psize
    }
    window.main_.send(vmp_jrp_write(msg,payload));
  }
  /**View component,if this.socket_==null "login" otherwise "session"*/
  main()
  {
    this.search_cmp("waiting").stop();
    if(this.socket_ === null)this.childReplace(this.search_cmp("login"),0);
    else if(this.status_ == "init")
      this.waiting("Connect",this.login_timeout_*1000);
    else if(this.status_ == "session")
      this.waiting("Session",this.login_timeout_*1000);
    else
      this.childReplace(this.search_cmp("session"),0);
  }
  /**
    *View the waiting component
    *@param msg {String} dispaly message waiting
    *@param timeout {Number} timeout waiting
    *@param retcode {Number} Error code sent in case of timeout
    *@param waitmsg {String} Timeout error message
  */
  waiting(msg,timeout,retcode=0,waitmsg="")
  {
    this.waitcode_=retcode;
    this.waitmsg_=waitmsg;
    let waiting=this.search_cmp("waiting");
    waiting.start(msg,timeout);
    this.childReplace(waiting,0);
  }
  /**
    *Receives a timeout 
    *@param msg {String} timeout message
  */
  timeout(msg)
  {
    if(this.status_ === "init")this.close(Vmp_Jrp_StatusCode.timeout_,"timeout init");
    else if(this.status_ === "session")this.close(Vmp_Jrp_StatusCode.timeout_,"timeout session");
    else if(this.status_ === "close")
    {
      this.reset();
      this.main();  
    }
    else if(this.waitcode_ == 0) this.close(Vmp_Jrp_StatusCode.timeout_);
    else this.close(this.waitcode_,this.waitmsg_);
  }
  /**Loading the components in the component loading list*/
  loading()
  {
    let tmp=vmp_wrap_set();
    this.cmploads_.forEach((cmp, valueAgain, set) => tmp.add(cmp));
    tmp.forEach((cmp, valueAgain, set) => cmp.load());
  }
  /**UnLoading the components in the component loading list*/
  unloading(){this.cmploads_.forEach((cmp, valueAgain, set) => cmp.unload());}
  /**
    *load a component
    *@param cmp {Component} component loading
  */
  loadCmp(cmp)
  {
    let s=this.status_;
    if(!this.cmploads_.has(cmp))this.cmploads_.add(cmp);
    cmp.load();
  }
  /**
    *unload a component 
    *@param cmp {Component} component loading
  */
  unloadCmp(cmp)
  {
    if(this.cmploads_.has(cmp))
    {
      this.cmploads_.delete(cmp); 
      cmp.unload();
    }
  }
  /** 
    *Connect to a websocket application
    *@param url {String} websocket url
    *@param user {String} user login
  */
  connect(url,user)
  {
    if(window.socket_ != null)
    {
      vmp_wrap_alert("socket already");
      return;
    }
    this.user_=user;
    this.socket_ = new WebSocket(url,this.wsprotocols_);
    this.socket_.binaryType = "arraybuffer";
    this.main();
    this.socket_.onopen = function(event){window.status_="session"};
    this.socket_.onmessage = function(event) 
    {
      if(event.data instanceof ArrayBuffer)
      {
        let buf=new Vmp_Buf();
        buf.write_arraybuffer(event.data);
        buf.index();
        let jrp=vmp_jrp_parse(buf);
        if(jrp[0] === Vmp_Jrp_StatusCode.status_ok)
          window.main_.recv_(jrp[1],jrp[2]);  
        else window.main_.abort(jrp[0],Vmp_Jrp_StatusCode.reason(jrp[0]));
      }
      else
        window.main_.close(Vmp_WebSocketCode.Type);  
    };
    this.socket_.onclose = function(event) 
    {
      if(event.wasClean || window.main_.code_ === 0)
      {
        if(event.code == Vmp_WebSocketCode.Abnormal)vmp_wrap_alert("Abnormal closed");
        else if (event.reason == "")vmp_wrap_alert(`Remote closed code=${event.code}`);
        else vmp_wrap_alert(`Remote closed code=${event.code} reason=${event.reason}`);
      }
      else if(window.main_.code_ != Vmp_WebSocketCode.Normal)
      {
        if(window.main_.code_ == Vmp_WebSocketCode.Abnormal)        
          vmp_wrap_alert("Abnormal closed");
        else if (window.main_.reason_ === "")
          vmp_wrap_alert(`UserAgent closed code=${window.main_.code_}`);
        else vmp_wrap_alert(`UserAgent closed code=${window.main_.code_} reason=${window.main_.reason_}`);
      }
      window.main_.reset();
      window.location.reload(true);
      window.main_.main();
      
    }
    this.socket_.onerror = function(event) 
    {
        if(window.main_.status_ === "init")
          vmp_wrap_alert(`Connection refused`);
        else
        {
          if(event.isTrusted)
            vmp_wrap_alert(`Websocket error:user agent`);
          else
            vmp_wrap_alert(`Websocket error:${event.message}`);
        }
        window.main_.reset();
        window.location.reload(true);
        window.main_.main();
    }
  }
  /**
    *sends a buffer via websocket
    *@param buf {Vmp_Buf} input buffer
  */
  send(buf)
  {
    buf.index();
    if(this.socket_ != null)
      this.socket_.send(buf.read_arraybuffer(buf.size()));
  }
  /**
    *Abort jrp websocket connection
    *@param code {Number} jrp close code
    *@param reason {String} jrp reason message 
  */
  abort(code,reason="")
  {
    let msg={
      msgtype:"abort",
      status:code,
      msg:reason
    }
    this.close(Vmp_WebSocketCode.Unexpected,vmp_wrap_json_stringify(msg));  
  }
  /**
    *Close websocket connection
    *@param code {Number} close code
    *@param reason {String} reason message
  */
  close(code,reason="")
  {
    if(this.socket_ != null)
    {  
      this.status_="close";
      this.code_=code;
      if(code !=Vmp_WebSocketCode.Normal && reason==="")
        reason=Vmp_WebSocketCode.reason(code);
      this.waiting(`UnLoading components`,this.loading_timeout_*1000);
      this.unloading();
      this.socket_.close(code+2000,reason);
    }
  }
  /**logout session*/
  logout()
  {
    if(vmp_wrap_confirm("Do you want logout connection?"))
      this.close(Vmp_WebSocketCode.Normal);
  }
  /**
    *virtual function called by When a broadcast message is received
    *@param jdata {Vmp_JData} json data
    *@param payload {Vmp_Buf} message payload
  */
  recv_broadcast(jdata,payload){}
};
/**
  *Submit a request and wait for it to complete
  *@param req {Vmp_Jrp_Request} request send
*/
function vmp_main_operwait(cmp,jdata,text)
{
  let ret=window.main_.request(cmp,jdata);
  window.main_.waiting(text,10000,Vmp_WebSocketCode.Protocol,`${text} close timeout`);
  return ret;
}

/**
  *Close a pending request
  *@param req {Vmp_Jrp_Request} request close
*/
function vmp_main_operwait_close(req)
{
  let voper=window.main_.search_cmp("waiting").msg_;
  window.main_.main();
  if(req.close_status_ === Vmp_Jrp_StatusCode.ok)
    vmp_wrap_alert(`${voper} ok`);
  else
    vmp_wrap_alert(`${voper} error '${req.close_msg_}'`);
}
