/* -*- Mode:Javascript; javscript-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 26/02/2025
 */
/**
  *decorator who transforms a tag into a component
  *@param element tag element type
  *@return tag element 
*/
/** 
  *@constant
  *@typedef {Object} List of system images
  *@default
  *@property {String} clear img
  *@property {String} wait img
  *@property {String} close img
  *@property {String} down img
  *@property {String} up img
  *@property {String} view img
  *@property {String} info img
  *@property {String} load img
  *@property {String} logout img
  *@property {String} main img
  *@property {String} logo img
  *@property {String} select img
*/
const Vmp_Img={
  clear:"lib/img/clear.png",
  wait:"lib/img/clessidra43.gif",
  close:"lib/img/close.png",
  down:"lib/img/down.png",
  up:"lib/img/up.png",
  view:"lib/img/eye.png",
  info:"lib/img/info.png",
  load:"lib/img/load_page.png",
  logout:"lib/img/logout.png",
  main:"lib/img/main.png",
  logo:"lib/img/ragnu.jpg",
  select:"lib/img/select.png",
}
function vmp_component(element)
{
  /**Returns parent component*/
  element.prototype.top=function(){return top_;}
  /**
    *register component
    *@name vmp_component#register_cmp
    *@function
    *@throws in case of failure
    *@param name component name
    *@param cmp component register
  */
  element.prototype.register_cmp=function(name,cmp)
  {
    if(vmp_isstring(name) && (name.length != 0))
    {
      for(let s of name)
      {
        s=s.charCodeAt(0);
        if((s <= 0x20)||(s >= 0x7E))
          vmp_except(`vmp_component(name=${name}) bad value`);
      }
      cmp.cmpid_=name;
      cmp.top_=this;
      this.register_.set(name,cmp);
    }
    else
      vmp_except(`vmp_compnent(name=${name}) bad value`);
  }
  /**
    *unregister component
    *@name vmp_component#unregister_cmp
    *@function
    *@param name component name
    *@return component unregister
  */
  element.prototype.unregister_cmp=function(name)
  {
    if(this.register_.has(name))
    {
      let ret=this.register_.get(name);
      this.register_.delete(name);
      return ret;
    }
    return undefined;
  }
  /**
    *Search Component
    *@name vmp_component#search_cmp
    *@function 
    *@param {String} The name of the hierarchy of components to search for(cmp3.cmp2.cmp1 where cmp(n+1) a children of cmp(n),'' this component
    *@return {Component} the component found or undefined if not found
  */
  element.prototype.search_cmp=function(cmpname)
  {
    if(cmpname == '')
      return this;
    cmps=cmpname.split(".");
    let cmp=this.register_.get(cmps[cmps.length-1]);
    if(cmp == undefined)
      return undefined;
    cmps.pop();
    return cmp.search_cmp(cmps.join("."));
  }
   /**
    *clear components
    *@name vmp_component#clear_cmp
    *@function
  */
  element.prototype.clear_cmp=function(){this.register_.clear();}
  /**
    *Enable tag
    *@name vmp_component#enable
    *@function  
  */
  element.prototype.enable=function()
  {
    for(let amount of this.register_.values())
      amount.enable();
    this.enable_impl();
  }
  /**
    *Virtual function called by enable
    *@name vmp_component#enable_impl
    *@function 
  */
  element.prototype.enable_impl=function(){}
  /**
    *Disable tag
    *@name vmp_component#disable
    *@function  
  */
  element.prototype.disable=function()
  {
    for(let amount of this.register_.values())
      amount.disable();
    this.disable_impl();
  }
  /**
    *Virtual function called by disable
    *@name vmp_component#disable_impl
    *@function 
  */
  element.prototype.disable_impl=function(){}
  /**
    *Reset component element
    *@name vmp_component#reset
    *@function  
  */
  element.prototype.reset=function()
  {
    for(let amount of this.register_.values())
      amount.reset();
    this.reset_impl();
  }
  /**
    *Virtual function called by reset
    *@name vmp_component#reset_impl
    *@function 
  */
  element.prototype.reset_impl=function(){}
  /**
    *@name vmp_component#cmpid
    *@function
    *@return {String} id component remote node to which the module is associated
  */
  element.prototype.cmpid=function(){return(this.top_ === null)?this.cmpid_:`${this.cmpid_}.${this.top_.cmpid()}`;}
  /**
    *@name vmp_component#parents
    *@function
    *Return parent tag
    *@return {Tag} parents tag o null in case of root tag
  */
  element.prototype.parent=function(){return vmp_tag_parents(this);}
  /**
    *@name vmp_component#parents
    *@function
    *Return index children tag
    *@param children index
    *@return {Tag} the children referred to the index
*/
  element.prototype.children=function(index){return vmp_tag_children(this,index);}
  /**
    *Insert an item in the index position
    *@name vmp_component#childInsert
    *@function
    *@param child element to be inserted
    *@param index {Number} index position
    *@return the appended node 
  */
  element.prototype.childInsert=function(child,index){return vmp_tag_childInsert(this,child,index);}
  /**
    *Replace an item in the index position,if index is out range it append at the end
    *@name vmp_component#childReplace
    *@function
    *@param child element to be inserted
    *@param index {Number} index position
    *@return the replaced node 
  */
  element.prototype.childReplace=function(child,index){return vmp_tag_childReplace(this,child,index);}
  /*
    *Remove an item in the index position
    *@name vmp_component#childRemove
    *@function
    *@param index {Number} index position
  */
  element.prototype.childRemove=function(index){return vmp_tag_childRemove(this,index);}
  /**
    *Removes the first item
    *@name vmp_component#childRemovePop
    *@public
    *@function 
  */
  element.prototype.childRemovePop=function(){return vmp_tag_childRemovePop(this);}
  /**
    *removes the last item
    *@name vmp_component#childRemoveBack
    *@function   
  */
  element.prototype.childRemoveBack=function(){return vmp_tag_childRemoveBack(this);}
  /**
    *Remove all childs
    *@name vmp_component#childRemoveAll
    *@function
  */
  element.prototype.childRemoveAll=function(){return vmp_tag_childRemoveAll(this);}
  /**
    *Send an http get
    *@name vmp_component#getremote node to which the module is associated
    *@function
    *@param url {String} url address
    *@param id  operation id
    *@param timeout {Number} timeout operation
  */
  element.prototype.get=function(url,id,timeout=5000)
  {
    let xhr = new XMLHttpRequest();
    let cmp=this;
    xhr.open("GET",url, true);
    xhr.timeout = timeout;
    xhr.onload = (e) => {
      if (xhr.readyState === 4) {cmp.get_recv(id,xhr.status,xhr.responseText);}
    };
    xhr.ontimeout = (e) => {cmp.get_recv(408,"Request Timeout");};
    xhr.onerror = (e) =>{cmp.get_recv(id,xhr.status,xhr.statusText);};
    xhr.send(null);
  }
  /**
    *Virtual function for recv get operation
    *@name vmp_component#get_recv
    *@function 
    *@param id  operation id
    *@param status {Number} operation status
    *@param text {String} operation text response
  */
  element.prototype.get_recv=function(id,status,text){}
  /**
    *Send an http postremote node to which the module is associated
    *@name vmp_component#post
    *@function 
    *@param url {String} url address
    *@param id  {String} operation id
    *@param ctype {String} content-type header data
    *@param data {String}  input data to send
    *@param timeout {number} timeout operation
  */
  element.prototype.post=function(url,id,ctype,data,timeout=5000)
  {
    let xhr = new XMLHttpRequest();
    let cmp=this;
    xhr.open("POST",url, true);
    xhr.timeout = timoout;
    xhr.onload = (e) => {
      if (xhr.readyState === 4){cmp.post_recv(id,xhr.status,xhr.responseText);}
    };
    xhr.ontimeout = (e) => {cmp.post_recv(id,408,"Request Timeout");};
    xhr.onerror = (e) => {cmp.post_recv(id,xhr.status,xhr.statusText);};
    xhr.setRequestHeader('Content-type',ctype);
    xhr.send(data);
  }
  /**
    *Virtual function for recv post operation
    *@name vmp_component#post_recv
    *@function 
    *@param id {String} operation id
    *@param status {Number} operation status
    *@param text {String} operation text response
  */
  element.prototype.post_recv=function(id,status,text){}
  /** 
    *Virtual function for load component
    *@name vmp_component#load
    *@function 
  */
  element.prototype.load=function(){}
  /** 
    *Virtual function for unload component
    *@name vmp_component#unload
    *@function
  */
  element.prototype.unload=function(){}
  /**
    *Virtual function that receives a message generic
    *@name vmp_component#msg_recv
    *@param jdata {Vmp_JData} json data
    *@param buf {Vmp_Buf} Buffer data
    *@function
  */
  element.prototype.msg_recv=function(jdata,buf){}
  /**
    *Virtual function that receives a response from an associated request
    *@name vmp_component#req_recv
    *@param req request associated
    *@param jdata {Vmp_JData} json data
    *@param payload {Vmp_Buf} buffer data
    *@function
  */
  element.prototype.req_recv=function(req,jdata,payload){}
  /**
    *Virtual function that close request
    *@name vmp_component#req_close
    *@param req request associated
    *@function
  */
  element.prototype.req_close=function(req){}
  return element;
}
/**Init component data,constructor*/
function vmp_component_init(element)
{
  element.cmpid_='';
  element.top_=null;
  element.register_=vmp_wrap_map();
}
/**Module Component*/ 
class Vmp_Module extends vmp_component(HTMLDivElement)
{
  /**constructor*/
  constructor(){super();vmp_component_init(this);}
}
/**Module tags*/
window.modules_=vmp_wrap_map();
/**
  *Register a module in the project
  *@param nodetype {String} remote node to which the module is associated
  *@param modname {String} module name(the name must have a hyphen in the middle)
  *@param module {Module} Component modules
*/
function vmp_module_register(nodetype,modname,module)
{
  vmp_wrap_define(modname,module,'div');
  window.modules_.set(nodetype,modname);
}
/**
  *Load a module based on the nodetype
  *@param nodetype {String} remote node to which the module is associated
  *@return {Module} module element or undefined in case of failure
*/
function vmp_module_load(nodetype)
{
  let modname=window.modules_.get(nodetype);
  if(modname != undefined)
  {
    let ret=vmp_wrap_createElement(modname,'div');
    return ret;
  }
  return undefined;
}
/**Transform an element div into a vmp_component div */
class Vmp_Cmp_Div extends vmp_component(HTMLDivElement)
{
  /**a constructor*/
  constructor(){super();vmp_component_init(this);}
};
/**Transform an element table into a vmp_component table */
class Vmp_Cmp_Table extends vmp_component(HTMLTableElement)
{
  /**A constructor*/
  constructor()
  {
    super();
    vmp_component_init(this);
    this.thead_=this.createTHead();
    this.tbody_=this.createTBody();
    this.tfoot_=this.createTFoot();  
  }
  /**
    *Creates and inserts a row in thead_
    *@param index {Size} index value
    *@return {Tag} row created  
  */
  thead_row(index=undefined){return (index===undefined)?this.thead_.insertRow():this.thead_.insertRow(index);}
  /**
    *Creates and inserts a row in tbody_
    *@param index {Size} index value
    *@return {Tag} row created  
  */
  tbody_row(index=undefined){return (index===undefined)?this.tbody_.insertRow():this.tbody_.insertRow(index);}
  /**
    *Creates and inserts a row in tfoot_
    *@param index {Size} index value
    *@return {Tag} row created
  */
  tfoot_row(index=undefined){return (index===undefined)?this.tfoot_.insertRow():this.tfoot_.insertRow;}
  /**
    *returns the number of lines in thead
    @return {Size} returns the number of lines in thead
  */
  thead_nrow(){return this.thead_.rows.length;}
  /**
    *returns the number of lines in tbody
    @return {Size} returns the number of lines in tbody
  */
  tbody_nrow(){return this.thead_.rows.length;}
  /**
    *returns the number of lines in tfoot
    @return {Size} returns the number of lines in tfoot
  */
  tfoot_nrow(){return this.thead_.rows.length;}
  /**
    *returns the line referring to index in thead
    @param index {Index} index value
    @return {Size} returns the number of lines in thead
  */
  thead_getrow(index){return this.thead_.rows[index];}
  /**
    *returns the line referring to index in tbody
    @param index {Index} index value
    @return {Size} returns the number of lines in tbody
  */
  tbody_getrow(index){return this.tbody_.rows[index];}
  /**
    *returns the line referring to index in tfoot
    @param index {Index} index value
    @return {Size} returns the number of lines in tfoot
  */
  tfoot_getrow(index){return this.tfoot_.rows[index];}
  /**
    *Delete the line referring to index in thead
    @param index {Index} index value
  */
  thead_delrow(index){return this.thead_.rows[index];}
  /**
    *Delete the line referring to index in tbody
    @param index {Index} index value
  */
  tbody_delrow(index){return this.tbody_.deleteRow(index);}
  /**
    *returns the line referring to index in tfoot
    @param index {Index} index value
  */
  tfoot_delrow(index){return this.tfoot_.rows[index];}
  /**
    *Inserts a column td element in a row
    *@param row {Tag} row element
    *@param index {Index} position Cell
    *@return {Tag} coloumn td element
  */
  row_td(row,index=undefined){return (index === undefined)?row.insertCell():row.insertCell(index);}
  /**
    *Inserts a column th element in a row
    *@param row {Tag} row element
    *@param index {Index} position Cell
    *@return {Tag} coloumn td element
  */
  row_th(row,index=undefined)
  {
    let th=vmp_wrap_createElement('th');
    if(index === undefined)
      vmp_tag_childInsert(row,th,row.children.length-1);
    else
      vmp_tag_childInsert(row,th,index);
    return th;
  }
  /**
    *Returns the number of cells in a row
    *@param row {Tag} row element
    *@return {Size} the number of cells
  **/
  row_ncells(row){return row.cells.length;}
  /**
    *Returns the cell referred to in index
    *@param row {Tag} row element
    *@param index {Index} position Cell
    *@return {Size} the number of cells
  **/
  row_getcell(row,index){return row.cells[index];}
  /**
    *Delete the cell referred to in index
    *@param row {Tag} row element
    *@param index {Index} position Cell
    *@return {Size} the number of cells
  **/
  row_delcell(row,index){return row.deleteCell(index);}
};
/**Create component input text elements*/
class Vmp_Cmp_Input extends vmp_component(HTMLInputElement)
{
  /**A constructor*/
  constructor()
  {
    super();
    vmp_component_init(this);
    this.setAttribute("type","text");
  }
  /**
    *Initialize Component
    *@param size {Size} visual size
    *@param placeholder {String} containing a hint to the user of what can be entered in the control
    *@param default_v {String} default value
  */
  init(size,placeholder,default_v="")
  {
    this.setAttribute("size",size);
    this.setAttribute("placeholder",`Enter ${placeholder}`);
    this.set_value(default_v);
  }
  /**virtual function*/
  enable_impl(){vmp_tag_enable(this);}
  /**virtual function*/
  disable_impl(){vmp_tag_disable(this);}
  /**virtual function*/
  reset_impl(){this.set_value();}
  /**
    *Set input text value
    *@param v {String} value setting
  */
  set_value(v=""){this.value=v;}
  /**
    *Return input value
    *@return {String} text value
  */
  get_value(){return this.value;}
};
/**Create component input text elements*/
class Vmp_Cmp_CheckBox extends vmp_component(HTMLInputElement)
{
  /**A constructor*/
  constructor()
  {
    super();
    vmp_component_init(this);
    this.setAttribute("type","checkbox");
  }
  /**
    *Init Component
    *@param value {String} Checkbox value
  */
  init(value){this.setAttribute("value",value);};
  /**virtual function*/
  enable_impl(){vmp_tag_enable(this);}
  /**virtual function*/
  disable_impl(){vmp_tag_disable(this);}
  /**virtual function*/
  reset_impl(){this.set_select(false);}
  /**
    *Get Checkbox value
    *@return {String} Checkbox value
  */
  get_value(){return this.value;}
  /**
    *set checkebox select
    *@value {Boolean} checkbox value
  */
  set_select(value){this.checked=value;}
  /**
    *Get if checkbox is select
    *@return {Boolean} is select checkbox?
  */ 
  get_select(){return this.checked;}
};
/**Button base*/
class Vmp_Cmp_Button_Base extends vmp_component(HTMLButtonElement)
{
  /**A constructor*/
  constructor(){super();vmp_component_init(this);}
  /**
    *Init component button text
    *@param label {String} Button label
    *@param id {String} button id
    *@param cb {Callback} callback event
  */
  init(label,id)
  {
    this.innerHTML=label;
    this.setAttribute("data-id",id);
  }
  /**virtual function*/
  enable_impl(){vmp_tag_enable(this);}
  /**virtual function*/
  disable_impl(){vmp_tag_disable(this);}
  /**Returns button id*/
  bid(){return this.getAttribute("data-id");}
};
/**Button component*/
class Vmp_Cmp_Button extends Vmp_Cmp_Button_Base 
{
  /**A constructor*/
  constructor()
  {
    super();
    this.style.setProperty("font-size",vmp_rootvar("--font-size-medium"));
    this.style.setProperty("margin","2px 2px 2px 2px");
    this.style.setProperty("padding","2px 5px 2px 5px");
    this.style.setProperty("background-color","#efefef");
    this.style.setProperty("cursor",vmp_rootvar("--cursor-pointer"));
    this.style.setProperty("border-radius","10px 10px 10px 10px");
  }
};
/**Img component*/
class Vmp_Cmp_Img extends vmp_component(HTMLImageElement)
{
  /**A constructor*/
  constructor(){super();vmp_component_init(this);}
  /**
    *Initialize Component
    *@param src {String} target img
    *@param alt {String} alt attribute
  */
  init(src,alt)
  {
    this.setAttribute("src",src);
    this.setAttribute("alt",alt);
  }
};
/**Select Component*/
class Vmp_Cmp_Select extends vmp_component(HTMLSelectElement)
{
  /**A constructor*/
  constructor(){super();vmp_component_init(this);}  
  /**
    *Initialize Component
    *@param values {ArrayObject} Objects option {value:"value options",text:"text value"}  
  */
  init(...values)
  {
    let option;
    for(let v of values)
    {
      option=vmp_wrap_createElement("option");
      option.setAttribute("value",v.value);
      option.innerHTML=v.text;
      this.appendChild(option);
    }
  }
  /**virtual function*/
  reset_impl(){this.set_value(0);}
  /**virtual function*/
  enable_impl(){vmp_tag_enable(this);}
  /**virtual function*/
  disable_impl(){vmp_tag_disable(this);}
  /**Return value selected*/
  get_value(){return this.value;}
  /**Setting index value*/
  set_value(index=0){this.selectedIndex=index;}
};
try
{
  vmp_wrap_define("vmp-div",Vmp_Cmp_Div,'div');
  vmp_wrap_define("vmp-table",Vmp_Cmp_Table,'table');
  vmp_wrap_define("vmp-input",Vmp_Cmp_Input,'input');
  vmp_wrap_define("vmp-checkbox",Vmp_Cmp_CheckBox,'input');
  vmp_wrap_define("vmp-bbutton",Vmp_Cmp_Button_Base,'button');
  vmp_wrap_define("vmp-button",Vmp_Cmp_Button,'button');
  vmp_wrap_define("vmp-img",Vmp_Cmp_Img,'img');
  vmp_wrap_define("vmp-select",Vmp_Cmp_Select,'select');
}
catch(error)
{
  vmp_wrap_alert(error);
}
/**
  *Create div component
  *@param id component id
  *@param parent component parent  
*/
function vmp_cmp_div(){return vmp_wrap_createElement("vmp-div",'div');}
/**
  *Create table component
  *@param id component id
  *@param parent component parent 
*/
function vmp_cmp_table(){return vmp_wrap_createElement("vmp-table",'table');}
/**
  *Create Component input text
  *@param size {Size} visual size
  *@param placeholder {String} containing a hint to the user of what can be entered in the control
  *@param default_v {String} default value
  *@return {Tag} component input
*/
function vmp_cmp_input(size,placeholder,default_v="")
{
  let ret=vmp_wrap_createElement("vmp-input",'input');
  ret.init(size,placeholder,default_v);
  return ret;
}
/**
  *Create Component CheckBox
  *@param value {String} checkbox value  
  *@return {Tag} component checkbox
*/
function vmp_cmp_checkbox(value,sdefault=false)
{
  let ret=vmp_wrap_createElement("vmp-checkbox",'input');
  ret.init(value);
  ret.set_select(sdefault);
  return ret;
}
/**
  *Create Component button base,Styles not implemented
  *@param label {String} label
  *@param bid {String} button id
  *@return {Tag} component base button
*/
function vmp_cmp_bbutton(label,bid="")
{
  let ret=vmp_wrap_createElement("vmp-bbutton",'button');
  ret.init(label,bid);
  return ret;
}
/**
  *Create Component button text
  *@param label {String} label
  *@param bid {String} button id
  *@return {Tag} component button
*/
function vmp_cmp_button(label,bid="")
{
  let ret=vmp_wrap_createElement("vmp-button",'button');
  ret.init(label,bid);
  return ret;
}
/**
  *Create Component button with img
  *@param img {String} img path
  *@param alt {String} image alt attribute
  *@param bid {String} button id
  *@return {Tag} component password
*/
function vmp_cmp_button_img(img,alt,bid="")
{
  let ret=vmp_wrap_createElement("vmp-button",'button');
  ret.init(`<img src='${img}' alt='${alt}' height=${vmp_rootvar("--font-size-medium")} width=${vmp_rootvar("--font-size-medium")} />`,bid);
  return ret;  
}
/**
  *Create Component img
  *@param src {String} target img
  *@param alt {String} alt attribute
  *@return {Tag} component img
*/
function vmp_cmp_img(src,alt)
{
  let ret=vmp_wrap_createElement("vmp-img",'img');
  ret.init(src,alt);
  return ret;
}
/**
  *Create Component select
  *@param values {Object} Objects option {value:"value options",text:"text value"}  
  *@return {Tag} component img
*/
function vmp_cmp_select(...values)
{
  let ret=vmp_wrap_createElement("vmp-select",'select');
  ret.init(...values);
  return ret;
}
/**Password form + view button*/
class Vmp_Cmp_Password extends Vmp_Cmp_Div
{
  constructor()
  {
    super();
    /**
     *Password text input
     *@type {Tag}
     *@public
    */
    this.passwd_=vmp_wrap_createElement('input');
    this.passwd_.setAttribute("type","password");
    this.appendChild(this.passwd_);
    /**
     *Password button view
     *@type {Tag}
     *@public
    */
    this.view_=vmp_cmp_button_img(Vmp_Img.view,"password view","");
    this.view_.addEventListener("mousedown",this.view.bind(this));
    this.view_.addEventListener("mouseup",this.unview.bind(this));
    this.appendChild(this.view_);
  }
  /**
    *Initialize Component
    *@param size {Size} visual size
    *@param placeholder {String} containing a hint to the user of what can be entered in the control
    *@param default_v {String} default value
  */
  init(size,placeholder,default_v="")
  {
    this.passwd_.setAttribute("size",size);
    this.passwd_.setAttribute("placeholder",`Enter ${placeholder}`);
    this.set_value(default_v);
  }
  /**virtual function*/
  enable_impl(){vmp_tag_enable(this.passwd_);}
  /**virtual function*/
  disable_impl(){vmp_tag_disable(this.passwd_);}
  /**virtual function*/
  reset_impl(){this.set_value();}
  /**
    *Set input text value
    *@param v {String} value setting
  */
  set_value(v=""){this.passwd_.value=v;}
  /**
    *Return input value
    *return {String} password value
  */
  get_value(){return this.passwd_.value;}
  view(event){this.passwd_.setAttribute("type","text");}
  unview(event){this.passwd_.setAttribute("type","password");}
};
/**A menu for splitting sections of an application */
class Vmp_Cmp_Menu extends Vmp_Cmp_Div
{
  constructor()
  {
    super();
    this.menuindex_=0;
    this.menu_='';
    this.drop_=null;
    this.style.setProperty("width","100%");
    this.table_=vmp_wrap_map();
    this.nav_=vmp_cmp_div();
    this.nav_.style.setProperty("background-color",vmp_rootvar("--background-color-bar"));
    this.nav_.style.setProperty("margin","10px 0px 30px 0px");
    this.nav_.style.setProperty("text-align","center");
    this.childInsert(this.nav_,0);
  }
  /**virtual function*/
  reset_impl()
  {
    this.nav_.childRemoveAll();
    this.clear_cmp();
    this.table_.clear();
    this.menuindex_=0;
    this.menu_='';
    this.drop_=null;
  }
  /**Internal usage*/
  select_tag(info,value)
  {
    if(value)
    {  
      if(info.type == "submenu")
        info.menu.children[0].style.setProperty("color",vmp_rootvar("--background-color-main"));
      info.div.children[0].style.setProperty("cursor",vmp_rootvar("--cursor-unpointer"));
      info.div.children[0].style.setProperty("color",vmp_rootvar("--background-color-main"));
    }
    else
    {
      if(info.type == "submenu")
        info.menu.children[0].style.setProperty("color",vmp_rootvar("--font-color"));
      info.div.children[0].style.setProperty("cursor",vmp_rootvar("--cursor-pointer"));
      info.div.children[0].style.setProperty("color",vmp_rootvar("--font-color"));
    }
  }
  /**Internal usage*/
  box(name,key,w,click,display,decorator="none")
  {
    let div=vmp_cmp_div();
    div.style.setProperty("display",display);
    div.style.setProperty("width",w);
    div.style.setProperty("height","100%");
    div.style.setProperty("overflow","hidden");
    let b=vmp_cmp_bbutton(name,key);
    b.style.setProperty("width","100%");
    b.style.setProperty("height","100%");
    b.style.setProperty("font-size",vmp_rootvar("--font-size-medium"));
    b.style.setProperty("border","none");
    b.style.setProperty("cursor",vmp_rootvar("--cursor-pointer"));
    b.style.setProperty("background-color",vmp_rootvar("--background-color-bar"));
    b.style.setProperty("text-decoration",decorator);
    b.addEventListener("mouseover",this.mouseover.bind(this));
    b.addEventListener("mouseout",this.mouseout.bind(this));
    b.addEventListener("click",click);
    div.appendChild(b);
    this.register_cmp(key,b);
    this.nav_.appendChild(div);
    return div;
  }
  /**
    *Adds a section to the menu
    *@param name {String} section name
    *@param tag {Tag} tag displayed
  */
  add_section(name,tag)
  {
    if(this.menuindex_ < 7)
    {
      let key=`${this.menuindex_}`;
      let div=this.box(name,key,"14%",this.clickme.bind(this),"inline-block","underline");
      this.table_.set(key,{type:'menu',div:div,tag:tag});
      this.select_tag(this.table_.get(key),false);
      if(this.menuindex_ == 0)
      {
        this.menu_=key;
        this.select_tag(this.table_.get(key),true);
        this.childInsert(tag,1);
      }
    }
    this.menuindex_ +=1;
  }
  /**
    *Add a submenu to menu
    *@param name {String} menu name
    *@param submenu {Array} menu list [{name:'menu name',tag:submenu tag displayed},...]
  */
  add_submenu(name,submenu)
  {
    if((this.menuindex_ < 7) && (submenu.length != 0))
    {
      let key=`${this.menuindex_}`;
      let top=this.box(name,key,"14%",this.menu.bind(this),"inline-block");
      let dropdown=vmp_wrap_createElement("div");
      dropdown.style.setProperty("display","none");
      dropdown.style.setProperty("position","absolute");
      dropdown.style.setProperty("background-color",vmp_rootvar("--background-color-bar"));
      dropdown.style.setProperty("min-width","14%");
      dropdown.style.setProperty("box-shadow","0px 8px 16px 0px rgba(0,0,0,0.2)");
      dropdown.style.setProperty("z-index","1");
      dropdown.setAttribute("top","top");
      top.appendChild(dropdown);
      for(let i=0;i<submenu.length;i++)
      {
        let key=`${this.menuindex_}_${i}`;
        let div=this.box(submenu[i].name,key,"100%",this.clickme.bind(this),"block","underline");
        dropdown.appendChild(div);
        this.table_.set(key,{type:'submenu',div:div,tag:submenu[i].tag,menu:top});
        this.select_tag(this.table_.get(key),false);
      }
      if(this.menuindex_ == 0)
      {
        let key=`${this.menuindex_}_0`;
        this.menu_=key;
        let info=this.table_.get(key);
        this.select_tag(info,true);
        this.childInsert(info.tag,1);
      }
    }
    this.menuindex_ +=1;
  }
  /**internal usage*/
  clickme(e)
  {
    if(this.drop_ != null)
    {
      this.drop_.style.setProperty("display","none");
      this.drop_=null;
    }
    let target=e.currentTarget;
    let name=target.bid();
    if(name != this.menu_)
    {
      let info=this.table_.get(name);
      let old=this.table_.get(this.menu_);
      this.select_tag(old,false);
      this.menu_=name;
      this.select_tag(info,true);
      this.childReplace(info.tag,1);
    }
  }
  mouseover(e)
  {
    let name=e.currentTarget.getAttribute('data-name');
    if(name != this.menu_)
      e.target.style.setProperty("background-color",vmp_rootvar("--background-color-window2"));
  }
  mouseout(e)
  {
    e.target.style.setProperty("background-color",vmp_rootvar("--background-color-bar"));
  }
  menu(e)
  {
    let target=e.target.parentElement.children[1];
    if(target.style.display == "none")
    {
      if(this.drop_ != null)
      {
        this.drop_.style.setProperty("display","none");
        this.drop_=null;
      }
      target.style.setProperty("display","block");
      this.drop_=target;
    }
    else
      target.style.setProperty("display","none");
  }
};
/**Component that represents a list of objects (one for each row in the table)*/
class Vmp_Cmp_List extends Vmp_Cmp_Table
{
  /**A constructor*/
  constructor()
  {
    super();
    /**
     *table lines associated with keys
     *@type {Map}
     *@public
    */
    this.lines_=new vmp_wrap_map();
    this.style.setProperty("width","98%");
    this.style.setProperty("border","1px solid #000000");
    this.style.setProperty("border-collapse","collapse");
    this.style.setProperty("margin-left","2px");
    this.style.setProperty("height","auto");
    this.style.setProperty("table-layout","auto");
    this.style.setProperty("overflow","auto");
  }
  /**
    *init new object type list
    *@param cols {Object} Objects with fields {name:"column name",width:"column width"(default notr assigned),order:Can the data be sorted?(default false)}
  */
  init(...cols)
  {
    this.index_=-1;
    this.order_="";
    this.hrow_=this.thead_row();
    let name,order,col,i;
    for(let c of cols)
    {
       if(c.name != undefined)
       {
         name=c.name;
         i=this.row_ncells(this.hrow_);
         order=(!vmp_isbool(c.order))?false:c.order;
         col=this.row_th(this.hrow_,i);
         col.style.setProperty("border","1px solid #000000");
         col.style.setProperty("border-collapse","collapse");
         col.style.setProperty("text-align","center");
         if(c.width != undefined)col.setAttribute('width',c.width);
         col.innerHTML=name;
         if(order)
         {
           if(this.index_ === -1)
           {  
             this.index_=i;
             this.order_="inc";
           }
           col.style.setProperty("cursor",vmp_rootvar("--cursor-pointer"));
           col.setAttribute("data-index",i);
           col.addEventListener("click",this.clickme.bind(this));
         }
       }
    }
    this.order();
  }
  /**Internal usage*/
  clickme(e)
  {
    let i=Number(e.currentTarget.getAttribute('data-index'));
    if(i == this.index_)this.order_=(this.order_ === "inc")?"dec":"inc";
    else this.index_=i;
    this.order();
  }
  /**Internal usage insert data*/
  col_style(col)
  {
     col.style.setProperty("border","1px solid #000000");
     col.style.setProperty("border-collapse","collapse");
     col.style.setProperty("text-align","center");
     col.style.setProperty("vertical-align","middle");
  }
  /**
    *Inserts a line in the list
    *@param key {String} key data
    *@param data {Array} data entered
    *@param enabled {Boolean} is the row enabled?This value is used for new entries, if the key is updated it is ignored
    *@param aux {Object} Object not displayed in the table
  */
  insert(key,data,enabled=true,aux={})
  {
    let line=this.lines_.get(key);
    if(line != undefined)
      enabled=line.enabled;
    this.cancel(key);
    let row=this.tbody_row();
    let i=0;
    let ncols=this.row_ncells(this.hrow_);
    for(let d of data)
    {
      if(i<ncols)
      {
        let col=this.row_td(row,i);
        try
        {
          col.appendChild(d);
        }
        catch(e)
        {
          col.innerHTML=d;
        }
        this.col_style(col);
      }
      else
        break;
      i++;
    }
    while(i<ncols)
    {
      let col=this.row_td(row,i++);
      this.col_style(col);
    }
    this.lines_.set(key,{row:row,enabled:enabled,aux:aux});
    this.order();
    this.display();
  }
  /**
    *Delete a line from the list
    *@param key {String} key data
  */
  cancel(key)
  {
    let line=this.lines_.get(key);
    if(line != undefined)
    {
      line.row.remove();
      this.lines_.delete(key);
    }  
  }
  /**
    *enable lines with input keys
    *@param keys {Array_String} input keys
  */
  enable(keys)
  {
    for(k of key)
    {
      let line=this.lines_.get(key);
      if(line != undefined) 
        line.enabled=true;
    }
    this.display();
  }
  /**
    *Enable lines with input keys and disable everything else
    *@param keys {Array_String} input keys
  */
  config_enable(keys)
  {
    for(const [key,value] of this.lines_)
    {
      if(vmp_invector(key,keys))
        value.enabled=true;
      else
        value.enabled=false;
    }
    this.display();
  }
  /**
    *Return Enabled Row Keys
    *@return {Array_string} enabled key
  */
  get_enabled()
  {
    ret=[]
    for(const [key,value] of this.lines_)
    {
      if(value.enabled)
        ret.push(key); 
    }
    return ret;
  }
  /**
    *Disable lines with input keys
    *@param keys {Array_String} input keys
  */
  disable(keys)
  {
    for(k of key)
    {
      let line=this.lines_.get(key);
      if(line != undefined) 
        line.enabled=false;
    }
    this.display();
  }
  /**
    *Disable lines with input keys and enable everything else
    *@param keys {Array_String} input keys
  */
  config_disable(keys)
  {
    for(const [key,value] of this.lines_)
    {
      if(vmp_invector(key,keys))
        value.enabled=false;
      else
        value.enabled=true;
    }
    this.display();
  }
  /**
    *Return disbaledd Row Keys
    *@return {Array_string} disabled key
  */
  get_disabled()
  {
    ret=[]
    for(const [key,value] of this.lines_)
    {
      if(!value.enabled)
        ret.push(key); 
    }
    return ret;
  }
  /**Internal usage*/
  order()
  {
    if(this.index_ >= 0)
    {
      let sortedRows;
      if(this.order_ == "inc")
        sortedRows = Array.from(this.tbody_.rows).sort((rowA,rowB)=>rowA.cells[this.index_].innerHTML.localeCompare(rowB.cells[this.index_].innerHTML));
      else
         sortedRows = Array.from(this.tbody_.rows).sort((rowA, rowB) =>-rowA.cells[this.index_].innerHTML.localeCompare(rowB.cells[this.index_].innerHTML));
      this.tbody_.append(...sortedRows);
    }
  }
  /**Internal usage*/
  display()
  {
    for(const [key,value] of this.lines_)
    {
      if(value.enabled)
        value.row.style.display="table-row";
      else
        value.row.style.display="none";
    }
  }
  /**Virtual function*/
  reset_impl()
  {
    for(const [key,value] of this.lines_)this.cancel(key);
    this.index_=-1;
    this.order_="";
  }
};
/**Component to insert tags with title and value, or single objects*/
class Vmp_Cmp_Info extends Vmp_Cmp_Table
{
  /**constructor**/
  constructor()
  {
    super();
    this.style.setProperty("height","auto");
    this.style.setProperty("table-layout","auto");
    this.style.setProperty("overflow","auto");
  }
  /**Internal usage*/
  col_style(col)
  {
    col.style.setProperty("text-align","left");
    col.style.setProperty("padding-top","5px");
    col.style.setProperty("padding-left","5px");
  }
  /**
    *Adds a value title pair
    *@param title {Tag_or_String} title
    *@param value {Tag_or_String} value
    *@param cols {Size} Number of Value Columns
  **/
  add_line(title,value,cols=1)
  {
    let row=this.tbody_row();
    let col=this.row_td(row,0);
    try
    {
      col.appendChild(title);
    }
    catch(e)
    {
      col.innerHTML=title;
    }
    this.col_style(col);
    col=this.row_td(row,1);
    col.setAttribute("colspan",cols);
    try
    {
      col.appendChild(value);
    }
    catch(e)
    {
      col.innerHTML=value;
    }
    this.col_style(col);
  }
  /**
    *Adds values to the info
    *@param title {Tag_or_String} title
    *@param values {Array_Tag} value
  **/
  add_multilines(title,values)
  {
    let row=this.tbody_row();
    let col=this.row_td(row,0);
    try
    {
      col.appendChild(title);
    }
    catch(e)
    {
      col.innerHTML=title;
    }
    this.col_style(col);
    for (let i=0; i<values.length;i++)
    {
      col=this.row_td(row,i+1);
      try
      {
        col.appendChild(values[i]);
      }
      catch(e)
      {
        col.innerHTML=values[i];
      }
      this.col_style(col);
    }
  }
  /**
    *Adds a single value in the center
    *@param value {Tag_or_String} value
    *@param cols {Size} Number of column it includes
  **/
  add_single_line(value,cols=2)
  {
    let row=this.tbody_row()
    let col=this.row_td(row,0);
    col.setAttribute("colspan",cols);
    try
    {
      col.appendChild(value);
    }
    catch(e)
    {
      col.innerHTML=value;
    }
    col.style.setProperty("text-align","center");
    col.style.setProperty("padding-top","5px");
  }
  /**virtual function*/
  reset_impl(){this.tbody_.innerHTML="";}
};
/**Component used for project presentation*/
class Vmp_Cmp_Top extends Vmp_Cmp_Table
{
  /**constructor**/
  constructor()
  {
    super();
    this.setAttribute("width","100%");
  }
  /**
    *Init component
    *@param img {String} Image paths
    *@param title {String} Project title
    *@param info {Tag} Info tag
  */
  init(img,title,info)
  {
    let irow=this.tbody_row();
    this.tbody_row();
    this.tbody_row();
    let trow=this.tbody_row();
    let col=this.row_td(irow);
    col.setAttribute("width","50%");
    col.setAttribute("rowspan","3");
    let center=vmp_wrap_createElement("center");
    /**
      *Logo image
      *@public
    */
    this.tagimg=vmp_cmp_img(img,"project logo");
    this.tagimg.setAttribute("width","100px");
    this.tagimg.setAttribute("height","56px");
    center.appendChild(this.tagimg);
    col.appendChild(center);
    let ttitle=this.row_td(trow);
    ttitle.setAttribute("width","50%");
    ttitle.innerHTML=`<center><strong>${title}</strong></center>`;
    this.info_=this.row_td(irow);
    this.info_.setAttribute("width","50%");
    this.info_.setAttribute("rowspan","4");
    this.change_info(info);
  }
  /**
    *@param info {Tag} info tag
  */
  change_info(info){vmp_tag_childReplace(this.info_,info,0);}
};
try
{
  vmp_wrap_define("vmp-password",Vmp_Cmp_Password,'div');
  vmp_wrap_define("vmp-menu",Vmp_Cmp_Menu,'div');
  vmp_wrap_define("vmp-list",Vmp_Cmp_List,'table');
  vmp_wrap_define("vmp-info",Vmp_Cmp_Info,'table');
  vmp_wrap_define("vmp-top",Vmp_Cmp_Top,'table');
}
catch(error)
{
  vmp_wrap_alert(error);
}
/**
  *Create Component input text
  *@param size {Size} visual size
  *@param placeholder {String} containing a hint to the user of what can be entered in the control
  *@param default_v {String} default value
  *@return {Tag} component password
*/
function vmp_cmp_password(size,placeholder,default_v="")
{
  let ret=vmp_wrap_createElement("vmp-password",'div');
  ret.init(size,placeholder,default_v);
  return ret;
}
/**
  *Build menu component
  *@return {Tag} Menu component
*/
function vmp_cmp_menu(){return vmp_wrap_createElement("vmp-menu",'div');}
/**
  *Build new component list
  *@param cols {Object} Objects with fields {name:"column name",width:"column width"(default not assigned),order:Can the data be sorted?(default false)}
  *@return {Tag} Menu component
*/
function vmp_cmp_list(...cols)
{
  let ret=vmp_wrap_createElement("vmp-list",'table');
  ret.init(...cols);
  return ret;
}
/**
  *Build new component info
  *@return {Tag} component info
*/
function vmp_cmp_info(){return vmp_wrap_createElement("vmp-info",'table');}
/**
  *Build new component Top
  *@param img {String} Image paths
  *@param title {String} Project title
  *@param info {Tag} Info tag
  *@return {Tag} component top
*/
function vmp_cmp_top(img,title,info)
{
  let ret=vmp_wrap_createElement("vmp-top",'table');
  ret.init(img,title,info);
  return ret;
}
