/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 16/01/2025
 */

#ifndef VAMPIRIA_VMP_UTILS_H

#define VAMPIRIA_VMP_UTILS_H 1

namespace vampiria { namespace vmp { namespace utils {

//!Container object utility.Pointer memory must be dynamically freed outside the class.
template<class KEY>
class Container
{
    private:
        vmp::Table<KEY,void *> vars_;/*!< Data Container*/
    public:
        //! A constructor
        Container() {}

        //! A destructor
        ~Container(){vars_.clear();}

        //! Insert a value in container
        /*!
            @param key key value
            @param value pointer data value
            @sa vars_
            @return except error
        */        
        template<class VALUE>        
        void setvar(KEY key,VALUE *value)
        {
            try
            {
                vars_.insert(key,(void *)value);
 
            }
            catch(vmp::exception &x)
            {
                vmp::except_s("Container insert error key already");
            }
        }
                
        //!Returns a value of the container
        /*!
            @param key input key
            @sa vars_
            @return Returns the value associated with the key value if there,0 otherwise        
        */        
        template<class VALUE>
        VALUE *getvar(KEY key)
        {
            void *ret;
            if(vars_.search(key,&ret))
               return (VALUE *)ret;
            return 0;
        }
                
        //!Remove a value of the container
        /*!
            @param key input key
            @sa vars_
            @return Returns the value associated with the key value if there,0 otherwise        
        */
        template<class VALUE>
        VALUE *remvar(KEY key)
        {
            void *ret;
            try
            {
                vars_.cancel(key,&ret);
            }
            catch(vmp::exception &x)
            {
                return 0;
            }
            return (VALUE *)ret;
        }

        //! returns all container keys
        /*!
            @sa vars_            
            @return container keys        
        */
        vmp::vector<KEY> all_keys()
        {
            return vars_.all_keys();
        }
};

//! Storage free object utility
template <class Obj>
class Storage
{
    private:
        vmp::vector<Obj *> free_;/*!< Data storage*/    
    public:
        //! A constructor
        Storage(){}
        
        //! A destructor
        ~Storage()
        {
            for(vmp_index i=0;i<free_.size();i++)
                delete free_[i];
            free_.clear();
        }

        //*! Get object,if it is not on the build storage
        /*!
            @sa free_            
            @return object storage
        */        
        Obj *get()
        {
            Obj *ret;
            if(free_.size() == 0)
                ret=new Obj();
            else
            {
                ret=free_[free_.size() -1];
                free_.pop_back();
            }
            return ret;
        }
        
        //*! Get object,if it is not storage return 0
        /*!
            @sa free_            
            @return object storage or 0
        */
        Obj *get_free()
        {
            Obj *ret=0;
            if(free_.size() != 0)
            {
               ret=free_[free_.size() -1];
               free_.pop_back();    
            }
            return ret;
        }

        //*! Insert object in storage
        /*!
            @param object object to free        
        */
        void free(Obj *object)
        {
            if(object != 0)
               free_.push_back(object);
        }
};

//! Logger object utility
class Logger
{
    private:
        vmp::str logname_; /*!<File log path*/
        vmp_byte loglevel_; /*! Level log*/      
    public:
        //! A constructor        
        Logger();
        
        //! A destructor        
        ~Logger();

        //! Reset logger        
        void reset();
        
        //! Returns file log path
        /*!
            @sa logname_
            @return file log path
        */
        vmp::str logname();
        
        //! Returns file log level
        /*!
            @sa loglevel_
            @return file log level
        */
        vmp_byte loglevel();

        //! Setting logger
        /*!
            @param filename File log path
            @param facility Level log
            @param newfile true new file empty create,otherwise concatenared data
            @sa logname_
            @sa loglevel_
            @return exception in case of error    
        */        
        void set(vmp::str filepath,vmp_byte facility,vmp_bool newfile);
                
        //! Write log file(string mode)
        /*!
            @param facility level log data
            @param value write data
            @sa logname_            
            @sa loglevel_
        */
        void write_s(vmp_byte facility,vmp::str value);

        //! Write log file(format mode)
        /*!
            @param facility level log data
            @param fmt format string
            @param ... format args
            @sa logname_            
            @sa loglevel_
        */
        void write(vmp_byte facility,const vmp_char *fmt,...);
        
        //!Rotate log files, backups are named.log. x where x is the rotation number
        /*!
            @param n number of rotation files
        */
        void rotate(vmp_index n);
};

const vmp_byte LOG_NO=0x00;    /*!< Log type No*/
const vmp_byte LOG_DEBUG=0x01; /*!< Log type Debug*/
const vmp_byte LOG_INFO=0x02;  /*!< Log type Info*/
const vmp_byte LOG_WARN=0x03;  /*!< Log type Warn*/
const vmp_byte LOG_ERR=0x04;   /*!< Log type Err*/

//! Build Logger for debugging library
/*!
    @path syslog path
    @return void or except in case of failure
*/
void syslog_build(vmp::str filepath);

//! Write log file(string mode)
/*!
    @param facility level log data
    @param value write data
*/
void syslog_s(vmp_byte facility,vmp::str value);

//! Write syslog file(format mode)
/*!
    @param facility level log data
    @param fmt format string
    @param ... format args
*/
void syslog(vmp_byte facility,const vmp_char *fmt,...);

}}

}

#endif

