/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 27/12/2024
 */

#include "vmp.h"

namespace vampiria { namespace vmp {

vmp_int signum(vmp::str sigstr)
{
    if(sigstr == "sighup") 
        return SIGHUP;
    else if(sigstr == "sigint") 
        return SIGINT;
    else if(sigstr == "sigquit")
        return SIGQUIT;
    else if(sigstr == "sigill")
        return SIGILL;
    else if(sigstr == "sigabrt")
        return SIGABRT;
    else if(sigstr =="sigfpe") 
        return SIGFPE;
    else if(sigstr =="sigkill")
        return SIGKILL;
    else if(sigstr =="sigsegv")
        return SIGSEGV;
    else if(sigstr =="sigpipe")
        return SIGPIPE;
    else if(sigstr =="sigalrm")
        return SIGALRM;
    else if(sigstr =="sigterm")
        return SIGTERM;
    else if(sigstr == "sigusr1")
        return SIGUSR1;
    else if(sigstr =="sigusr2")
        return SIGUSR2;
    else if(sigstr =="sigchld")
        return SIGCHLD;
    else if(sigstr =="sigcont")
        return SIGCONT;
    else if(sigstr =="sigtop")
        return SIGSTOP;
    else if(sigstr =="sigtsp")
        return SIGTSTP;
    else if(sigstr == "sigttin")
        return SIGTTIN;
    else if(sigstr =="sigttou")
        return SIGTTOU;
    vmp::except("Invalid signal '%s'",sigstr.c_str());
    return -1;
}

vmp::str sigstr(vmp_int sig)
{
    switch(sig)
    {
        case SIGHUP:
            return "sighup";
        case SIGINT:
            return "sigint";
        case SIGQUIT:
            return "sigquit";
        case SIGILL:
            return "sigill";
        case SIGABRT:
            return "sigabrt";
        case SIGFPE:
            return "sigfpe";
        case SIGKILL:
            return "sigkill";
        case SIGSEGV:
            return "sigsegv";
        case SIGPIPE:
            return "sigpipe";
        case SIGALRM:
            return "sigalrm";
        case SIGTERM:
            return "sigterm";
        case SIGUSR1:
            return "sigusr1";
        case SIGUSR2:
            return "sigusr2";
        case SIGCHLD:
            return "sigchld";
        case SIGCONT:
            return "sigcont";
        case SIGSTOP:
            return "sigtop";
        case SIGTSTP:
            return "sigtsp";
        case SIGTTIN:
            return "sigttin";
        case SIGTTOU:
            return "sigttou";
        default: 
            vmp::except("Invalid signal number '%d'",sig);
    }
    return "";
}

vmp_bool sigcmp(vmp::str sig,vmp_int isig)
{
    try
    {
        if(vmp::signum(sig) == isig)
            return true;
    }
    catch(vmp::exception &x)
    {
    }
    return false;    
}

static vmp::Table<vmp::str,struct sigaction *> sigaction_;

void signal_wrap(vmp::vector<vmp::str> sig,void (*callback)(vmp_int))
{
    vmp_assert(_POSIX_C_SOURCE);
    for(vmp_index i=0;i<sig.size();i++)
    {
        struct sigaction *act=(struct sigaction *)vmp::malloc_wrap(sizeof(struct sigaction));
        act->sa_flags = 0;
        act->sa_handler = callback;
        sigemptyset(&act->sa_mask);
        if(sigaction(vmp::signum(sig[i]),act,0) == -1)
        {    
            vmp::free_wrap((void **)&act);
            vmp::except_errno();
        }
        if(vmp::sigaction_.update(sig[i],&act))
            vmp::free_wrap((void **)&act); 
    }
}

void pthread_sigblock_wrap(vmp::vector<vmp::str> sig)
{
    vmp_int ret;
    sigset_t set;
    sigemptyset(&set);
    for(vmp_index i=0;i < sig.size();i++)
        sigaddset(&set,vmp::signum(sig[i]));
    if((ret=pthread_sigmask(SIG_BLOCK,&set, NULL)) != 0)
        vmp::except_errno_set(ret);
}

}}

