/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 30/01/2025
 */
 
#ifndef VAMPIRIA_VMP_LIB_PROC_H

#define VAMPIRIA_VMP_LIB_PROC_H 1

namespace vampiria { namespace vmp {

//! User id type
typedef uid_t Uid;

//! User id type
typedef gid_t Gid;

//! Process id type
typedef pid_t Pid;

//!Causes the calling thread to sleep either until the number of real-time seconds
/*!
    @param seconds seconds to sleep thread
*/
void sleep_wrap(vmp_uint seconds);

//!causes the calling process (or thread) to sleep until a signal is delivered that either terminates the process or causes  the invocation of a signal-catching function.
void pause_wrap();

//!Suspends execution of the calling thread for useconds microseconds
/*!
    @param useconds micro seconds to sleep
*/
void usleep_wrap(vmp_uint useconds);

//! Fork wrapper; Create child process;
/*!
    @return pid=0 in child process;pid=child process; or except in case of failure
*/
vmp::Pid fork_wrap();

//!Returns the process ID (PID) of the calling process.
/*!
    @return ID process
*/
vmp::Pid getpid_wrap();

//!Returns the real user ID of the calling process
/*!
    @return user id
*/
vmp::Uid getuid_wrap();

//!Returns the effective user ID of the calling process
/*!
    @return user id
*/
vmp::Uid geteuid_wrap();

//!Returns the real group ID of the calling process
/*!
    @return group id
*/
vmp::Gid getgid_wrap();

//!Returns the effective group ID of the calling process
/*!
    @return group id
*/
vmp::Gid getegid_wrap();

//!Returns a pointer to a structure containing the broken-out fields of the record in the password database  that matches the username
/*!
    @param username username input
    @return struct passwd or except in case of failure 
*/
struct passwd *getpwnam_wrap(vmp::str username);

//!Returns the user's userid in input
/*!
    @param username username input
    @return userid or except in case of failure
*/
vmp::Uid getuid_user(vmp::str username);

//!Returns the user's groupid in input
/*!
    @param username username input
    @return groupid or except in case of failure
*/
vmp::Gid getgid_user(vmp::str username);

//! Sets the  effective user ID of the calling process
/*!
   @param uid user id
   @return void or except in case of failure
*/
void setuid_wrap(vmp::Uid uid);

//! Sets the  effective user GID of the calling process
/*!
   @param gid group id
   @return void or except in case of failure
*/
void setgid_wrap(vmp::Gid gid);

//! Sets the  effective user ID of the calling process of the input user
/*!
   @param username input user
   @return void or except in case of failure
*/
void setuid_user(vmp::str username);

//! Sets the  effective group ID of the calling process of the input user
/*!
   @param username input user
   @return void or except in case of failure
*/
void setgid_user(vmp::str username);

//! Used to send any signal to any process group or process
/*!
    @param pid process id
    @param sig signal in string format ex. "sigint" 
*/
void kill_wrap(vmp::Pid pid,vmp::str sig);

//! Waiting  the terminated process child
/*!
     @param pid pointer to pid process setting -1 after call
     @return exit status process child 
*/
vmp_int waitpid_wrap(vmp::Pid *pid);

//! Create a pipe and assign the filedescriptor to piper and pipew
/*!
    @param piper pipe read output
    @param pipew pipe write output
    @return void or except in case of failure
*/
void pipe_wrap(vmp_int *piper,vmp_int *pipew);

//!Finds an available pseudoterminal and return the file descriptors for the master and slave in master  and  slave
/*!
    @param master the pty master returned
    @param slave the pty slave returned
    @return void or except in case of failure
*/
void pty_wrap(vmp_int *master,vmp_int *slave);

//! Exec a command and return after the command has been completed
/*!
    @param cmd input command
    @return void or except in case of failure
*/
void system_wrap(vmp::str cmd);

//! Executes the program pointed to by filename.  This causes the program that is currently being run by the calling process  to  be  re‐placed  with  a new program,with newly initialized stack, heap, and (initialized and uninitialized) data segments.
/*!
    @param filename binary executable
    @param argv an array of argument strings passed to the new program
    @param envp an array of strings, conventionally of the form key=value, which are  passed an environment to the new program
    @return void or except in case of failure
*/
void execve_wrap(vmp::str filename,vmp_char *const argv[],vmp_char *const envp[]);

//! Function parses the command-line arguments
/*!
    @param argc Total number of process input arguments
    @param argv Arguments list
    @param optstring options string
    @return  If an option was successfully found, then getopt() returns  the  option
             character.  If all command-line options have been parsed, then getopt()
             returns -1.  If getopt() encounters an option character that was not in
             optstring, then '?' is returned
*/
vmp_int getopt_wrap(vmp_int argc, vmp_char *const argv[],vmp::str optstring);

}}

#endif

