/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 09/05/2024
 */
 
#ifndef VAMPIRIA_VMP_LIB_FD_H

#define VAMPIRIA_VMP_LIB_FD_H 1

namespace vampiria { namespace vmp {

//!Creates a copy of the file descriptor oldfd it uses the file descriptor number specified in newfd.
/*!
    @param oldfd old file descriptor
    @param newfd new file descriptor in old
    @return void or except in case of failure
*/
void dup2_wrap(vmp_int oldfd,vmp_int newfd);

//! Set a filedescriptor in non-block mode
/*!
    @param fd input fd
*/
void fd_noblock(vmp_int fd);

//! Set a filedescriptor in block mode
/*!
    @param fd input fd
*/
void fd_block(vmp_int fd);

//!Transfers ("flushes") all modified in-core data of filedescriptor
/*!
    @param fd  input fd
*/ 
void fd_fsync(vmp_int fd);

//!Reads the data of a filedescriptor in a buf
/*!
    @param fd filedescriptor from read
    @param buf read buf
    @return Number of bytes read or 0 if fd close or -1 if errno==EAGAIN(no blocking no data) or errno=EINTR(signal interrupt)or except in case of failure  
*/
vmp_int fd_read_buf(vmp_int fd,vmp::Buf *buf);

//!Reads the data of a filedescriptor in a vector
/*!
     @param fd filedescriptor from read
     @param bytes output reading(memory already allocated)
     @param count maximum number of bytes read
     @return Number of bytes read or 0 if fd close or -1 if errno==EAGAIN(no blocking no data) or errno=EINTR(signal interrupt)or except in case of failure   
*/
vmp_int fd_read_bytes(vmp_int fd,vmp_byte *bytes,vmp_size count);

//!Read data from filedescriptor
/*!
    @param fd filedescriptor from read
    @param ret read data
    @return Number of bytes read or 0 if fd close or -1 if errno==EAGAIN(no blocking no data) or errno=EINTR(signal interrupt)or except in case of failure
*/
vmp_int fd_read(vmp_int fd,vmp::str *ret);

//! Write data to filedescriptor in a buf
/*!
    @param fd filedescriptor to write
    @param buf buf to write
    @return Number of bytes write or except in case of failure
*/
vmp_int fd_write_buf(vmp_int fd,vmp::Buf *buf);

//!Write data to filedescriptor in bytes
/*!
    @param fd filedescriptor to write
    @param bytes bytes to write
    @param count bytes number to write
    @return Number of bytes write or except in case of failure
*/
vmp_int fd_write_bytes(vmp_int fd,vmp_byte *bytes,vmp_size count);

//! Write data to filedescriptor
/*!
    @param fd filedescriptor to write
    @param data data to write
    @return Number of bytes write or except in case of failure
*/
vmp_int fd_write(vmp_int fd,vmp::str data);

//! Close fd and assign -1 value
/*!
    @param fd pointre filedecriptor to close
*/
void fd_close(vmp_int *fd);

//!Allow a program to  monitor  multiple  filedescriptors(select wrapper)
class Select_FD
{
    private:
        vmp_int maxfd_;  /*!< Max fd in list*/
        fd_set frds_;     /*!< List file descriptors read*/
        fd_set fwds_;     /*!< List file descriptors write*/
        fd_set feds_;     /*!< List file descriptors exception*/
    public:
        //! A costructor
        Select_FD();

        //! A destructor
        ~Select_FD();

        //!Clears a setting filedescriptors list
        void reset();
 
        //!Adding filedescriptor in reading list to monitoring
        /*!
            @param fd filedescrptors in input
        */  
        void fd_add(vmp_int fd);

        //!Adding filedescriptor in writing list to monitoring
        /*!
            @param fd filedescrptors in input
        */  
        void fd_add_w(vmp_int fd);

        //!Adding filedescriptor in exception list to monitoring
        /*!
            @param fd filedescrptors in input
        */  
        void fd_add_e(vmp_int fd);
      
        //!Monitoring multiple file descriptors and exit when an event occurs
        void fd_monitor(vmp::time::Time timeout=0.0);
};

}}

#endif

