/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 09/05/2024
 */

#include "vmp.h"

namespace vampiria { namespace vmp {

void dup2_wrap(vmp_int oldfd, vmp_int newfd)
{
    if(dup2(oldfd,newfd) == -1)
        vmp::except_errno();
}



void fd_noblock(vmp_int fd)
{
    fcntl(fd,F_SETFL,fcntl(fd,F_GETFL, 0)|O_NONBLOCK);
}

void fd_block(vmp_int fd)
{
    fcntl(fd,F_SETFL,fcntl(fd,F_GETFL, 0) & ~O_NONBLOCK);
}

void fd_fsync(vmp_int fd)
{
    if(fsync(fd) == -1)
        vmp::except_errno();
}

vmp_int fd_read_buf(vmp_int fd,vmp::Buf *buf)
{
    vmp_int tmp,retval=0;
    buf->reset();
    while(1)
    {
        buf->newsize(buf->size()+vmp::MAXSIZEREAD);
        tmp=read(fd,buf->pointer(retval),vmp::MAXSIZEREAD);
        if(tmp == -1)
        {
             if((errno == EAGAIN) || (errno == EINTR))
             {
                 if(retval==0)
                     return -1;
                 else
                     break;
             }
             buf->reset();
             vmp::except_errno();
        }
        retval+=tmp;
        if((vmp_uint) tmp < vmp::MAXSIZEREAD)
            break;
    }
    buf->newsize(retval);
    return retval;
}

vmp_int fd_read_bytes(vmp_int fd,vmp_byte *bytes,vmp_size count)
{
    vmp_int retval=read(fd,(void *)bytes,count);
    if((retval == -1) && (!((errno == EAGAIN) || (errno == EINTR))))
        vmp::except_errno();
    return retval;
}

vmp_int fd_read(vmp_int fd,vmp::str *ret)
{
    vmp::Buf buf;
    vmp_int retval=vmp::fd_read_buf(fd,&buf);
    if(retval > 0)
       (*ret)=buf.read_str(retval);
    else
       (*ret)="";
    return retval;
}

vmp_int fd_write_buf(vmp_int fd,vmp::Buf *buf)
{
    vmp_int retval;
    if(buf != 0)
       if((retval=write(fd,buf->pointer(),buf->size())) == -1)
          vmp::except_errno();
    return retval;  
}

vmp_int fd_write_bytes(vmp_int fd,vmp_byte *bytes,vmp_size count)
{
    return write(fd,(void *)bytes,count);
}

vmp_int fd_write(vmp_int fd,vmp::str data)
{
    vmp::Buf buf;
    buf.write_str(data);
    return vmp::fd_write_buf(fd,&buf);
}

void fd_close(vmp_int *fd)
{
     if(*fd >= 0)
        close(*fd);
     (*fd)=-1;
}

Select_FD::Select_FD()
{
    reset();
}

Select_FD::~Select_FD()
{
    reset();
}

void Select_FD::reset()
{
    maxfd_=-1;
    FD_ZERO(&frds_);
    FD_ZERO(&fwds_);
    FD_ZERO(&feds_);
}

void Select_FD::fd_add(vmp_int fd)
{
    if(fd >= 0)
    {   
        FD_SET(fd,&frds_);
        if(fd > maxfd_)
            maxfd_=fd;
    }
}

void Select_FD::fd_add_w(vmp_int fd)
{
    if(fd >= 0)
    {   
        FD_SET(fd,&fwds_);
        if(fd > maxfd_)
            maxfd_=fd;
    }
}

void Select_FD::fd_add_e(vmp_int fd)
{
    if(fd >= 0)
    {   
        FD_SET(fd,&feds_);
        if(fd > maxfd_)
            maxfd_=fd;
    }
}

void Select_FD::fd_monitor(vmp::time::Time timeout)
{
    if(maxfd_ == -1)
       vmp::except_s("Filedescriptors list is empty in monitoring");
    vmp::time::Timeval tv=vmp::time::time_to_timeval(timeout);
    select(maxfd_+1,&frds_,&fwds_,&feds_,tv);
    vmp::time::timeval_free(&tv);
}

}}

