/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 *Date last update: 28/10/2024
 */

#include "net.h"

namespace vampiria { namespace net {

TcpUI::TcpUI(event::Manager *manager):UI(manager)
{
     set_event_client(0,0,0);
     set_event_server(0,0,0,0); 
     set_event_bind(0,0,0,0);  
}

TcpUI::~TcpUI()
{
}

vmp::str TcpUI::identity(event::Cell *cell)
{
    vmp::str ret="";
    net::EventConnection *conn=cell->event<net::EventConnection>();
    vmp_int type=conn->evt_connection_type();
    vmp::str strtype=conn->evt_connection_strtype();
    net::Address *local=conn->evt_connection_localaddr(),*peer,*bind;
    if(type == net::CONN_TCPLISTEN)
        vmp::unicode::str_write(&ret,"%s:[%s:%s]",strtype.c_str(),local->host().c_str(),local->service().c_str());
    else if((type == net::CONN_TCPSERVER) || (type == net::CONN_TCPCLIENT))
    {
        peer=conn->evt_connection_peeraddr();
        vmp::unicode::str_write(&ret,"%s:[%s:%s]<->[%s:%s]",strtype.c_str(),local->host().c_str(),local->service().c_str(),
                                                            peer->host().c_str(),peer->service().c_str());
    }
    else if(type ==net::CONN_TCPPROXY)
    {
        peer=conn->evt_connection_peeraddr();
        vmp::unicode::str_write(&ret,"%s:[%s:%s]<->[proxy]<->[%s:%s]",strtype.c_str(),local->host().c_str(),local->service().c_str(),peer->host().c_str(),peer->service().c_str());
    }
    else if(type ==net::CONN_TCPBIND)
    {
        peer=conn->evt_connection_peeraddr();
        bind=conn->evt_connection_bindaddr();
        vmp::unicode::str_write(&ret,"%s:[%s:%s]<->[%s:%s]",strtype.c_str(),bind->host().c_str(),bind->service().c_str(),peer->host().c_str(),peer->service().c_str());
    }
    else if(type==net::CONN_TCPBINDPROXY)
    {
        peer=conn->evt_connection_peeraddr();
        bind=conn->evt_connection_bindaddr();
        vmp::unicode::str_write(&ret,"%s:[local]<->[proxy %s:%s]<->[%s:%s]",strtype.c_str(),bind->host().c_str(),bind->service().c_str(),peer->host().c_str(),peer->service().c_str());
    }
    return ret;
}
       
void TcpUI::close_event(event::Cell *cell)
{
    net::EventConnection *conn=cell->event<net::EventConnection>();   
    conn->evt_connection_close();
}
       
void TcpUI::free_ref(event::Cell *cell)
{
    net::EventConnection *conn=cell->event<net::EventConnection>();
    conn->evt_connection_free();
    cref_.free(conn);
}

event::Event *TcpUI::child_event_new(event::Cell *cell)
{
    return (event::Event *) cref_.get();  
}

void TcpUI::set_event_client(event::EVTCB connect,event::EVTCBBUF crecv,event::EVTCB cclose)
{
    connect_=connect;
    crecv_=crecv;
    cclose_=cclose;
}

void TcpUI::set_event_server(net::EVTCBACCEPT acptevent,event::EVTCB svlcevent,event::EVTCBBUF revent,event::EVTCB svcevent)
{
    acptevent_=acptevent;
    svlcevent_=svlcevent;
    svrevent_=revent;
    svcevent_=svcevent;
}

void TcpUI::set_event_bind(event::EVTCB bind,event::EVTCB bindconnect,event::EVTCBBUF bindrecv,event::EVTCB bindclose)
{
    bind_=bind;
    bindconnect_=bindconnect;
    bindrecv_=bindrecv;
    bindclose_=bindclose;
}

event::Cell *TcpUI::new_client(net::Address *peer,net::ProxyChain *proxy)
{
    event::Cell *cell=0;
    manager_->lock();
    net::EventConnection *conn=cref_.get();
    try
    {
        cell=conn->evt_connection_tcpclient(this,peer,connect_,crecv_,cclose_,proxy);
    }
    catch(vmp::exception &x)
    {
        cref_.free(conn);
        manager_->unlock();
        vmp::except("event::TcpUI::new_client() '%s'",x.what());
    }
    manager_->unlock();
    return cell;
}

event::Cell *TcpUI::new_server(net::Address *local,vmp_size backlog)
{
    event::Cell *cell=0;
    manager_->lock();
    net::EventConnection *conn=cref_.get();
    try
    {
        cell=conn->evt_connection_tcplisten(this,local,backlog,acptevent_,svlcevent_,svrevent_,svcevent_);
    }
    catch(vmp::exception &x)
    {
        cref_.free(conn);
        manager_->unlock();
        vmp::except("event::TcpUI::new_server() '%s'",x.what());
    }
    manager_->unlock();
    return cell;
}

event::Cell *TcpUI::new_bind(net::Address *bindref,net::Address *bindrefservice,vmp::time::Time bindwait,net::ProxyChain *proxy)
{
    event::Cell *cell=0;
    manager_->lock();
    net::EventConnection *conn=cref_.get();
    try
    {
        cell=conn->evt_connection_tcpbind(this,bindref,bindrefservice,bind_,bindconnect_,bindrecv_,bindclose_,bindwait,proxy);
    }
    catch(vmp::exception &x)
    {
        cref_.free(conn);
        manager_->unlock();
        vmp::except("event::TcpUI::new_bind() '%s'",x.what());
    }
    manager_->unlock();
    conn->evt_connection_tcpbind_bind();
    return cell;
}

void TcpUI::send(event::Cell *cell,vmp::Buf *buf)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventConnection *conn=cell->event<net::EventConnection>();
    conn->evt_connection_send(buf);
    manager->unlock();
}

net::Address *TcpUI::localaddr(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventConnection *conn=cell->event<net::EventConnection>();
    net::Address *ret=conn->evt_connection_localaddr();
    manager->unlock();
    return ret;
}

net::Address *TcpUI::peeraddr(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventConnection *conn=cell->event<net::EventConnection>();
    net::Address *ret=conn->evt_connection_peeraddr();
    manager->unlock();
    return ret; 
}

net::Address *TcpUI::bindaddr(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventConnection *conn=cell->event<net::EventConnection>();
    net::Address *ret=conn->evt_connection_bindaddr();
    manager->unlock();
    return ret; 
}

net::Address *TcpUI::bindservice(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventConnection *conn=cell->event<net::EventConnection>();
    net::Address *ret=conn->evt_connection_bindservice();
    manager->unlock();
    return ret; 
}

net::ProxyChain *TcpUI::proxychain(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventConnection *conn=cell->event<net::EventConnection>();
    net::ProxyChain *ret=conn->evt_connection_proxychain();
    manager->unlock();
    return ret; 
}

vmp_int TcpUI::connection_type(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventConnection *conn=cell->event<net::EventConnection>();
    vmp_int ret=conn->evt_connection_type();
    manager->unlock();
    return ret;
}

vmp::str TcpUI::connection_strtype(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::EventConnection *conn=cell->event<net::EventConnection>();
    vmp::str ret=conn->evt_connection_strtype();
    manager->unlock();
    return ret;
}

}}

