/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 28/10/2024
 */

#ifndef VAMPIRIA_NET_CONNECTION_PROXY_H

#define VAMPIRIA_NET_CONNECTION_PROXY_H 1

namespace vampiria { namespace net {

const vmp_byte socks_cd_connect=0x01;/*!<Socks command code connect*/
const vmp_byte socks_cd_bind=0x02;/*!<Socks command code bind*/
const vmp_byte socks_cd_udpassociate=0x03;/*!<Socks command code udp associate*/

const vmp_byte socks4_result_granted=0x90;/*!<Socks4 reply request granted*/
const vmp_byte socks4_result_failed=0x91;/*!<Socks4 reply request rejected or failed*/
const vmp_byte socks4_result_reject=0x92;/*!<Socks4 reply request rejected because SOCKS server cannot connect to identd on the client*/
const vmp_byte socks4_result_rejectuserid=0x93;/*!<Socks4 reply request rejected because the client program and identd report different user-ids*/

const vmp_byte socks5_auth_noauth=0x00;/*!<Socks5 no authentication method*/
const vmp_byte socks5_auth_gssapi=0x01;/*!<Socks5 gssapi authentication method*/
const vmp_byte socks5_auth_user=0x02;/*!<Socks5 username/password authentication method*/
const vmp_byte socks5_auth_noaccetable=0xFF;/*!<Socks5 no accetable methods,for reply error*/

const vmp_byte socks5_atype_ipv4=0x01;/*!<Socks5 address type ipv4*/
const vmp_byte socks5_atype_domain=0x03;/*!<Socks5 address type domain name*/
const vmp_byte socks5_atype_ipv6=0x04;/*!<Socks5 address type ipv6*/

const vmp_byte socks5_result_ok=0x00;/*!<Socks5 result request granted*/
const vmp_byte socks5_result_failure=0x01;/*!<Socks5 result general SOCKS server failure */
const vmp_byte socks5_result_notallowed=0x02;/*!<Socks5 result connection not allowed by ruleset*/
const vmp_byte socks5_result_netunreachable=0x03;/*!<Socks5 result network unreachable*/
const vmp_byte socks5_result_hostunreachable=0x04;/*!<Socks5 result host unreachable*/
const vmp_byte socks5_result_refused=0x05;/*!<Socks5 result connection refused*/
const vmp_byte socks5_result_ttlexpired=0x06;/*!<Socks5 result ttl expired*/
const vmp_byte socks5_result_badcmd=0x07;/*!<Socks5 result command not supported*/
const vmp_byte socks5_result_badatype=0x08;/*!<Socks5 result address type not supported*/

const vmp_byte socks_internal_badauth=0xF0;/*!<Internal error (no protocol value) Recv bad authentication code*/
const vmp_byte socks_internal_badaddr=0xF1;/*!<Internal error (no protocol value) Reply bad address*/
const vmp_byte socks_internal_accessdenied=0xF2;/*!<Internal error (no protocol value) Access Denied*/

//! Takes in input a Socks4 reply result or Socks5 reply result or an internal error is returns the associated string 
/*!
    @param result result code
    @return result string
*/
vmp::str socks_msg_result(vmp_byte result);

//! Create a socks4/4a request message in the output buffer
/*!
    @param out output buffer
    @param cd command code
    @param address request address
    @param userid userid request,for socks4a message is ignored 
    @return void or except in case of failure 
*/
void socks4_request(vmp::Buf *out,vmp_byte cd,net::Address *address,vmp::str userid="");

//! Reads a socks4/4a request message from an input buffer
/*!
    @param input input buffer
    @param address output address(the address output is setting onlyhost if domain name is received(see net::Address))
    @param userid get userid request,for socks4a message and userid if value==0 is ignored 
    @return command code or except in case of failure
*/
vmp_byte socks4_request_get(vmp::Buf *input,net::Address *address,vmp::str *userid=0);

//! Create a socks4/4a request message in the output buffer
/*!
    @param out output buffer
    @param cd result code
    @param address request address
    @return void or except in case of failure 
*/
void socks4_reply(vmp::Buf *out,vmp_byte cd,net::Address *address=0);

//! Reads a socks4/4a reply message from an input buffer
/*!
    @param input input buffer
    @param address output address
    @return result code or except in case of failure
*/
vmp_byte socks4_reply_get(vmp::Buf *input,net::Address *address);

//! Create a socks5 client init message in the output buffer
/*!
    @param out output buffer
    @param methods authentication methods list
    @return void or except in case of failure 
*/
void socks5_clientinit(vmp::Buf *out,vmp::vector<vmp_byte> auths);

//! Reads a socks5 client init message from an input buffer
/*!
    @param input input buffer
    @return authentication methods list or except in case of failure 
*/
vmp::vector<vmp_byte> socks5_clientinit_get(vmp::Buf *input);

//! Create a socks5 server init message in the output buffer
/*!
    @param out output buffer
    @param auth authentication method accepted
    @return void or except in case of failure 
*/
void socks5_serverinit(vmp::Buf *out,vmp_byte auth);

//! Reads a socks5 server init message from an input buffer
/*!
    @param input input buffer
    @return authentication method accepted or except in case of failure 
*/
vmp_byte socks5_serverinit_get(vmp::Buf *input);

//! Create a socks5 request message in the output buffer
/*!
    @param out output buffer
    @param cd command code
    @param address request address
    @return void or except in case of failure 
*/
void socks5_request(vmp::Buf *out,vmp_byte cd,net::Address *address);

//! Reads a socks5 request message from an input buffer
/*!
    @param input input buffer
    @param address output address(the address output is setting onlyhost if domain name is received(see net::Address))
    @return command code or except in case of failure
*/
vmp_byte socks5_request_get(vmp::Buf *input,net::Address *address);

//! Create a socks5 reply message in the output buffer
/*!
    @param out output buffer
    @param cd command result
    @param address reply address
    @return void or except in case of failure 
*/
void socks5_reply(vmp::Buf *out,vmp_byte cd,net::Address *address);

//! Reads a socks5 reply message from an input buffer
/*!
    @param input input buffer
    @param address output address(the address output is setting onlyhost if domain name is received(see net::Address))
    @return result code or except in case of failure
*/
vmp_byte socks5_reply_get(vmp::Buf *input,net::Address *address);

//! Create a socks5 udp datagram message(for upd associate connection) in the output buffer
/*!
    @param out output buffer
    @param frag current fragment number
    @param address destination address
    @param data variable data
    @return void or except in case of failure 
*/
void socks5_udp(vmp::Buf *out,vmp_byte frag,net::Address *address,vmp::Buf *data);

//! Read a socks5 udp datagram message(for upd associate connection) in the output buffer
/*!
    @param input input buffer
    @param frag current fragment number
    @param address destination address
    @param data variable data
    @return void or except in case of failure 
*/
void socks5_udp_get(vmp::Buf *input,vmp_byte *frag,net::Address *address,vmp::Buf *data);

//! Create a socks5 username/password authentication message in the output buffer
/*!
    @param out output buffer
    @param user user 
    @param password password
    @return void or except in case of failure   
*/
void socks5_userauth(vmp::Buf *out,vmp::str user,vmp::str password);

//! Create a socks5 username/password authentication message in the output buffer
/*!
    @param input input buffer
    @param user user 
    @param password password
    @return void or except in case of failure   
*/
void socks5_userauth_get(vmp::Buf *input,vmp::str *user,vmp::str *password);

//! Create a socks5 username/password authentication message reply in the output buffer
/*!
    @param out output buffer
    @param status status field(0x00 indicates success)   
*/
void socks5_userauth_reply(vmp::Buf *out,vmp_byte status);

//! Read a socks5 username/password authentication message reply in the output buffer
/*!
    @param input input buffer
    @return status field(0x00 indicates success) or except in case of failure   
*/
vmp_byte socks5_userauth_reply_get(vmp::Buf *input);

//!Contains information for a proxy connection(see net::Proxyinfo)
class Proxy
{
    private:
        vmp::str type_;/*!<Proxy type*/
        net::Address address_;/*!<Proxy address*/
        vmp::str user_;/*!<user for proxy with authentication*/
        vmp::str password_;/*!<password for proxy with authentication*/
    public:
        //! A constructor
        /*!
            @param proxy type
            @param proxy address
        */
        Proxy(vmp::str type,net::Address *address);
        
        //! A destructor
        ~Proxy();
        
        //! Setting user and password for proxy withauthentivcation
        /*!
            @param user user authentication
            @param password apssword authentication
        */
        void set_userauth(vmp::str user,vmp::str password);
        
        //!Returns Proxy type
        /*!
            @sa type_
            @return proxy type
        */
        vmp::str type();
        
        //!Returns Proxy address
        /*!
            @sa address_
            @return proxy address
        */
        net::Address *address();
        
        //!Returns Proxy user for proxy with authentication
        /*!
            @sa user_
            @return proxy user
        */
        vmp::str user();
        
        //!Returns Proxy password for proxy with authentication
        /*!
            @sa password_
            @return proxy password
        */
        vmp::str password();
};

//!Used to store information for proxy connections
class ProxyChain:public event::UI
{
    private:
        //! Storage event
        vmp::utils::Storage<net::EventConnection> cref_;
        
        //! Internal usage
        vmp_index replace(net::Proxy *proxy,vmp_index i);
        
        //! Internal usage
        vmp_index insert(net::Proxy *proxy,vmp_index i);
        
        //! Internal usage
        void socks5_request(net::EventConnection *evt);
        
        vmp::time::Time timeout_;/*!<Timeout proxy connection(default 5.0)*/
    public:
        //! A Constructor
        /*!
            @param logger logger object utility(if == 0 not logger active)
            @param manager event manager
        */
        ProxyChain(event::Manager *manager,vmp::utils::Logger *logger=0);
        
        //! A Destructor
        ~ProxyChain();
        
        //! Set Proxy Connection timeout,if timeout == 0.0 setting default timeout
        /*!
            @param timeout proxy timeout
            @ref timeout_
        */
        void set_timeout(vmp::time::Time timeout);
        
        //! Get Proxy Connection timeout
        /*!
            @ref timeout_
            @return proxy timeout
        */
        vmp::time::Time get_timeout();
        
        //!virtual function
        void close_event(event::Cell *cell);
        
        //!vuirtual function
        void free_ref(event::Cell *cell);
        
        vmp::utils::Logger *logger_;/*!<logger object utility*/
        vmp::vector<net::Proxy *> chain_;/*!<proxy chain*/
        
        //! Reset
        void reset();
        
        //! Returns the size of the proxy chain
        /*!
            @return the size of the proxy chain
        */
        vmp_size size();
        
        //! Inserts a socks4 connection at the end of the proxy chain
        /*!
            @param address proxy address
            @return the proxy location in the chain or except in case of failure
        */
        vmp_index push_socks4(net::Address *address);
        
        //! Inserts a socks4 connection at the i position of the proxy chain,
        //!proxies already inserted from the position i are shifted 
        //!to the right,if i is greater than the size of the array it is inserted 
        //!at the end of the proxy chain
        /*!
            @param address proxy address
            @param i index position
            @return the proxy location in the chain or except in case of failure
        */
        vmp_index insert_socks4(net::Address *address,vmp_index i);
        
        //!Replace a socks4 connection at the i position of the proxy chain,
        //!if i is greater than the size of the array it is inserted 
        //!at the end of the proxy chain
        /*!
            @param address proxy address
            @param i index position
            @return the proxy location in the chain or except in case of failure
        */
        vmp_index replace_socks4(net::Address *address,vmp_index i);
        
        //! Inserts a socks5 connection at the end of the proxy chain
        /*!
            @param address proxy address
            @return the proxy location in the chain or except in case of failure
        */
        vmp_index push_socks5(net::Address *address);
        
        //! Inserts a socks5 connection at the i position of the proxy chain,
        //!proxies already inserted from the position i are shifted 
        //!to the right,if i is greater than the size of the array it is inserted 
        //!at the end of the proxy chain
        /*!
            @param address proxy address
            @param i index position
            @return the proxy location in the chain or except in case of failure
        */
        vmp_index insert_socks5(net::Address *address,vmp_index i);
        
        //!Replace a socks5 connection at the i position of the proxy chain,
        //!if i is greater than the size of the array it is inserted 
        //!at the end of the proxy chain
        /*!
            @param address proxy address
            @param i index position
            @return the proxy location in the chain or except in case of failure
        */
        vmp_index replace_socks5(net::Address *address,vmp_index i);
        
        //! Inserts a socks5 connection with user/password authentication at the end of the proxy chain
        /*!
            @param address proxy address
            @param user user authentication
            @param password password authentication
            @return the proxy location in the chain or except in case of failure
        */
        vmp_index push_socks5_userauth(net::Address *address,vmp::str user,vmp::str password);
        
        //! Inserts a socks4 connection with user/password authentication
        //! at the i position of the proxy chain,
        //!proxies already inserted from the position i are shifted 
        //!to the right,if i is greater than the size of the array it is inserted 
        //!at the end of the proxy chain
        /*!
            @param address proxy address
            @param i index position
            @param user user authentication
            @param password password authentication
            @return the proxy location in the chain or except in case of failure
        */
        vmp_index insert_socks5_userauth(net::Address *address,vmp_index i,vmp::str user,vmp::str password);
        
        //!Replace a socks4 connection with user/password authentication,
        //!at the i position of the proxy chain,
        //!if i is greater than the size of the array it is inserted 
        //!at the end of the proxy chain
        /*!
            @param address proxy address
            @param i index position
            @param user user authentication
            @param password password authentication
            @return the proxy location in the chain or except in case of failure
        */
        vmp_index replace_socks5_userauth(net::Address *address,vmp_index i,vmp::str user,vmp::str password);
        
        //! Returns the proxy info of the proxy of the position i
        /*!
            @param i index position
            @return proxy info or 0 in case of failure
        */
        net::Proxy *get(vmp_index i);
        
        //! Remove the proxy of the position i is shifta all the proxy info with position to the right of i of uan position to the left
        /*!
            @param i index position
            @return new chain size
        */
        vmp_size cancel(vmp_index i);
        
        //! Internal usage
        event::Cell *tcpconnect(net::EventConnection *evt,event::UI *ui,event::EVTCB close);
        
        //! Internal usage
        event::Cell *tcpbind(net::EventConnection *evt,net::Address *ref,event::UI *ui,event::EVTCB close);
        
        //! Internal usage
        event::Cell *udpassociate(net::EventConnection *evt,event::UI *ui,event::EVTCB close);
        
        //! Internal usage
        vmp_bool next(net::EventConnection *evt,vmp::Buf *buf=0);
};

}}

#endif

