/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta net::Address::
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 20/01/2024
 */

#include "net.h"

namespace vampiria { namespace net {

vmp_bool is_ipv4_raw(vmp::str ip)
{
    vmp::vector<vmp::str> split=vmp::unicode::str_split(ip,".");
    vmp_size s=split.size();
    if(s == 4)
    {
        vmp_index i=0;
        for(i=0;i<s;i++)
        {
            try
            {
                vmp::unicode::str_todigit_range(split[i],0,255);
            }
            catch(vmp::exception &x)
            {
                break;
            }        
 
        }
        if(i == 4)
            return true;
    }
    return false;
}

vmp::vector<vmp::str> ipv4_expand(vmp::str subnet,vmp_uint mask)
{
    vmp::vector<vmp::str> ret;
    if(!net::is_ipv4_raw(subnet))
        vmp::except("net::ipv4_expand(subnet=%s) input bad value",subnet.c_str());
    if(mask > 32)
        vmp::except("net::ipv4_expand(mask=%u) input bad value",mask);
    vmp_byte snet[4];
    vmp_byte shost[4];
    vmp_uint tmp,tmp2,bmask;
    net::Address addr;
    addr.set(subnet);
    vmp::str sret;
    struct sockaddr_in *saddr=(struct sockaddr_in *) addr.addr();
    tmp=(saddr->sin_addr.s_addr);
    snet[0]=(vmp_byte)(tmp & 0xFF);
    snet[1]=(vmp_byte)((tmp >> 8) & 0xFF);
    snet[2]=(vmp_byte)((tmp >> 16) & 0xFF);
    snet[3]=(vmp_byte)((tmp >> 24) & 0xFF);
    tmp=mask;
    for(vmp_index i=0;i<4;i++)
    {
        if(tmp == 0)
            snet[i]=0x00;
        else if(tmp < 8)
        {
            tmp2=7;
            bmask=0;
            while(tmp > 0)
            {
                bmask+= (vmp_uint) vmp::math::pow_wrap(2,tmp2);
                tmp--;
                tmp2--;
            }
            snet[i]=snet[i] & bmask;
        }
        else
            tmp-=8;
    }
    for(vmp_index i=0;i<= (vmp_index)(vmp::math::pow_wrap(2,32-mask)-1);i++)
    {
        shost[0]=(vmp_byte)(((i >> 24) & 0xFF) | snet[0]);
        shost[1]=(vmp_byte)(((i >> 16) & 0xFF) | snet[1]);
        shost[2]=(vmp_byte)(((i >> 8) & 0xFF)  | snet[2]);
        shost[3]=(vmp_byte)((i & 0xFF) | snet[3]);
        vmp::unicode::str_write(&sret,"%u.%u.%u.%u",shost[0],shost[1],shost[2],shost[3]);
        ret.push_back(sret);
    }
    return ret;
}

vmp_bool is_ipv6_raw_frag(vmp::str ip,vmp_size *size)
{
    vmp::vector<vmp::str> split=vmp::unicode::str_split(ip,":");	
    vmp_size n = split.size(),l;
    (*size)=0;
    if(n > 8)
        return false;
    else
    {
       for(vmp_index i=0;i<n;i++)
       {
           l=split[i].size();
           if(l>0 && l<=4 && vmp::unicode::str_istype(split[i],"xdigit"))
	       (*size) +=1;	
           else
               return false;
       }
    }
    return true;
}

vmp_bool is_ipv6_raw(vmp::str ip)
{
    vmp::vector<vmp::str> split=vmp::unicode::str_split(ip,"::");
    vmp_size n = split.size(),size1,size2;       
    if(n == 0 && ip == "::")
        return true;
    else if(n==1 && ip.size() > 2)
    {
        if((ip.substr(0,2) == "::") && (ip.substr(ip.size()-2,2)) != "::" && is_ipv6_raw_frag(split[0],&size1))
            return true;
        else if((ip.substr(0,2) != "::") && (ip.substr(ip.size()-2,2) == "::") && is_ipv6_raw_frag(split[0],&size1))
            return true;
        else if((ip.substr(0,2) != "::") && (ip.substr(ip.size()-2,2) != "::") && is_ipv6_raw_frag(split[0],&size1))               
            return true;            
    }        
    else if(n==2 && is_ipv6_raw_frag(split[0],&size1) && is_ipv6_raw_frag(split[1],&size2) && ((size1+size2) < 8))
        return true;
    return false;    
}

vmp_bool is_ip_raw(vmp::str ip)
{
    if(is_ipv4_raw(ip) || is_ipv6_raw(ip))
        return true;
    return false; 
}

Address::Address()
{
    vmp_assert(_POSIX_C_SOURCE);
    addrinfo_=0;
    host_="";
    ip_="";
    service_="";
    
    dnstype_="system";
    dnsip_="";
    dnsport_="";
    dnstout_=0.0;
}

Address::Address(vmp::str dnstype,vmp::str dnsip,vmp::str dnsport,vmp::time::Time dnstout)
{
    vmp_assert(_POSIX_C_SOURCE);
    addrinfo_=0;
    host_="";
    ip_="";
    service_="";
    
    if((dnstype != "system") && (dnstype != "tcp4") && (dnstype != "tcp6") && (dnstype != "tcp46") && (dnstype != "tcp64"))
        vmp::except("net::Address(dnstype='%s') invalid dns type value",dnstype.c_str());
    dnstype_=dnstype;
    if((!net::is_ipv4_raw(dnsip)) && (!net::is_ipv6_raw(dnsip)))
        vmp::except("net::Address(dnsip='%s') invalid dns ip value",dnsip.c_str());
    dnsip_=dnsip;
    try
    {
        vmp::unicode::str_todigit_range(dnsport,1,65535);
    }
    catch(vmp::exception &x)
    {
        vmp::except("net::Address(dnsport='%s') invalid dns port value",dnsport.c_str());
    }
    dnsport_=dnsport;
    if(dnstout <= 0.0)
        vmp::except("net::Address(dnstout='%f') invalid dns timeout value",dnstout);
    dnstout_=dnstout;
}
     
Address::~Address()
{
    reset();
    dnstype_="";
    dnsip_="";
    dnsport_="";
}

void Address::reset()
{
    if(addrinfo_ != 0)
        freeaddrinfo(addrinfo_);
    addrinfo_=0;
    host_="";
    ip_="";
    service_="";
    onlyhost_=false;
}

void Address::init()
{
    vmp_int ret;
    struct addrinfo hints;

    vmp::bzero_wrap(&hints, sizeof(struct addrinfo));
    hints.ai_family = PF_UNSPEC;
    hints.ai_socktype = 0;
    hints.ai_flags = 0;
    hints.ai_protocol = 0;
    hints.ai_canonname = 0;
    hints.ai_addr = 0;
    hints.ai_next = 0;
    if((ret=getaddrinfo(host_.c_str(),service_.c_str(),&hints,&addrinfo_)) != 0)
       vmp::except("net::Address::init() host='%s' service='%s' error '%s'",host_.c_str(),service_.c_str(),gai_strerror(ret));
}

void Address::init_s()
{
    vmp_char hbuf[NI_MAXHOST],sbuf[NI_MAXSERV];
    vmp_int ret;
    vmp::bzero_wrap(hbuf,NI_MAXHOST);
    vmp::bzero_wrap(sbuf,NI_MAXSERV);
    if(addrinfo_ == 0)
        vmp::except_s("net::Address::init() error 'null addrinfo address'");

    switch(iptype())
    {
        case AF_INET: 
            if((ret=getnameinfo(addrinfo_->ai_addr,sizeof(struct sockaddr_in),hbuf,NI_MAXHOST,sbuf,NI_MAXSERV,NI_NUMERICHOST|NI_NUMERICSERV)) == -1)
                vmp::except("net::Address::init_s() host='%s' service='%s' error '%s'",host_.c_str(),service_.c_str(),gai_strerror(ret));
            break;
        case AF_INET6:
            if ((ret=getnameinfo(addrinfo_->ai_addr,sizeof(struct sockaddr_in6),hbuf,NI_MAXHOST,sbuf,NI_MAXSERV,NI_NUMERICHOST|NI_NUMERICSERV)) == -1)
                vmp::except("net::Address::init_s() host='%s' service='%s' error '%s'",host_.c_str(),service_.c_str(),gai_strerror(ret));
            break;
        default:
            vmp::except("net::Address::init_s() host='%s' service='%s' error 'invalid address family'",host_.c_str(),service_.c_str());
            return;  
    }
    ip_=hbuf;
    service_=sbuf;        
}

void Address::set(vmp::str host,vmp::str service,vmp_bool onlyhost)
{
    reset();
    try
    {
        vmp::unicode::str_todigit_range(service,0,65535);
    }
    catch(vmp::exception &x)
    {
        vmp::except("net::Address::set(service='%s') invalid service input",service.c_str());
    }
    service_=service;
    if(onlyhost)
    {
        host_=host;
        onlyhost_=onlyhost;
    }
    else
    {
        if(net::is_ip_raw(host) || (dnstype_ == "system"))
        {
            host_=host;
            init();
        }
        else 
        {
            vmp::vector<vmp::str> resolve;
            if(dnstype_ == "tcp4")
                resolve=net::tcp_resolve4(host,dnsip_,dnsport_,dnstout_);
            else if(dnstype_ == "tcp6")
                resolve=net::tcp_resolve6(host,dnsip_,dnsport_,dnstout_);
            else if(dnstype_ == "tcp46")
            {
                resolve=net::tcp_resolve4(host,dnsip_,dnsport_,dnstout_);
                if(resolve.size() == 0)
                    resolve=net::tcp_resolve6(host,dnsip_,dnsport_,dnstout_);
            }
            else if(dnstype_ == "tcp64")
            {
                resolve=net::tcp_resolve6(host,dnsip_,dnsport_,dnstout_);
                if(resolve.size() == 0)
                resolve=net::tcp_resolve4(host,dnsip_,dnsport_,dnstout_);
            }
            if(resolve.size() == 0)
                vmp::except("net::Address::set(host='%s') not resolve host",host.c_str()); 
            host_=resolve[0];
            init();
            host_=host;
        }
        init_s();
    }
}

void Address::set(net::Sockaddr *addr,vmp_size addrsize)
{
    reset();
    vmp_char hbuf[NI_MAXHOST],sbuf[NI_MAXSERV];
    vmp::bzero_wrap(hbuf,NI_MAXHOST);
    vmp::bzero_wrap(sbuf,NI_MAXSERV);
    vmp_int ret;
    if((ret=getnameinfo(addr,addrsize,hbuf,NI_MAXHOST,sbuf,NI_MAXSERV,NI_NUMERICHOST|NI_NUMERICSERV)) == -1)   
        vmp::except("net::Address::set() error '%s'",gai_strerror(ret));
    host_=hbuf;
    ip_=hbuf;
    service_=sbuf;
    init();
}

void Address::set_ip_raw(vmp::str ip,vmp::str service)
{
    reset();
    if(is_ipv4_raw(ip) || is_ipv6_raw(ip))
    {
        host_=ip;
        ip_=ip;
        try
        {
            vmp::unicode::str_todigit_range(service,0,65535);
        }
        catch(vmp::exception &x)
        {
            vmp::except("net::Address::set_ip_raw() invalid service input '%s'",service.c_str());
        }
        service_=service;
    }
    else
        vmp::except("net::Address::set_ip_raw() invalid ip input address '%s'",ip.c_str());
    init();
}

void Address::copy(Address *address)
{
    struct addrinfo *src,*dst,*prev;
    if(address != 0)
    {
        if(addrinfo_ != 0)
        {    
            freeaddrinfo(addrinfo_);
            addrinfo_=0;
        }
        if(address->addrinfo_ != 0)
        {
            src=address->addrinfo_;
            while(src != 0)
            {
                dst=(struct addrinfo *) vmp::malloc_wrap(sizeof(struct addrinfo));
                vmp::bzero_wrap((void *)dst,sizeof(struct addrinfo));
                dst->ai_next=0;
                dst->ai_flags = src->ai_flags; 
                dst->ai_family = src->ai_family;
                dst->ai_socktype = src->ai_socktype;
                dst->ai_protocol = src->ai_protocol;
                dst->ai_canonname = src->ai_canonname;
                dst->ai_addrlen = src->ai_addrlen;
                dst->ai_addr = (struct sockaddr *) vmp::malloc_wrap(src->ai_addrlen);
                vmp::memcpy_wrap((void *)dst->ai_addr,(void *)src->ai_addr,src->ai_addrlen);
                if(addrinfo_ == 0)
                    addrinfo_=dst;
                else
                    prev->ai_next=dst;
                prev=dst;
                src=src->ai_next;
            }
        }   
        host_=address->host_;
        ip_=address->ip_;
        service_=address->service_;
        dnstype_=address->dnstype_;
        dnsip_=address->dnsip_;
        dnsport_=address->dnsport_;
        dnstout_=address->dnstout_;
        onlyhost_=address->onlyhost_;        
    }
    else
        reset();
}

net::Sockaddr *Address::addr()
{
    if(addrinfo_ == 0)
        return 0;
    return addrinfo_->ai_addr;
}

vmp_size Address::addrsize()
{
    if(addrinfo_ == 0)
        return 0;
    return addrinfo_->ai_addrlen;
}

vmp::str Address::host()
{
    return host_;
}

vmp::str Address::ip()
{
    if(onlyhost_)
        vmp::except("net::Address::ip() host=%s setting in onlyhost mode",host_.c_str());
    return ip_;
}
        
vmp::str Address::service()
{
    return service_;
}

vmp::vector<vmp_byte> Address::bytes_ip()
{
   vmp::vector<vmp_byte> ret; 
   if(is_ipv4())
   {
       vmp::vector<vmp::str> split=vmp::unicode::str_split(ip_,".");
       for(vmp_index i=0;i<split.size();i++)
           ret.push_back((vmp_byte)vmp::unicode::str_todigit_range(split[i],0,255));
       
   }
   else if(is_ipv6())
   {
     struct sockaddr_in6 *inaddr=(struct sockaddr_in6 *)addr();
     for(vmp_index i=0;i<16;i++)
        ret.push_back((vmp_byte)(inaddr->sin6_addr.s6_addr[i]));
   }
   else
      vmp::except_s("net::Address::bytes_ip() error 'unspec ip'");
   return ret;
}

vmp_int Address::iptype()
{
    if(addrinfo_ == 0)
        return PF_UNSPEC;
    return addrinfo_->ai_family;
}

vmp_bool Address::is_ipv4()
{
    if(iptype() == AF_INET)
        return true;
    return false;
}
        
vmp_bool Address::is_ipv6()
{
    if(iptype() == AF_INET6)
        return true;
    return false;
}

vmp_bool Address::is_onlyhost()
{
    return onlyhost_;    
}

void ip_to_buf(net::Address *address,vmp::Buf *buf)
{
    if((address == 0) || address->iptype() == PF_UNSPEC)
        vmp::except_s("net::ip_to_buf() unspec input address");
    if(buf == 0)
        vmp::except_s("net::ip_to_buf() null pointer input buf");
    vmp::vector<vmp_byte> ip=address->bytes_ip();
    buf->write_vector(ip);
}

void ipv4_from_buf(net::Address *address,vmp::Buf *buf)
{
    if(address == 0)
        vmp::except_s("net::ipv4_from_buf() null pointer input address");
    if(buf == 0)
        vmp::except_s("net::ipv4_from_buf() null pointer input buf");
    vmp::vector<vmp_byte> ip;
    try
    {
        ip=buf->read_vector(4);
    }
    catch(vmp::exception &x)
    {
        vmp::except("%s read ipv4 address",vmp::buf_eob_strerror().c_str());
    }
    vmp::str ipstr;
    vmp::unicode::str_write(&ipstr,"%d.%d.%d.%d",(vmp_uint)ip[0],(vmp_uint)ip[1],(vmp_uint)ip[2],(vmp_uint)ip[3]);
    address->set_ip_raw(ipstr);
}

void ipv6_from_buf(net::Address *address,vmp::Buf *buf)
{
    if(address == 0)
        vmp::except_s("net::ipv4_from_buf() null pointer input address");
    if(buf == 0)
        vmp::except_s("net::ipv4_from_buf() null pointer input buf");
    vmp::vector<vmp_byte> ip;
    try
    {
        ip=buf->read_vector(16);
    }
    catch(vmp::exception &x)
    {
        vmp::except("%s read ipv6 address",vmp::buf_eob_strerror().c_str());
    }
    vmp::str ipstr;
    vmp::unicode::str_write(&ipstr,"%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x",ip[0],ip[1],ip[2],ip[3],ip[4],ip[5],ip[6],ip[7],ip[8],ip[9],ip[10],ip[11],ip[12],ip[13],ip[14],ip[15]);
    address->set_ip_raw(ipstr);
}

MacAddress::MacAddress()
{
    saddr_="00:00:00:00:00:00";
    for(vmp_index i=0;i<6;i++)
        baddr_.push_back(0x00);
}
        
MacAddress::~MacAddress()
{
    saddr_.clear();
    baddr_.clear();
}

void MacAddress::reset()
{
    saddr_="00:00:00:00:00:00";
    for(vmp_index i=0;i<6;i++)
        baddr_[i]=0x00;
}

void MacAddress::set(vmp::str addr)
{
    vmp::vector<vmp::str> split=vmp::unicode::str_split(addr,":");
    vmp_bool err=false;
    vmp::vector<vmp_byte> tmp;
    if(split.size() == 6)
    {
        for(vmp_index i=0;i<6;i++)
        {
            try
            {
                if(split[i].size() == 2)
                {    
                    tmp=vmp::unicode::xstr_tobytes(split[i]);
                    baddr_[i]=tmp[0];
                }
                else
                {
                    err=true;
                    break;
                }
            }
            catch(vmp::exception &x) 
            {
                err=true;
                break;
            }
        }
        
    }
    else
        err=true;
    if(err)
    {
       reset();
       vmp::except("net::MacAddress::set() invalid mac address input '%s'",addr.c_str());
    }
    saddr_=addr;
}
        
void MacAddress::set(vmp::vector<vmp_byte> baddr)
{
    if(baddr.size() != 6)
        vmp::except("net::MacAddress::set() invalid vector address len '%u != 6'",baddr.size());
    saddr_="";
    for(vmp_index i=0;i<baddr.size();i++)
    {    
       if(i < baddr.size()-1)
           vmp::unicode::str_cwrite(&saddr_,"%02X:",baddr[i]);
       else
           vmp::unicode::str_cwrite(&saddr_,"%02X",baddr[i]);
    }
    baddr_=baddr;
}

vmp::str MacAddress::str_addr()
{
    return saddr_;
}
        
vmp::vector<vmp_byte> MacAddress::byte_addr()
{
    return baddr_;
}

vmp_bool is_macaddress_raw(vmp::str mac)
{
    MacAddress m;
    try
    {
        m.set(mac);
    }
    catch(vmp::exception &x)
    {
        return false;
    }
    return true;
}

void macaddress_to_buf(net::MacAddress *mac,vmp::Buf *buf)
{
    if(mac == 0)
        vmp::except_s("net::macaddress_to_buf() null pointer input address");
    if(buf == 0)
        vmp::except_s("net::macaddress_to_buf() null pointer input buf");
    vmp::vector<vmp_byte> maddr=mac->byte_addr();
    buf->write_vector(maddr);
}

void macaddress_from_buf(net::MacAddress *mac,vmp::Buf *buf)
{
    if(mac == 0)
        vmp::except_s("net::macaddress_from_buf() null pointer input address");
    if(buf == 0)
        vmp::except_s("net::macaddress_from_buf() null pointer input buf");
    vmp::vector<vmp_byte> maddr;
    try
    {
        maddr=buf->read_vector(6);
    }
    catch(vmp::exception &x)
    {
        vmp::except("%s read mac ethernet address",vmp::buf_eob_strerror().c_str());
    }
    mac->set(maddr);
}

}}

