/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 20/03/2025
*/

#ifndef VAMPIRIA_JSON_JAPI_USER_H

#define VAMPIRIA_JSON_JAPI_USER_H 1

namespace vampiria { namespace json {

//! Create a json data json::japi("login") used for authentication token
/*!
    <pre>
    { 
       jtype: json::japi("login"),
       jbody: 
       {
           login:'login',
           password:'password',
           md5_password:bool
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @param user user name(read data json::jdata.get_text("user"))
    @param password password(read data json::jdata.get_text("password"))
    @param md5_password if true the input password parameter is encrypted with the md5 function,default false (read data json::jdata.get_bool("md5_password"))
    @return void or except in case of failure
*/
void japi_login(json::JsonObj *obj,vmp::str user,vmp::str password,vmp_bool md5_password=false);

//! Create an authentication token,a jdata json::japi("login") encoded in base64
/*!
    @param user user name
    @param password password
    @param md5_password  if true the input password parameter is encrypted with the md5 function,default false
    @return string token or except in case of failure
*/
vmp::str japi_login_b64_e(vmp::str user,vmp::str password,vmp_bool md5_password=false);

//! Decode a authentication token and Read obj a jdata json::japi("login")
/*!
    @param json parser where the json data is inserted
    @param etoken input b64 token
    @return void or except in case of failure
*/
void japi_login_b64_d(json::Json *json,vmp::str etoken);

//! Create a json data json::japi("passwd") used for change user password
/*!
    <pre>
    { 
       jtype: json::japi("passwd"),
       jbody: 
       {
           oldpassword:'oldpassword',
           password:'password',
           md5_password:bool
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @param oldpassword old password(read data json::jdata.get_text("oldpassword"))
    @param password new password(read data json::jdata.get_text("password"))
    @param md5_password if true the input password and old password parameter are encrypted with the md5 function,default false (read data json::jdata.get_bool("md5_password"))
    @return void or except in case of failure
*/
void japi_passwd(json::JsonObj *obj,vmp::str oldpassword,vmp::str password,vmp_bool md5_password=false);

//! Create a json data json::japi("userlist") used for request list users
/*!
    <pre>
    { 
       jtype: json::japi("userlist"),
       jbody: 
       {
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @return void or except in case of failure
*/
void japi_userlist(json::JsonObj *obj);

//! Create a json data json::japi("userinfo") used for send userinfo data
/*!
    <pre>
    { 
       jtype: json::japi("userinfo"),
       jbody: 
       {
           user:'user',
           permits:permits,
           only_localhost:only_localhost
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @param user user (read data json::jdata.get_text("user"))
    @param permits permits user (read data json::jdata.get_integer_range("permits",0,vmp::INTMAX))
    @param only_localhost is the user logged in only on localhost? (read data json::jdata.get_bool("only_localhost"))
    @return void or except in case of failure
*/
void japi_userinfo(json::JsonObj *obj,vmp::str user,vmp_uint permits,vmp_bool only_localhost);

//! Create a json data json::japi("useradd") used for add a new user
/*!
    <pre>
    { 
       jtype: json::japi("useradd"),
       jbody: 
       {
           user:'user',
           password:'password',
           permits:permits,
           only_localhost:only_localhost
           
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @param user user (read data json::jdata.get_text("user"))
    @param password password (read data json::jdata.get_text("password"))
    @param permits permits user (read data json::jdata.get_integer_range("permits",0,vmp::INTMAX))
    @param only_localhost is the user logged in only on localhost? (read data json::jdata.get_bool("only_localhost"))
    @param md5_password if true the input password and old password parameter are encrypted with the md5 function,default false (read data json::jdata.get_bool("md5_password"))
    @return void or except in case of failure
*/
void japi_useradd(json::JsonObj *obj,vmp::str user,vmp::str password,vmp_uint permits,vmp_bool only_localhost,vmp_bool md5_password=false);

//! Create a json data json::japi("usermod") used for Change user values
/*!
    <pre>
    { 
       jtype: json::japi("usermod"),
       jbody: 
       {
           user:'user',
           permits:permits,
           only_localhost:only_localhost
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @param user user (read data json::jdata.get_text("user"))
    @param permits permits user (read data json::jdata.get_integer_range("permits",0,vmp::INTMAX))
    @param only_localhost is the user logged in only on localhost? (read data json::jdata.get_bool("only_localhost"))
    @return void or except in case of failure
*/
void japi_usermod(json::JsonObj *obj,vmp::str user,vmp_uint permits,vmp_bool only_localhost);

//! Create a json data json::japi("userpasswd") used for Change user password
/*!
    <pre>
    { 
       jtype: json::japi("userpasswd"),
       jbody: 
       {
           user:'user',
           password:'password',
           md5_password:md5_password
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @param user user (read data json::jdata.get_text("user"))
    @param password new password(read data json::jdata.get_text("password"))
    @param md5_password if true the input password and old password parameter are encrypted with the md5 function,default false (read data json::jdata.get_bool("md5_password"))
    @return void or except in case of failure
*/
void japi_userpasswd(json::JsonObj *obj,vmp::str user,vmp::str password,vmp_bool md5_password=false);

//! Create a json data json::japi("userdel") used for delete user
/*!
    <pre>
    { 
       jtype: json::japi("userdel"),
       jbody: 
       {
           user:'user'
       }
    }
    </pre>
    @param obj object where the json data is inserted
    @param user user (read data json::jdata.get_text("user"))
    @return void or except in case of failure
*/
void japi_userdel(json::JsonObj *obj,vmp::str user);

}}

#endif

