/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 05/12/2024
*/

#include "jrp.h"

namespace vampiria { namespace jrp {

void pointer_check(void *pointer,vmp::str func,vmp::str pname)
{
    if(pointer == 0)
        vmp::except("jrp::%s() null input pointer '%s'",func.c_str(),pname.c_str());
}

JrpReqData::JrpReqData()
{
    input_="";
    permits_s_="";
}

JrpReqData::~JrpReqData()
{
    input_="";
    push_.clear();
    response_.clear();
    permits_s_="";
    permits_.clear();
}

vmp::str JrpReqData::input()
{
    return input_;
}

vmp::vector<vmp::str> JrpReqData::push()
{
    return push_;
}

vmp::vector<vmp::str> JrpReqData::response()
{
    return response_;
}

void JrpReqData::set_permits(vmp::str permits_s)
{
permits_=vmp::unicode::str_toindex_list(permits_s,jrp::min_permits,jrp::max_permits);
    permits_s_=permits_s;
}

vmp::str JrpReqData::permits()
{
    return permits_s_;
}

JrpReqSession::JrpReqSession()
{
    reset();
}

JrpReqSession::~JrpReqSession()
{
    reset();
}

void JrpReqSession::reset()
{
    vmp::table_delete_alldata<vmp::str,jrp::JrpReqData *>(&reqdata_);
    reqdata_.clear();
    bdata_.clear();
    permits_=0;
    nodetype_="";
}

vmp_bool JrpReqSession::match_reqdata(vmp::str input,vmp_uint permits)
{
    jrp::JrpReqData *data;
    if(reqdata_.search(input,&data))
        return vmp::invector<vmp_index>(permits,data->permits_);
    return false;
}

vmp_bool JrpReqSession::match_bdata(vmp::str input)
{
    void *p;
    return bdata_.search(input,&p);
}

void JrpReqSession::json_reqdata(json::JsonObj *out,vmp_uint permits)
{
    vmp::except_check_pointer((void *)out,"jrp::JrpReqSession::json_reqdata(out=Null)");
    json::JsonObj req,array;
    vmp::vector<jrp::JrpReqData *> data=reqdata_.all_data();
    for(vmp_index i=0;i<data.size();i++)
    {
        if((permits == 0) || match_reqdata(data[i]->input_,permits))
        {
            out->add_object_obj(data[i]->input_,&req);
            req.add_object_array_strings("push",data[i]->push_,&array);
            req.add_object_array_strings("response",data[i]->response_,&array);
            req.add_object_str("permits",data[i]->permits_s_,&array);
        }    
    }
}

void JrpReqSession::session(vmp::Buf *buf,vmp_uint permits)
{
    vmp::except_check_pointer((void *)buf,"jrp::JrpReqSession::session(buf=Null)");
    if(permits == 0)
        vmp::except_s("jrp::JrpReqSession::session(permits=0)");
    json::Json json;
    json::JsonObj *root,obj;
    root=json.json_new();
    root->add_object_str("msgtype","session",&obj);
    root->add_object_str("nodetype",nodetype_,&obj);
    root->add_object_obj("reqdata",&obj);
    json_reqdata(&obj,permits);
    root->add_object_array_strings("broadcast",bdata_.all_keys(),&obj);
    root->add_object_number("permits",permits,&obj);
    buf->reset();
    buf->write_str(json.json_str());
    buf->index();
}

void JrpReqSession::session_get(json::JsonObj *root)
{
    reset();
    vmp::except_check_pointer((void *) root,"jrp::JrpReqSession::session_get(root=Null)");
    vmp::vector<vmp::str> keys=root->keys(),bdata;
    json::JsonObj obj,input;
    vmp_int permits=(vmp_int) root->get_object_number("permits");
    if(permits <= 0)
        vmp::except("jrp::JrpReqSession::session_get(json.permits=%d) bad value",permits);
    permits_=permits;
    nodetype_=root->get_object_str("nodetype");
    root->get_object("reqdata",&obj);
    vmp::vector<vmp::str> kinput=obj.keys();
    jrp::JrpReqData *data;
    for(vmp_index i=0;i<kinput.size();i++)
    {
        data=new jrp::JrpReqData();
        try
        {
             obj.get_object(kinput[i],&input);
             data->input_=kinput[i];
             data->push_=input.get_object_array_strings("push");
             data->response_=input.get_object_array_strings("response");
             data->set_permits(input.get_object_str("permits"));
             reqdata_.insert(data->input_,data);
        }
        catch(vmp::exception &x)
        {
            delete data;
            vmp::except_s("jrp::JrpReqSession::session_get(json.reqdata='bad  format')");
        }
    }
    bdata=root->get_object_array_strings("broadcast");
    for(vmp_index i=0;i<bdata.size();i++)
        bdata_.insert(bdata[i],0);
}

void JrpReqSession::add_reqdata(vmp::str input,vmp::vector<vmp::str> push,vmp::vector<vmp::str> response,vmp::str permits)
{
    jrp::JrpReqData *data;
    vmp_bool newdata=false;
    if(!reqdata_.search(input,&data))
    {
        data=new jrp::JrpReqData();
        newdata=true;
    }
    data->input_=input;
    data->push_=push;
    data->response_=response;
    try
    {
        data->set_permits(permits);
        if(newdata)
            reqdata_.insert(input,data);
    }
    catch(vmp::exception &x)
    {
        if(newdata)
            delete data;   
        vmp::except("jrp::JrpReqSession::add_reqdata(permits=%s) %s",permits.c_str(),x.what()); 
    }
}

void JrpReqSession::add_broadcastdata(vmp::str input)
{
    try
    {
        bdata_.insert(input,0);
    }
    catch(vmp::exception &x)
    {
    }
}

jrp::JrpReqData *JrpReqSession::search_reqdata(vmp::str input)
{
    jrp::JrpReqData *ret;
    if(reqdata_.search(input,&ret))
        return ret;
    return 0;   
}

vmp::vector<vmp::str> JrpReqSession::all_input()
{
    return reqdata_.all_keys();
}

vmp::vector<vmp::str> JrpReqSession::search_push(vmp::str push)
{
    vmp::vector<vmp::str> ret;
    vmp::vector<jrp::JrpReqData *> data=reqdata_.all_data();
    for(vmp_index i=0;i<data.size();i++)
    {
        if(vmp::unicode::str_invector(push,data[i]->push_))
            ret.push_back(data[i]->input_);   
    }
    return ret;
}

vmp::vector<vmp::str> JrpReqSession::search_response(vmp::str response)
{
    vmp::vector<vmp::str> ret;
    vmp::vector<jrp::JrpReqData *> data=reqdata_.all_data();
    for(vmp_index i=0;i<data.size();i++)
    {
        if(vmp::unicode::str_invector(response,data[i]->response_))
            ret.push_back(data[i]->input_);    
    }
    return ret;
}

vmp::str JrpReqSession::nodetype()
{
    return nodetype_;
}

vmp_uint JrpReqSession::permits()
{
    return permits_;
}

MsgParse::MsgParse()
{
    reset();
}

MsgParse::~MsgParse()
{
    reset();
}

void MsgParse::reset()
{
    json_.json_new();
    msgtype_="";
    status_=0;
    msg_="";
    rid_=0;
    input_.reset();
    payload_.reset();
    outputs_.reset();
}

vmp_int parse(vmp::Buf *buf,jrp::MsgParse *msg,jrp::JrpReqSession *session)
{
    jrp::pointer_check((void *) buf,"parse","buf");
    jrp::pointer_check((void *) msg,"parse","msg");
    vmp_size psize;
    json::JsonObj *root,vect,obj;
    json::JData tmp;
    vmp_int code;
    buf->index();
    msg->reset();
    try
    {
        psize=buf->read_size(4);
        root=msg->json_.parse_from_str(buf->read_str(psize));
        msg->msgtype_=root->get_object_str("msgtype");
        vmp::vector<vmp::str> keys=root->keys();
        if((msg->msgtype_ == "session") && (keys.size() == 5))
        {    
            if(session != 0)
                session->session_get(root);
            return jrp::status_ok;
        }
        else if ((msg->msgtype_ == "request") && (keys.size() == 4))
        {
            msg->rid_=root->get_object_number("rid");
            root->get_object("input",&msg->input_);
            tmp.set(&msg->input_);
            code=root->get_object_number("payload");
            if(code > 0)
                buf->read_buf(&msg->payload_,(vmp_size)code);
            return jrp::status_ok;
        }
        else if ((msg->msgtype_ == "push") && (keys.size() == 4))
        {
            msg->rid_=root->get_object_number("rid");
            root->get_object("outputs",&msg->outputs_);
            vmp_size s=msg->outputs_.get_array_size();
            code=root->get_object_number("payload");
            if(s == 0 ||(code > 0 && s != 1))
                vmp::except_s("");
            for(vmp_index i=0;i<s;i++)
            {
                msg->outputs_.get_array_idx(i,&obj);
                tmp.set(&obj);
            }
            code=root->get_object_number("payload");
            if(code > 0)
                buf->read_buf(&msg->payload_,(vmp_size)code);
            return jrp::status_ok;
        }
        else if ((msg->msgtype_ == "response") && (keys.size() == 4))
        {
            msg->rid_=root->get_object_number("rid");
            root->get_object("outputs",&msg->outputs_);
            vmp_size s=msg->outputs_.get_array_size();
            code=root->get_object_number("payload");
            if(s == 0 ||(code > 0 && s != 1))
                vmp::except_s("");
            for(vmp_index i=0;i<s;i++)
            {
                msg->outputs_.get_array_idx(i,&obj);
                tmp.set(&obj);
            }
            code=root->get_object_number("payload");
            if(code > 0)
                buf->read_buf(&msg->payload_,(vmp_size)code);
            return jrp::status_ok;
        }
        else if ((msg->msgtype_ == "close") && (keys.size() == 4))
        {
            msg->rid_=root->get_object_number("rid");
            msg->status_=root->get_object_number("status");
            msg->msg_=root->get_object_str("msg");
            return jrp::status_ok;   

        }
        else if ((msg->msgtype_ == "kill") && (keys.size() == 2))
        {
            msg->rid_=root->get_object_number("rid");
            return jrp::status_ok; 
        }
        else if ((msg->msgtype_ == "broadcast") && (keys.size() == 3))
        {
            root->get_object("input",&msg->input_);
            tmp.set(&msg->input_);
            code=root->get_object_number("payload");
            if(code > 0)
                buf->read_buf(&msg->payload_,(vmp_size)code);
            return jrp::status_ok;
        }
    }
    catch(vmp::exception &x)
    {
    }
    msg->reset();
    return jrp::status_malformed_msg;
}

void jrp_write_str(vmp::Buf *buf,vmp::str jrp,vmp::Buf *payload)
{
    buf->reset();
    buf->write_size(jrp.size(),4);
    buf->write_str(jrp);
    if(payload != 0 && payload->size() != 0)
        buf->write_buf(payload);
    buf->index();
}
 
void jrp_write(vmp::Buf *buf,json::Json *json,vmp::Buf *payload)
{
    jrp_write_str(buf,json->json_str(),payload);
}

void abort(vmp::Buf *buf,vmp_int status,vmp::str msg)
{
    jrp::pointer_check((void *) buf,"abort","buf");
    json::Json json;
    json::JsonObj *root,obj;
    root=json.json_new();
    root->add_object_str("msgtype","abort",&obj);
    root->add_object_number("status",status,&obj);
    if(msg != "")
        root->add_object_str("msg",msg,&obj);
    else
        root->add_object_str("msg",jrp::msg_status(status).c_str(),&obj);
    buf->write_str(json.json_str());
    buf->index();
}

void request(vmp::Buf *buf,vmp_int rid,json::JsonObj *input,vmp::Buf *payload)
{
    jrp::pointer_check((void *) buf,"request","buf");
    jrp::pointer_check((void *) input,"request","input");
    json::Json json;
    json::JsonObj root,obj;
    json.root_cpy(&root);
    json::JData jdata;
    try
    {
        jdata.set(input);
    }
    catch(vmp::exception &x)
    {
        vmp::except_s("jrp::request() bad input json object");
    }
    vmp_size psize=0;
    if(payload != 0)
        psize=payload->size();
    root.add_object_str("msgtype","request",&obj);
    root.add_object_number("rid",rid,&obj);
    root.add_object_ex("input",input);
    root.add_object_number("payload",psize,&obj);
    jrp::jrp_write(buf,&json,payload);
}

void push(vmp::Buf *buf,vmp_int rid,json::JsonObj *outputs,vmp::Buf *payload)
{
    jrp::pointer_check((void *) buf,"response","buf");
    jrp::pointer_check((void *) outputs,"response","outputs");
    json::Json json;
    json::JsonObj root,obj,array;
    json.root_cpy(&root);
    json::JData jdata;
    vmp_size psize=0;
    if(payload != 0)
        psize=payload->size();
    root.add_object_str("msgtype","push",&obj);
    root.add_object_number("rid",rid,&obj);
    try
    {
        if(outputs->isarray())
        {
            vmp_size s=outputs->get_array_size();
            if(s == 0 ||(psize > 0 && s != 1))
                vmp::except_s("");
            for(vmp_index i=0;i<s;i++)
            {
                outputs->get_array_idx(i,&obj);
                jdata.set(&obj);
            }
            root.add_object_ex("outputs",outputs);
        }
        else
        {
            jdata.set(outputs);
            root.add_object_array("outputs",&array);
            array.push_array_obj_ex(outputs);
        }
    }
    catch(vmp::exception &x)
    {
        vmp::except_s("jrp::push() bad outputs json object");
    }
    root.add_object_number("payload",psize,&obj);
    jrp::jrp_write(buf,&json,payload);  
}

void response(vmp::Buf *buf,vmp_int rid,json::JsonObj *outputs,vmp::Buf *payload)
{
    jrp::pointer_check((void *) buf,"response","buf");
    jrp::pointer_check((void *) outputs,"response","outputs");
    json::Json json;
    json::JsonObj root,obj,array;
    json.root_cpy(&root);
    json::JData jdata;
    vmp_size psize=0;
    if(payload != 0)
        psize=payload->size();
    root.add_object_str("msgtype","response",&obj);
    root.add_object_number("rid",rid,&obj);
    try
    {
        if(outputs->isarray())
        {
            vmp_size s=outputs->get_array_size();
            if(s == 0 ||(psize > 0 && s != 1))
                vmp::except_s("");
            for(vmp_index i=0;i<s;i++)
            {
                outputs->get_array_idx(i,&obj);
                jdata.set(&obj);
            }
            root.add_object_ex("outputs",outputs);
        }
        else
        {
            jdata.set(outputs);
            root.add_object_array("outputs",&array);
            array.push_array_obj_ex(outputs);
        }
    }
    catch(vmp::exception &x)
    {
        vmp::except_s("jrp::response() bad outputs json object");
    }
    root.add_object_number("payload",psize,&obj);
    jrp::jrp_write(buf,&json,payload);
}

void close(vmp::Buf *buf,vmp_int rid,vmp_int status,vmp::str msg)
{
    jrp::pointer_check((void *) buf,"close","buf");
    json::Json json;
    json::JsonObj root,obj;
    json.root_cpy(&root);
    root.add_object_str("msgtype","close",&obj);
    root.add_object_number("rid",rid,&obj);
    root.add_object_number("status",status,&obj);
    if(msg != "")
        root.add_object_str("msg",msg,&obj);
    else
        root.add_object_str("msg",jrp::msg_status(status).c_str(),&obj);
    jrp::jrp_write(buf,&json);
}

void kill(vmp::Buf *buf,vmp_int rid)
{
    jrp::pointer_check((void *) buf,"close","buf");
    json::Json json;
    json::JsonObj root,obj;
    json.root_cpy(&root);
    root.add_object_str("msgtype","kill",&obj);
    root.add_object_number("rid",rid,&obj);
    jrp::jrp_write(buf,&json);
}

void broadcast(vmp::Buf *buf,json::JsonObj *input,vmp::Buf *payload)
{
    jrp::pointer_check((void *) buf,"broadcast","buf");
    jrp::pointer_check((void *) input,"broadcast","input");
    json::Json json;
    json::JsonObj root,obj;
    json.root_cpy(&root);
    json::JData jdata;
    try
    {
        jdata.set(input);
    }
    catch(vmp::exception &x)
    {
        vmp::except_s("jrp::broadcast() bad input json object");
    }
    vmp_size psize=0;
    if(payload != 0)
        psize=payload->size();
    root.add_object_str("msgtype","broadcast",&obj);
    root.add_object_ex("input",input);
    root.add_object_number("payload",psize,&obj);
    jrp::jrp_write(buf,&json,payload);
}

}}

