/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 30/11/2024
*/

#include "jrp.h"

namespace vampiria { namespace jrp {

void empty_jreqcb(jrp::JrpReq *jreq)
{
}

void empty_jreqreqcb(jrp::JrpReq *jreq,vmp::Buf *payload)
{
}

void empty_jreqerrcb(jrp::JrpReq *jreq,vmp_int errcode,vmp::str msg)
{
}

void empty_jreqdatacb(jrp::JrpReq *jreq,json::JsonObj *jdata,vmp::Buf *payload)
{
}

void empty_jbroadcastcb(event::Cell *cell,json::JsonObj *jdata,vmp::Buf *payload)
{
}

EventJrp::EventJrp():EventSslSub()
{
    recv_=jrp::jrp_recv_peer;
    close_=jrp::jrp_close_peer;
    evtsub_jrp_reset();
}

EventJrp::~EventJrp()
{
}

void EventJrp::evtsub_jrp_reset()
{
    common_=0;
    peersession_.reset();
    refrid_=0;
}

void EventJrp::evtsub_jrp_close(vmp_int code,vmp::str msg)
{
    event::Manager *manager=cell_->get_manager();
    if(code == jrp::status_ok)
        manager->cell_close(cell_,event::SUCCESS);
    else
    {
        if(msg == "")
            msg=jrp::msg_status(code);
        vmp::Buf buf; 
        jrp::abort(&buf,code,msg);
        manager->cell_close(cell_,event::ERROR,buf.read_str(buf.size()));
    }
}

vmp::pair<vmp::str,vmp::str> EventJrp::evtsub_session_client(vmp_uint permits)
{
    vmp::Buf buf;
    common_->session_.session(&buf,permits);
    return vmp::make_pair((vmp::str)jrp::protoname_,buf.read_str(buf.size()));
}

void EventJrp::evtsub_session_client_buf(vmp::pair<vmp::str,vmp::str> data,vmp::Buf *out)
{
    vmp::except_check_pointer((void *) out,"jrp::EventJrp::evtsub_session_client_buf(out=null)");
    jrp::jrp_write_str(out,data.second);
}
       
vmp::pair<vmp::str,vmp::str> EventJrp::evtsub_session_server(vmp_uint permits,vmp::str subprotocols,vmp::str session)
{
    if(subprotocols != jrp::protoname_)
        vmp::except_s("jrp::EventJrp::evtsub_session_server(subprocols='bad value')"); 
    if(session != "")
    {
        try
        {
            json::Json json;
            json.parse_from_str(session);
            peersession_.session_get(json.root());
        }
        catch(vmp::exception &x)
        {
            vmp::except_s("jrp::EventJrp::evtsub_session_server(session='bad value')");
        }
    }
    vmp::Buf buf;
    common_->session_.session(&buf,permits);
    return vmp::make_pair((vmp::str)jrp::protoname_,buf.read_str(buf.size()));
}

void EventJrp::evtsub_session_server_buf(vmp::pair<vmp::str,vmp::str> data,vmp::Buf *out)
{
    vmp::except_check_pointer((void *) out,"jrp::EventJrp::evtsub_session_client_buf(out=null)");
    jrp::jrp_write_str(out,data.second);
}
    
void EventJrp::evtsub_session_client_confirm(vmp::str subprotocols,vmp::str session)
{
    if(subprotocols != jrp::protoname_)
        vmp::except_s("jrp::EventJrp::evtsub_session_client_confirm(subprocols='bad value')"); 
    if(session != "")
    {
        try
        {
            json::Json json;
            json.parse_from_str(session);
            peersession_.session_get(json.root());
        }
        catch(vmp::exception &x)
        {
            vmp::except_s("jrp::EventJrp::evtsub_session_client_confirm(session='bad value')");
        }
    }
}

void EventJrp::evtsub_ssl_close_impl()
{
    vmp::vector<jrp::JrpReq *> sreqs=srvreq_.all_data(); 
    for(vmp_index i=0;i<sreqs.size();i++)
        sreqs[i]->recv_close(jrp::status_closeconnection);
    vmp::vector<jrp::JrpReq *> creqs=cltreq_.all_data();
    for(vmp_index i=0;i<creqs.size();i++)
        creqs[i]->recv_close(jrp::status_closeconnection);
}
        
void EventJrp::evtsub_ssl_free_impl()
{
    evtsub_jrp_reset();
}

JrpCommon::JrpCommon()
{
    maxclientreq_=100;
    maxssrvpeer_=5;
    killwait_=60;
    requestcb_=jrp::empty_jreqreqcb;
    requesterrcb_=jrp::empty_jreqerrcb;
    killcb_=jrp::empty_jreqcb;
    pushcb_=jrp::empty_jreqdatacb;
    responsecb_=jrp::empty_jreqdatacb;
    closecb_=jrp::empty_jreqcb;
    broadcastcb_=jrp::empty_jbroadcastcb;
}
        
JrpCommon::~JrpCommon()
{
    destroy();
}

void JrpCommon::destroy()
{
}

void JrpCommon::set_limits(vmp_size maxssrvpeer,vmp_size maxclientreq,vmp::time::Time killwait)
{
    if(maxssrvpeer > 0)
        maxssrvpeer_=maxssrvpeer;
    if(maxclientreq > 0)
        maxclientreq_=maxclientreq;
    if(killwait > 0)
        killwait_=killwait;
}

void JrpCommon::jrp_callback(jrp::JREQREQCB requestcb,jrp::JREQERRCB requesterrcb,jrp::JREQCB killcb,jrp::JREQDATACB pushcb,jrp::JREQDATACB responsecb,jrp::JREQCB closecb)
{
    if(requestcb != 0)
        requestcb_=requestcb;
    if(requesterrcb != 0)
        requesterrcb_=requesterrcb;
    if(killcb != 0)
        killcb_=killcb;
    if(pushcb != 0)
        pushcb_=pushcb;
    if(responsecb != 0)
        responsecb_=responsecb;
    if(closecb != 0)
        closecb_=closecb; 
}

void JrpCommon::jrp_broadcast_callback(jrp::JBROADCASTCB broadcastcb)
{
    if(broadcastcb != 0)
        broadcastcb_=broadcastcb;   
}

void JrpCommon::init_session(vmp::str nodetype)
{
    session_.nodetype_=nodetype;
}

void JrpCommon::add_reqdata(vmp::str input,vmp::vector<vmp::str> push,vmp::vector<vmp::str> response,vmp::str permits)
{
    session_.add_reqdata(input,push,response,permits);
}

void JrpCommon::add_broadcastdata(vmp::str input)
{
    session_.add_broadcastdata(input);
}

void JrpCommon::reset_session()
{
    session_.reset();
}

vmp::str JrpCommon::nodetype()
{
    return session_.nodetype();
}

jrp::EventJrp *JrpCommon::new_event(event::EVTCBBUF recv,event::EVTCB close)
{
    jrp::EventJrp *jevt=jevt_.get();
    jevt->common_=this;
    if(recv != 0)
        jevt->recv_=recv;
    if(close != 0)
        jevt->close_=close;
    return jevt;
}

void JrpCommon::free_event(jrp::EventJrp *jrp)
{
    jevt_.free(jrp);
}

jrp::JrpReq *JrpCommon::new_request(event::Cell *cell,json::JsonObj *jdata,vmp::Buf *payload)
{
    vmp::except_check_pointer((void *)cell,"json::jrp::JrpCommon::new_request(cell=null)");
    vmp::except_check_pointer((void *)jdata,"json::jrp::JrpCommon::new_request(jdata=null)");
    crypto::EventSsl *ssl=cell->event<crypto::EventSsl>();
    jrp::EventJrp *jrp=(jrp::EventJrp *) ssl->sub_;
    jrp::JrpReq *req;
    vmp_index count=0; 
    while(jrp->cltreq_.search(jrp->refrid_,&req))
    {
        count++;
        jrp->refrid_++;
        if(count == maxclientreq_)
            vmp::except_s("Overflow request");
        if(jrp->refrid_ == maxclientreq_)
            jrp->refrid_=0;
    }
    req=jreq_.get();
    try
    {
        req->setevent(this,cell,jrp->refrid_,jdata,"request");
        vmp::Buf buf;
        jrp::request(&buf,jrp->refrid_,jdata,payload);
        ssl->evt_ssl_send(&buf);
    }
    catch(vmp::exception &x)
    {
        jreq_.free(req);
        vmp::except("jrp::JrpCommon::new_request() %s",x.what());
    }
    req->alloc_internal();
    jrp->cltreq_.insert(jrp->refrid_,req);
    jrp->refrid_++;
    return req;
}

jrp::JrpReq *JrpCommon::recv_request(jrp::MsgParse *msg,event::Cell *cell)
{
    vmp::except_check_pointer((void *)msg,"jrp::JrpCommon::recv_request(msg=null)");
    vmp::except_check_pointer((void *)cell,"jrp::JrpCommon::new_request(cell=null)");
    jrp::JrpReq *req=jreq_.get();
    req->setevent(this,cell,msg->rid_,&msg->input_,"managed");
    req->alloc_internal();
    return req;
}

void JrpCommon::free_request(jrp::JrpReq *jreq)
{
    jreq_.free(jreq);
}

void JrpCommon::broadcast(event::Cell *cell,json::JsonObj *jdata,vmp::Buf *payload)
{
    vmp::except_check_pointer((void *)cell,"json::jrp::JrpCommon::new_request(cell=null)");
    vmp::except_check_pointer((void *)jdata,"json::jrp::JrpCommon::new_request(jdata=null)");
    crypto::EventSsl *ssl=cell->event<crypto::EventSsl>();
    try
    {
        vmp::Buf buf;
        jrp::broadcast(&buf,jdata,payload);
        ssl->evt_ssl_send(&buf);
    }
    catch(vmp::exception &x)
    {
        vmp::except("jrp::JrpCommon::new_broadcast() %s",x.what());
    }
}

vmp_size JrpCommon::maxclientreq()
{

    return maxclientreq_;
}

vmp_size JrpCommon::maxssrvpeer()
{
    return maxssrvpeer_;
}

vmp::time::Time JrpCommon::killwait()
{
    return killwait_;
}

vmp::str sessiontype(event::Cell *cell)
{
    vmp::except_check_pointer((void *)cell,"jrp::sessiontype(cell=null)");
    event::Manager *manager=cell->get_manager();
    manager->lock();
    jrp::EventJrp *jrp=(jrp::EventJrp *)cell->event<crypto::EventSsl>()->sub_;
    vmp::str ret=jrp->sessiontype_;
    manager->unlock();
    return ret;
}

vmp::str sessionid(event::Cell *cell)
{
    vmp::except_check_pointer((void *)cell,"jrp::sessionid(cell=null)");
    event::Manager *manager=cell->get_manager();
    manager->lock();
    jrp::EventJrp *jrp=(jrp::EventJrp *)cell->event<crypto::EventSsl>()->sub_;
    vmp::str ret=jrp->sessionid_;
    manager->unlock();
    return ret;
}

vmp_uint sessionpermits(event::Cell *cell)
{
    vmp::except_check_pointer((void *)cell,"jrp::sessionpermits(cell=null)");
    event::Manager *manager=cell->get_manager();
    manager->lock();
    jrp::EventJrp *jrp=(jrp::EventJrp *)cell->event<crypto::EventSsl>()->sub_;
    vmp_uint ret=jrp->sessionpermits_;
    manager->unlock();
    return ret;
}

}}

