/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 07/01/2025
 */

#include "event.h"

namespace vampiria { namespace event {

Cell::Cell()
{
    reset();
    id_=0;
}

Cell::~Cell()
{
}

void Cell::reset()
{
    state_=event::ACTIVE;
    timeout_=0.0;
    timewait_=0.0;
    nwait_=0;
    ret_=event::NOCLOSE;
    closelevel_="";
    errcode_=0;
    err_="";
    refalloc_=0;
    priority_=0;
    
    ui_=0;
    event_=0;
    manager_=0;
    priority_=0;
    read_=0;
    close_=0;
    
    writing_=false;
    evtype_="";
}

vmp_bool Cell::is_active()
{
    manager_->lock();
    vmp_bool ret=(state_ == event::ACTIVE);
    manager_->unlock();
    return ret;
}

vmp_bool Cell::is_closewait()
{
    manager_->lock();
    vmp_bool ret=(state_ == event::CLOSEWAIT);
    manager_->unlock();
    return ret;
}

vmp_bool Cell::is_close()
{
    manager_->lock();
    vmp_bool ret=(state_ == event::CLOSE);
    manager_->unlock();
    return ret;
}

vmp::str Cell::identity()
{
    manager_->lock();
    vmp::str ret=ui_->identity(this); 
    manager_->unlock();
    return ret;
}

vmp_index Cell::id()
{
    return id_;
}

void Cell::close(vmp::str closelevel)
{
    manager_->lock();
    manager_->cell_close_ok_spec(this,closelevel);
    manager_->unlock();
}

void Cell::close_wait(vmp_uint n,vmp::str closelevel)
{
    manager_->lock();
    manager_->cell_closewait(this,n,closelevel);
    manager_->unlock();
}

void Cell::close_err(vmp::str err)
{
    manager_->lock();
    manager_->cell_close(this,event::ERROR,err);
    manager_->unlock();
}

void Cell::close_err_spec(vmp::str errlevel,vmp_int errcode,vmp::str err)
{
    manager_->lock();
    manager_->cell_close_err_spec(this,errlevel,errcode,err);
    manager_->unlock();
}

void Cell::alloc()
{
    manager_->lock();
    manager_->cell_alloc(this);
    manager_->unlock();
}

void Cell::release()
{
    manager_->lock();
    manager_->cell_release(this);
    manager_->unlock();
}

void Cell::set_priority(vmp_uint priority)
{
   manager_->lock();
   manager_->cell_set_priority(this,priority);
   manager_->unlock();
}

vmp_uint Cell::get_priority()
{
   manager_->lock();
   vmp_uint ret=manager_->cell_get_priority(this);
   manager_->unlock();
   return ret;
}

event::CELLRET Cell::ret()
{
    manager_->lock();
    event::CELLRET ret=ret_;
    manager_->unlock();
    return ret;
}

vmp::str Cell::closelevel()
{
    manager_->lock();
    vmp::str ret=closelevel_;
    manager_->unlock();
    return ret;
}

vmp_int Cell::errcode()
{
    manager_->lock();
    vmp_int ret=errcode_;
    manager_->unlock();
    return ret;
}

vmp::str Cell::str_error()
{
    manager_->lock();
    vmp::str ret=err_;
    manager_->unlock();
    return ret;
}

void Cell::set_timewait(vmp::time::Time timewait)
{
    manager_->lock();
    manager_->cell_timewait(this,timewait);
    manager_->unlock();
}

vmp::time::Time Cell::get_timewait()
{
    manager_->lock();
    vmp::time::Time ret=manager_->cell_get_timewait(this);
    manager_->unlock();
    return ret;
     
}

vmp::time::Time Cell::get_timeout()
{
    manager_->lock();
    vmp::time::Time ret=manager_->cell_get_timeout(this);
    manager_->unlock();
    return ret;
}

event::Manager *Cell::get_manager()
{
    return manager_;
}

vmp_bool Cell::operator<(Cell &cell)
{
    if(this->priority_ < cell.priority_)
       return true;
    return false;
}

event::Cell *Cell::master()
{
    event::Cell *ret;
    manager_->lock();
    ret=manager_->cell_master(this);
    manager_->unlock();
    return ret;
}

vmp::str Cell::subid()
{
    vmp::str ret;
    manager_->lock();
    ret=manager_->cell_subid(this);
    manager_->unlock();
    return ret;
}

void Cell::addsub(vmp::str sid,event::Cell *sub)
{
    manager_->lock();
    manager_->cell_addsub(this,sid,sub);
    manager_->unlock();
}

event::Cell *Cell::searchsub(vmp::str sid)
{
    event::Cell *cell;
    manager_->lock();
    cell=manager_->cell_searchsub(this,sid);
    manager_->unlock();
    return cell;
}

void Cell::delsub(vmp::str sid)
{
    manager_->lock();
    manager_->cell_delsub(this,sid);
    manager_->unlock();
}

}}

