/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 07/01/2025
 */

#include "vmp.h"

#include "event/lib/lib.h"

#ifndef VAMPIRIA_EVENT_H

#define VAMPIRIA_EVENT_H 1

namespace vampiria { namespace event {

//!Base class for creating User Interface
class UI
{
    protected:
        friend event::Cell;
        event::Manager *manager_;/*!<Manager associated with the ui interface*/
    public:
        //! A Constructor
        /*!
            @param manager Manager to associate with the interface
        */
        UI(event::Manager *manager);
        
        //! A Destructor
        virtual ~UI();

        //!Returns manager associated with ui interface
        /*!
            @sa manager_
            @return ui Manager associated with the ui interface
        */
        event::Manager *manager();

        //!Used by some user interfaces to create child events (ex tcp server where the child is the connection)
        /*!
            @param cell Event Cell to generate new event
            @return Cell child(default 0)
        */
        virtual event::Event *child_event_new(event::Cell *cell);

        //!Cell identity
        /*!
            @param cell input cell
            @return A string that identifies the cell 
        */
        virtual vmp::str identity(event::Cell *cell);
        
        //!Called when an event is closed
        /*!
            @param cell input cell
        */
        virtual void close_event(event::Cell *cell)=0;
        
        //!Called when the memory of an event is freed
        /*!
            @param cell input cell
        */
        virtual void free_ref(event::Cell *cell)=0;
};

}}

#include "event/ui/breakloopui.h"

namespace vampiria { namespace event {

typedef enum cellstate {ACTIVE,CLOSEWAIT,CLOSE} CELLSTATE;
typedef enum cellret {NOCLOSE,SUCCESS,ERROR,TIMEOUT} CELLRET;

const vmp_uint MAX_CELL_PRIORITY=999;/*!<Max cell priority*/
const vmp_uint BREAKLOOP_PRIORITY=1000;/*!<Special event breakloop priority*/

//! Cell event reference.An event is associated with each cell
class Cell:public vmp::utils::Container<vmp::str>
{
    private:
       friend event::Manager;
       event::CELLSTATE state_;/*!< Cell state*/
       vmp_index id_;/*!<Cell identification*/
       
       vmp_index   nwait_;        /*!< Number of events executable in the state CLOSEWAIT*/
       vmp_index   refalloc_;     /*!<References Cell allocated*/
      
       event::UI *ui_;           /*!<User interface associated with event*/
       event::Event *event_;     /*!<Event handle*/
       event::Manager *manager_; /*!<Associated manager*/
       
       //! Reset cell
       void reset();
   public:
       //! A Constructor
       Cell();

       //! A Destructor
       ~Cell();

       vmp_uint priority_;/*!<Priority of the event execution*/
       vmp::time::Time timeout_;  /*!< Cell timeout*/
       vmp::time::Time timewait_; /*!< Maximum time of inactivity allowed to the Cell*/
       
       event::EVTCB read_;/*!<Read callback*/
       event::EVTCB close_;/*!<close callback*/
       
       event::CELLRET ret_;/*!<Exit code*/
       vmp::str closelevel_; /*!<specify the event that caused the closure*/
       vmp_int errcode_;   /*!<Used with close_err_spec() to specify the error code*/
       vmp::str err_;      /*!<Error info. With Exit code ERROR*/
       
       vmp::str evtype_;  /*!<Used in UI for managing multiple event types(default generic)*/
       vmp_bool writing_; /*!<True writing associated select fd,otherwise false(default false)*/
       
       //!Return true if the cell is active(monitor mode)
       /*!
           @return true if the cell is active,otherwise false
       */
       vmp_bool is_active();
       
       //!Return true if the cell status is closewait(monitor mode)
       /*!
           @return true if the cell status is closewait,otherwise false
       */
       vmp_bool is_closewait();
       
       //!Return true if the cell status is close(monitor mode)
       /*!
           @return true if the cell status is close,otherwise false
       */
       vmp_bool is_close();
       
       //!Return a string that identifies the event(monitor)
       /*!
           @return string identity
           @sa ui_
       */
       vmp::str identity();
       
       //!Returns the id associated with the cell
       /*!
           @return cell id
           @sa id_
       */
       vmp_index id();
       
       //!Close event with state SUCCESS(monitor)
       /*!
           @param closelevel close level value
           @ref ret_
       */
       void close(vmp::str closelevel="");

       //!Assigns a number of events to the event to be executed before closing
       /*!
            @param n number of events executed before closing
            @param closelevel close level value
       */
       void close_wait(vmp_uint n,vmp::str closelevel="");
       
       //!Close event with state ERROR(monitor)
       /*!
           @ref ret_
           @ref err_
           @param err error return string
       */
       void close_err(vmp::str err);
       
       //!Close event with state ERROR(monitor mode) and specifies the event that generated the error and its error code.
       /*!
           @ref ret_
           @ref errlevel_
           @ref errcode_
           @ref err_
           @param errlevel event level
           @param errcode error code
           @param err error return string
       */
       void close_err_spec(vmp::str closelevel,vmp_int errcode,vmp::str err);
       
       //!Alloc cell references(monitor mode)
       /*!
           @sa refalloc_
       */
       void alloc();

       //!Release cell.(monitor mode)
       /*!
           @sa refalloc_
       */      
       void release();
       
       //! Set the priority for the execution of events in a loop(0 Lowest priority default,max value event::MAX_CELL_PRIORITY)(monitor mode)
       /*!
           @param priority cell event priority
       */
       void set_priority(vmp_uint priority);
        
       //! Get the priority for the execution of events in a loop(0 Lowest priority default,max value event::MAX_CELL_PRIORITY)(monitor mode)
       /*!
           @return priority cell
       */
       vmp_uint get_priority();

       //!Return the closing value(monitor mode)
       /*!
           @return SUCCESS(Cell closed successfully),ERROR(Cell closed with error,call str_error for data err),TIMEOUT(Cell closed with timeout),NOCLOSE(cell not closed)
           @sa ret_
   
       */
       event::CELLRET ret();

       //!Return close level value(monitor mode)
       /*!
           @return close level value
           @sa closelevel_
       */
       vmp::str closelevel();
       
       //!Return error code value(monitor mode)
       /*!
           @return error code value
           @sa errcode_
       */
       vmp_int errcode();
       
       //!Returns error string(monitor mode)
       /*!
           @return error string
           @sa err_
       */
       vmp::str str_error();

       //! Set timewait cell,value 0.0 inactive timewait(monitor mode)
       /*!
           @param timewait input time wait
           @sa timewait_
       */
       void set_timewait(vmp::time::Time timewait=0.0);

       //! Get timewait cell(monitor mode)
       /*!
           @return timewait event or 0.0 if the timewait is not set
       */
       vmp::time::Time get_timewait();
       
       //! Get the time when the timeout is triggered(monitor mode)
       /*!
           @return timeout or 0.0 if the timeout is not set
       */
       vmp::time::Time get_timeout();

       //!Return manager associated with cell
       /*!
           @return manager
           @sa manager_
       */
       event::Manager *get_manager();

       //!Get user interface associated with cell
       /*!
           @return ui
           @sa ui_
       */
       template<class UI>
       UI *ui()
       {
           return (UI *) ui_;
       }

       //!Get Event associated with cell
       /*!
           @return event
           @sa event_
       */
       template<class EVENT>
       EVENT *event()
       {
           return (EVENT *) event_;
       }

       //! Operator to compare tag which has more or less of the other priorities                
       /*!
            @param cell cell compare
            @return true if cell has higher priority than this otherwise false(hight numbers have higher priority)                
       */                
       vmp_bool operator<(Cell &cell);
       
       //! Returns the master cell if the event is a subevent of another event(monitor mode)
       /*!
           @return the master cell or 0 in case of failure
       */
       event::Cell *master();
       
       //! Returns the id associated with the cell in the master if it is a sub-event of the master(monitor mode)
       /*!
           @return the subevent cell id
       */
       vmp::str subid();
       
       //! Adds a sub-event to the event if sub != 0(monitor mode)
       /*!
           @param sid subevent event id
           @param sub subevent cell
       */
       void addsub(vmp::str sid,event::Cell *sub);
       
       //! Search a subevent from event(monitor mode)
       /*!
           @param sid subevent event id
           @return event cell subevent or 0 in case of failure
       */
       event::Cell *searchsub(vmp::str sid);
        
       //! Delete a subevent from event(monitor mode)
       /*!
           @param sid subevent event id
       */
       void delsub(vmp::str sid);
};

//! Event manager 
class Manager
{
    private:
        friend event::Cell;
        vmp_uint cellids_;/*!<For assigning cell IDs*/
        vmp::time::Timer timer_;/*!<Used to calculate the time elapsed since the manager was created*/
        vmp::time::Time timeout_;/*!<Timeout setting*/
        vmp::utils::Storage<Cell> cellref_;/*!<Storage utility for cell create*/
        vmp_bool loop_;/*!<variable used internally to verify that the loop is active*/
        vmp_bool init_;/*!<Manager is init?*/
        vmp::priority_queue<event::Cell *,vmp::vector<event::Cell *>,vmp::less<vmp::vector<Cell*>::value_type> > pcell_;/*!<Priority cell build queue*/ 
        event::BreakloopUI *breakui_;/*!<Breakloop user interface*/
        vmp::thread::Mutex mutex_;/*!<variable mutual exclusion manager*/
        vmp::thread::Pthread self_;/*!<Thread id self*/
        
        //! Internal usage
        void cell_freedata(event::Cell *cell);
    public:
        //! A constructor
        Manager();

        //! A destructor
        ~Manager();

        //! Initializes a new session for the manager, must be called before the creation of events.(Automatic call in the constructor)
        void init();
       
        //!Performs a break loop event to be able to reinitialize the event environment
        void breakloop();

        //!Lock manager
        void lock();

        //!unlock manager
        void unlock();

        //! If the mutex is locked force the unlock mutual exclusion
        /*!
             @return 1 if unlocked otherwise 0(Used as input in recovery_lock() for recovery)
        */
        vmp_bool forced_unlock();

        //! Recovery lock after a forced_unlock() call
        /*!
            @param status value returned from forced_unlock() function
        */
        void recovery_lock(vmp_bool status);        

        //!Create new Cell
        /*!
            @param ui user interface associated
            @param event event associated
            @param priority execution priority of the associated event
            @param read read event callback
            @param close close event callback
            @return new cell
        */
        event::Cell *cell_new(event::UI *ui,event::Event *event,vmp_int priority,event::EVTCB read,event::EVTCB close);
       
        //!Update cell status.(call in read and write function)
        /*!
             @param cell cell update
             @return true if cell is active,otherwise false
        */
        vmp_bool cell_update(event::Cell *cell);

        //! Set timewait cell,value 0.0 inactive timewait
        /*!
            @param cell eventcell to set timewait
            @param timewait input time wait
            
        */
        void cell_timewait(event::Cell *cell,vmp::time::Time timewait);
        
        //! Get timewait cell
        /*!
           @param cell event cell to get timewait
           @return @return timewait event or 0.0 if the timewait is not set
        */
        vmp::time::Time cell_get_timewait(event::Cell *cell);
       
        //! Get the time when the timeout is triggered
        /*!
            @param cell event cell to get timewait
            @return timeout or 0.0 if the timeout is not set
        */
        vmp::time::Time cell_get_timeout(event::Cell *cell);
        
        //!Internal usage. Used for close event(no monitor)
        /*!
            @param cell cell to close
            @param retcode (see CELLRET)
            @param errinfo error info
        */
        void cell_close(event::Cell *cell,event::CELLRET retcode,vmp::str errinfo="");
       
        //!Used for close event with specific close level(no monitor)
        /*!
            @param cell cell to close
            @param closelevel event event that generates the error
            @param closelevel close level value
            @param errinfo error info
        */
        void cell_close_ok_spec(event::Cell *cell,vmp::str closelevel);
       
        //!Used for close event with specific close level and error value(no monitor)
        /*!
            @param cell cell to close
            @param closelevel close level value
            @param errcode error code
            @param errinfo error info
        */
        void cell_close_err_spec(event::Cell *cell,vmp::str closelevel,vmp_int errcode,vmp::str errinfo);
       
        //!Assigns a number of events to the event to be executed before closing
        /*!
            @param cell cell to close wait
            @param n number of events executed before closing
        */
        void cell_closewait(event::Cell *cell,vmp_uint n,vmp::str closelevel="");
       
        //!Returns the current time (since the creation of the manager)
        /*!
            @return the current time
            @sa timer_
        */
        vmp::time::Time time_now();

        //!Set timeout variable(Used in manager->loop_ and event timer)
        /*!
            @param timeout if timeout less than or equal to zero does nothing.if timeout_ is different from zero and timeout is less than timeout_ set the new timeout
        */
        void set_timeout(vmp::time::Time timeout);
        
        //!Alloc cell references
        /*!
            @param cell cell alloc     
        */
        void cell_alloc(event::Cell *cell);
       
        //!Release cell
        /*!
            @param cell to release
        */
        void cell_release(event::Cell *cell);
        
        //! Set the priority for the execution of events in a loop(0 Lowest priority default,max value event::MAX_CELL_PRIORITY)
        /*!
            @param cell event cell
            @param priority cell event priority
        */
        void cell_set_priority(event::Cell *cell,vmp_uint priority);
        
        //! Get the priority for the execution of events in a loop(0 Lowest priority default,max value event::MAX_CELL_PRIORITY)
        /*!
            @param cell event cell
            @return priority cell
        */
        vmp_uint cell_get_priority(event::Cell *cell);
        
        //! Returns the master cell if the event is a subevent of another event
        /*!
            @param event cell
            @return the master cell or 0 in case of failure
        */
        event::Cell *cell_master(event::Cell *cell);
       
        //! Returns the id associated with the cell in the master if it is a sub-event of the master
        /*!
            @param event cell
            @return the subevent cell id
        */
        vmp::str cell_subid(event::Cell *cell);
        
        //! Adds a sub-event to the event if sub != 0
        /*!
            @param cell event cell
            @param sid subevent event id
            @param sub subevent
        */
        void cell_addsub(event::Cell *cell,vmp::str sid,event::Cell *sub);
        
        //! Search a subevent from event
        /*!
            @param cell event cell
            @param sid subevent event id
            @return event cell subevent or 0 in case of failure
        */
        event::Cell *cell_searchsub(event::Cell *cell,vmp::str sid);
        
        //! Delete a subevent from event
        /*!
            @param sid subevent event id
        */
        void cell_delsub(event::Cell *cell,vmp::str sid);
        
        //!Stop loop event manager(monitor)
        void stop();

        //!Loops for the execution of events(monitor used in loop)
        void loop();
};

}}

#endif

#include "event/lib/eventsimplex.h"
#include "event/lib/eventinput.h"
#include "event/lib/eventtimer.h"
#include "event/lib/eventthread.h"

#include "event/ui/simplexui.h"
#include "event/ui/inputui.h"
#include "event/ui/timerui.h"
#include "event/ui/threadui.h"



