/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 19/03/2025
 */

#ifndef VAMPIRIA_CRYPTO_SSLBIO_H

#define VAMPIRIA_CRYPTO_SSLBIO_H 1

namespace vampiria { namespace crypto {

//!Class used for the connections ssl
class SslBio
{
    private:
        typedef enum biostatus{INIT,CONNECT,ACCEPT,ESTABLISHED,SHUTDOWN,CLOSE} BIOSTATUS;/*!<Internal usage*/
        crypto::Ctx ctx_;/*!<Connection context*/
        BIOSTATUS status_;/*!<Internal usage*/
        vmp_int errno_;/*!<Error value set by the last operation performed*/
        vmp::str error_;/*!<String Mode Error*/
        SSL *ssl_;/*!<Ssl handler*/
        BIO *rbio_;/*!<SSL reads from, we write to.*/
        BIO *wbio_;/*!<SSL writes to, we read from.*/
        vmp_int ctype_;/*!<Internal usage,0 no connection,1 client,2 server*/
        
        //!Internal usage
        void except_ctx(vmp::str func);
        
        //!Internal lusage
        void set_errno(vmp_int ret);
        
        //!Internal usage
        vmp_int bio_read(vmp::Buf *out);
        
    public:
        //! A Constructor
        SslBio();
        
        //! A Constructor
        /*
            @param ctx context associated with ssl connection
        */
        SslBio(crypto::Ctx *ctx);
        
        //!A Destructor
        ~SslBio();

        //! Reset ssl connection
        void reset();
        
        //! Create an ssl structure associated with the context
        /*!
            @param ctx context associated with ssl connection
            @return void or except in case of failure
        */
        void set(crypto::Ctx *ctx);
        
        //! Returns the ssl error associated with the ret value
        /*!
            @sa errno_
            @return errno value
        */
        vmp_int get_errno();
        
        //! Returns string mode error
        /*!
            @sa error_
            @return string mode error
        */
        vmp::str get_error();
        
        //! Returns the context associated with the connection
        /*!
            @sa ctx_
            @return the context or except in case of failure
        */
        crypto::Ctx *ctx();
        
        //!Gets the local's certificate
        /*!
            @param cout certificate output
            @return void or except in case of failure
        */
        void get_local_x509(crypto::X509_Wrap *cout);
        
        //!Gets the peer's certificate
        /*!
            @param cout certificate output
            @return void or except in case of failure
        */
        void get_peer_x509(crypto::X509_Wrap *cout);
        
        //!Initiates a client connection and handles the handshake
        /*!
            @param in remote received message buffer
            @param out output buffer
            @return 1 If the connection is stabilized,0 if the handshake is not finished,-1 ssl error or except in case of failure
        */
        vmp_int connect(vmp::Buf *in,vmp::Buf *out);
        
        //!Initiates a client connection and handles the handshake
        /*!
            @param in remote received message buffer
            @param out output buffer
            @return 1 If the connection is stabilized,0 if the handshake is not finished,-1 ssl error or except in case of failure
        */
        vmp_int accept(vmp::Buf *in,vmp::Buf *out);
        
        //! Returns true if the connection is of the client type
        vmp_bool is_client();
        
        //! Returns true if the connection is of the server type
        vmp_bool is_server();
        
        //!Receives a message remotely
        /*!
            @param in remote received message buffer
            @param out output buffer
            @return The number of bytes read,-2 the connection is close,-1 ssl error or except in case of failure
        */
        vmp_int recv(vmp::Buf *in,vmp::Buf *out);
        
        //!Prepare a message to be sent remotely
        /*!
            @param in plaintext
            @param out encrypted message
            @return the number of bytes written to the output,-1 ssl error or except in case of failure
        */
        vmp_int send(vmp::Buf *in,vmp::Buf *out);
        
        //!Requires an update of the session key
        /*!
            @return true if the operation was carried out successfully otherwise false or except in case of failure
        */
        vmp_bool key_update();
        
        //!Requires a shutdown operation
        /*!
            @param out output buffer
            @return the number of bytes written to the output,-1 ssl error or except in case of failure 
        */
        vmp_int shutdown(vmp::Buf *out);
        
        //!Closes a connection
        void close();
};

}}

#endif

