/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 02/01/2025
 */

#include "crypto.h"

namespace vampiria { namespace crypto {

WssCommon::WssCommon(event::Manager *manager):crypto::SslCommon(manager)
{
    mimes_.insert("aac","audio/aac");
    mimes_.insert("abw","application/x-abiword");
    mimes_.insert("apng","image/apng");
    mimes_.insert("arc","application/x-freearc");
    mimes_.insert("avif","image/avif");
    mimes_.insert("avi","video/x-msvideo");
    mimes_.insert("azw","application/vnd.amazon.ebook");
    mimes_.insert("bin","application/octet-stream");
    mimes_.insert("bmp","image/bmp");
    mimes_.insert("bz","application/x-bzip");
    mimes_.insert("bz2","application/x-bzip2");
    mimes_.insert("cda","application/x-cdf");
    mimes_.insert("csh","application/x-csh");
    mimes_.insert("css","text/css");
    mimes_.insert("csv","text/csv");
    mimes_.insert("doc","application/msword");
    mimes_.insert("docx ","application/vnd.openxmlformats-officedocument.wordprocessingml.document");
    mimes_.insert("eot","application/vnd.ms-fontobject");
    mimes_.insert("epub","application/epub+zip");
    mimes_.insert("gz","application/gzip");
    mimes_.insert("gif","image/gif");
    mimes_.insert("htm","text/html");
    mimes_.insert("html","text/html");
    mimes_.insert("ico","image/vnd.microsoft.icon");
    mimes_.insert("ics","text/calendar");
    mimes_.insert("jar","application/java-archive");
    mimes_.insert("jpg","image/jpeg");
    mimes_.insert("jpeg","image/jpeg");
    mimes_.insert("js","text/javascript");
    mimes_.insert("json ","application/json");
    mimes_.insert("jsonld","application/ld+json");
    mimes_.insert("mid","audio/midi");
    mimes_.insert("midi","audio/midi");
    mimes_.insert("mjs","text/javascript");
    mimes_.insert("mp3","audio/mpeg");
    mimes_.insert("mp4 ","video/mp4");
    mimes_.insert("mpeg","video/mpeg");
    mimes_.insert("mpkg","application/vnd.apple.installer+xml");
    mimes_.insert("odp","application/vnd.oasis.opendocument.presentation");
    mimes_.insert("ods","application/vnd.oasis.opendocument.spreadsheet");
    mimes_.insert("odt ","application/vnd.oasis.opendocument.text");
    mimes_.insert("oga","audio/ogg");
    mimes_.insert("ogv","video/ogg");
    mimes_.insert("ogx","application/ogg");
    mimes_.insert("opus ","audio/ogg");
    mimes_.insert("otf","font/otf");
    mimes_.insert("png","image/png");
    mimes_.insert("pdf","application/pdf");
    mimes_.insert("php","application/x-httpd-php");
    mimes_.insert("ppt","application/vnd.ms-powerpoint");
    mimes_.insert("pptx","application/vnd.openxmlformats-officedocument.presentationml.presentation");
    mimes_.insert("rar ","application/vnd.rar");
    mimes_.insert("rtf","application/rtf");
    mimes_.insert("sh","application/x-sh");
    mimes_.insert("svg","image/svg+xml");
    mimes_.insert("tar","application/x-tar");
    mimes_.insert("tif","image/tiff");
    mimes_.insert("tiff","image/tiff");
    mimes_.insert("ts","video/mp2t");
    mimes_.insert("ttf","font/ttf");
    mimes_.insert("txt ","text/plain");
    mimes_.insert("vsd","application/vnd.visio");
    mimes_.insert("wav","audio/wav");
    mimes_.insert("weba","audio/webm");
    mimes_.insert("webm ","video/webm");
    mimes_.insert("webp ","image/webp");
    mimes_.insert("woff","font/woff");
    mimes_.insert("woff2","font/woff2");
    mimes_.insert("xhtml","application/xhtml+xml");
    mimes_.insert("xls","application/vnd.ms-excel");
    mimes_.insert("xlsx ","application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
    mimes_.insert("xml","application/xml");
    mimes_.insert("xul","application/vnd.mozilla.xul+xml");
    mimes_.insert("zip","application/zip");
    mimes_.insert("3gp","video/3gpp");
    mimes_.insert("3g2 ","video/3gpp2");
    mimes_.insert("7z","application/x-7z-compressed");
    
    http_versions={"HTTP/1.1"};
    wss_versions={13};
    
    servername_="unknown";
    maxsizepayload_=16777215;
    maxsizeuri_=8192;
    maxsizeheader_=8192;
    
    ctx_=0;
    webroot_="";
    
    wss_tcpaccept_=net::empty_accept_ev;
    wss_auth_=crypto::wss_auth_default;
    wss_session_=event::empty_ev;

    set_ssl_server_event(crypto::wss_tcpaccept,0,crypto::wssrecv_http,0,0);
    set_ssl_framing_cb(0,crypto::ssl_subprotocols_recv);
}

WssCommon::~WssCommon()
{
    http_versions.clear();
    wss_versions.clear();
    servername_="";
    webroot_="";
    mimes_.clear();
    
    vmp::vector<event::Cell *> data=usertable_.all_data();
    for(vmp_index i=0;i<data.size();i++)
        data[i]->release();
    usertable_.clear();
}

void WssCommon::set_wss_init(crypto::Ctx_Peer_Web *ctx,vmp::str webroot,vmp::time::Time ctimeout)
{
    vmp::except_check_pointer((void *)ctx,"crypto::WssCommon::set_wss_init(ctx=Null)");
    if(!vmp::fs::isrdir(webroot))
        vmp::except("crypto::WssCommon::set_wss_init(webroot='%s') directory not found",webroot.c_str());
    ctx_=ctx;
    webroot_=webroot;
    set_ssl_init(ctx->get(),ctimeout);  
}

void WssCommon::set_wss_http(vmp::str servername,vmp_size maxsizeuri,vmp_size maxsizeheader,vmp_size maxsizepayload)
{
    if(servername != "")
        servername_=servername;
    if(maxsizeuri != 0)
        maxsizeuri_=maxsizeuri;
    if(maxsizeheader != 0)
        maxsizeheader_=maxsizeheader;
    if(maxsizepayload != 0)
        maxsizepayload_=maxsizepayload;
}

void WssCommon::set_wss_event(net::EVTCBACCEPT tcpaccept,event::EVTCB session,event::EVTCB lclose,event::EVTCB sclose)
{
    set_ssl_server_event(0,0,0,lclose,sclose);
    if(tcpaccept != 0)
        wss_tcpaccept_=tcpaccept;
    if(session != 0)
        wss_session_=session;    
}

EventWss::EventWss():crypto::EventSsl()
{
    evt_wss_reset();
}

EventWss::~EventWss()
{
}

void EventWss::evt_wss_reset()
{
    isopensession_=false;
    hosts_.clear();
    hostsp_="";
    localhost_=true;
    origins_.clear();
    
    httprequest_.reset();
    httpresponse_.reset();
    httpcode_=0;
    httpversion_="";
    httpmethod_="";
    path_="";
    httpbody_="";
    host_="";
    hostp_="";
    subprotocols_="";
    wsorigin_="";
    wskey_="";
    wsupgrade_="";
    wsconnection_="";
    wsversion_=0;
    cookies_="";
}

vmp::str EventWss::wss_utils_body(vmp::str body)
{
    vmp::str ret;
    crypto::WssCommon *common=(crypto::WssCommon *)common_;
    vmp::unicode::str_write(&ret,"<!doctype html><html lang=\"en\"><head><meta charset=\"utf-8\"><meta name=\"viewport\" content=\"width=device-width, initial-scale=1, shrink-to-fit=no\"><title>%s</title></head>""<body>%s</body></html>",common->servername_.c_str(),body.c_str());
    return ret;
}

void EventWss::wss_utils_web(packet::Packet **ret)
{
    crypto::WssCommon *common=(crypto::WssCommon *)common_;
    vmp::str pathfile,sha1,tmp;
    vmp::fs::FileStat stat;
    vmp::Buf body;
    if(path_ == "/")
        pathfile=vmp::fs::union_path(common->webroot_,"index.html");
    else
        pathfile=vmp::fs::union_path(common->webroot_,path_);
    vmp::str mime;
    packet::http::HeadersHelper headers;
    vmp::PairList<vmp::str,vmp::str> *hdr=headers.headers();
    try
    {
        stat.set(pathfile);
        body.write_file(pathfile);
        body.index();
        mime="application/octet-stream";
        common->mimes_.search(vmp::fs::file_ext(pathfile),&mime);
    }
    catch(vmp::exception &x)
    {
        wss_utils_response_except(ret,packet::http::HttpCode_NotFound);
    }
    try
    {
        sha1=crypto::sha1_bin(&body,"");
        hdr->push("Accept-Ranges","bytes");
        hdr->push("Connection","Upgrade, Keep-Alive");
        vmp::unicode::str_write(&tmp,"%u",body.size());
        hdr->push("Content-Length",tmp);
        hdr->push("Content-Type",mime);
        hdr->push("Connection","Closed");
        hdr->push("Keep-Alive","Timeout=5,max=100");
        hdr->push("Date",vmp::time::gmtime_wrap());
        hdr->push("Server",common->servername_);
        hdr->push("Etag",sha1);
        hdr->push("Upgrade","h2,h2c");
        hdr->push("X-Content-Type-Options","nosniff");
        hdr->push("Last-Modified",vmp::time::gmtime_wrap(stat.mtime()));
        httpcode_=packet::http::HttpCode_OK; (*ret)=packet::http::Http_Response(httpversion_,httpcode_,packet::http::HttpCode_reason(httpcode_),&headers,body.read_xstr(body.size()));
    }
    catch(vmp::exception &x)
    {
        wss_utils_response_except(ret,packet::http::HttpCode_InternalServerError);
    }
}

packet::Packet *EventWss::wss_utils_response_text(vmp_uint code,vmp::str body)
{
    crypto::WssCommon *common=(crypto::WssCommon *)common_;
    vmp::str reason=packet::http::HttpCode_reason(code);
    httpcode_=code;
    vmp::str content,tmp;
    if(body == "")
        vmp::unicode::str_write(&body,"<center><h1>%d %s</h1></center>",httpcode_,reason.c_str());   
    content=wss_utils_body(body);
    packet::http::HeadersHelper headers;
    vmp::PairList<vmp::str,vmp::str> *hdr=headers.headers();
    hdr->push("Content-Type","text/html; charset=utf-8");
    vmp::unicode::str_write(&tmp,"%u",content.size());
    hdr->push("Content-Length",tmp);
    hdr->push("Connection","Closed");
    hdr->push("Date",vmp::time::gmtime_wrap());
    hdr->push("Server",common->servername_);
    return packet::http::Http_Response(httpversion_,code,reason,&headers,vmp::unicode::str_toxstr(content,""));
}

void EventWss::wss_utils_response_except(packet::Packet **ret,vmp_uint code,vmp::str body)
{
    (*ret)=wss_utils_response_text(code,body);
    vmp::except_s("");
}

void EventWss::wss_utils_badversion(packet::Packet **ret)
{
    crypto::WssCommon *common=(crypto::WssCommon *)common_;
    httpcode_=1023;
    vmp::str reason="WebSocket Unsupported versions",content,tmp;
    vmp::unicode::str_write(&content,"%u %s",httpcode_,reason.c_str());
    packet::http::HeadersHelper headers;
    vmp::PairList<vmp::str,vmp::str> *hdr=headers.headers();
    vmp::unicode::str_write(&tmp,"%u",common->wss_versions[0]);
    for(vmp_index i=1;i<common->wss_versions.size();i++)
        vmp::unicode::str_cwrite(&tmp,", %u",common->wss_versions[i]);
    hdr->push("Sec-WebSocket-Version",tmp);
    hdr->push("Content-Type","text/html; charset=utf-8");
    vmp::unicode::str_write(&tmp,"%u",content.size());
    hdr->push("Content-Length",tmp);
    hdr->push("Connection","Closed");
    hdr->push("Date",vmp::time::gmtime_wrap());
    hdr->push("Server",common->servername_);
    content=wss_utils_body(content);
    (*ret)=packet::http::Http_Response(httpversion_,httpcode_,reason,&headers,vmp::unicode::str_toxstr(content,""));
    vmp::except_s("");  
}

void EventWss::wss_utils_upgrade_except(packet::Packet **ret)
{
    crypto::WssCommon *common=(crypto::WssCommon *)common_;
    httpcode_=packet::http::HttpCode_UpgradeRequired;
    vmp::str reason=packet::http::HttpCode_reason(httpcode_),content,tmp;
    vmp::unicode::str_write(&content,"<center><h1>This service requires use of the Websocket protocol</h1></center>");   
    content=wss_utils_body(content);
    packet::http::HeadersHelper headers;
    vmp::PairList<vmp::str,vmp::str> *hdr=headers.headers();
    hdr->push("Content-Type","text/plain");
    vmp::unicode::str_write(&tmp,"%u",content.size());
    hdr->push("Content-Length",tmp);
    hdr->push("Connection","Upgrade");
    hdr->push("Upgrade","websocket");
    vmp::unicode::str_write(&tmp,"%u",common->wss_versions[0]);
    hdr->push("Sec-WebSocket-Version",tmp);
    hdr->push("Server",common->servername_);
    (*ret)=packet::http::Http_Response(httpversion_,httpcode_,reason,&headers,vmp::unicode::str_toxstr(content,""));
    vmp::except_s("");
}
void EventWss::wss_utils_continue(vmp::str user,vmp_uint permits)
{
    crypto::WssCommon *common=(crypto::WssCommon *)common_;
    vmp::pair<vmp::str,vmp::str> sdata;
    vmp_bool exec=true;
    event::Manager *manager=cell_->get_manager();
    packet::Packet *response;
    vmp_uint code=packet::http::HttpCode_InternalServerError;
    manager->lock();
    try
    {
        if(permits == 0)
        {
            code=packet::http::HttpCode_Unauthorized;
            vmp::except_s("");
        }
        try
        {
            sdata=sub_->evtsub_session_server(permits,subprotocols_,"");
        }
        catch(vmp::exception &x)
        {
            code=packet::http::HttpCode_BadRequest;
            vmp::except_s("");
        }
        try
        {
            common->usertable_.insert(user,cell_);
            manager->cell_alloc(cell_);
            sub_->sessionid_=user;
            sub_->sessionpermits_=permits;
            isopensession_=true;
        }
        catch(vmp::exception &x)
        {
            code=packet::http::HttpCode_Forbidden;
            vmp::except_s("");
        }
        httprequest_.reset();
        vmp::str key,ext=framing_.exts_info();
        packet::http::HeadersHelper headers;
        vmp::PairList<vmp::str,vmp::str> *hdr=headers.headers();
        if(ext != "")
            hdr->push("Sec-Websocket-Extensions",ext);
        hdr->push("Sec-Websocket-Protocol",sdata.first);
        vmp::unicode::str_write(&key,"%s258EAFA5-E914-47DA-95CA-C5AB0DC85B11",wskey_.c_str());
        key=crypto::sha1(key,"");
        key=vmp::unicode::b64_encode(vmp::unicode::xstr_tostr(key,""));
        hdr->push("Sec-Websocket-Accept",key);
        hdr->push("Upgrade","websocket");
        hdr->push("Connection","Upgrade");
        hdr->push("Server",common->servername_);
        httpcode_=packet::http::HttpCode_SwitchingProtocols;
        vmp::str reason=packet::http::HttpCode_reason(httpcode_);
        response=packet::http::Http_Response(httpversion_,httpcode_,reason,&headers,"");
        common->helper_.write(response,&httpresponse_);
        evt_connection_send(&httpresponse_);
        exec=true;
    }
    catch(vmp::exception &x)
    {
        response=wss_utils_response_text(code);
        common->helper_.write(response,&httpresponse_);
        evt_connection_send(&httpresponse_);
        vmp::str err;
        vmp::unicode::str_write(&err,"'%s' %s",user.c_str(),response->get("reason").c_str());
        manager->cell_close_err_spec(cell_,"crypto::EventWss",httpcode_,err);
    }
    packet::packet_free(response);
    if(exec)
    {    
        evt_ssl_framing_active();
        vmp::Buf send;
        sub_->evtsub_session_server_buf(sdata,&send);
        evt_ssl_send(&send);
        manager->unlock();
        common->wss_session_(cell_);
    }
    else
        manager->unlock();
}

event::Cell *EventWss::evt_wss_listen(event::UI *ui,net::Address *local,crypto::WssCommon *common,vmp_uint backlog,vmp::vector<vmp::str> hosts,vmp::vector<vmp::str> origins,vmp_bool localhost)
{
    vmp::str tmp;
    vmp::unicode::str_write(&tmp,"https://%s:%s",local->ip().c_str(),local->host(),local->service());
    origins_.push_back(tmp);
    for(vmp_index i=0;i<origins.size();i++)
    {
       vmp::unicode::str_write(&tmp,"https://%s:%s",origins[i].c_str(),local->host(),local->service());
       if(!vmp::invector<vmp::str>(tmp,origins_))
            origins_.push_back(tmp);
    }
    hosts_=hosts;
    localhost_=localhost;
    return evt_ssl_listen(ui,local,(crypto::SslCommon *)common,backlog);
}

void EventWss::evt_wss_close()
{
    if(isopensession_)
    {
        try
        {
            crypto::WssCommon *common=(crypto::WssCommon *)common_;
            event::Cell *tmp;
            common->usertable_.cancel(sub_->sessionid_,&tmp);
            event::Manager *manager=tmp->get_manager();
            manager->cell_release(tmp);
        }
        catch(vmp::exception &x)
        {
        }
        isopensession_=false;
    }
    evt_ssl_close();
}
       
void EventWss::evt_wss_free()
{
    evt_wss_reset();
    evt_ssl_free();
}

}}

